// Copyright (c) 2025, Digital Asset (Switzerland) GmbH and/or its affiliates. All rights reserved.

package com.digitalasset.transcode.codegen

import scala.quoted.*

object TypeUtils:

  inline def shortSymbolName[T <: AnyKind]: String =
    symbolName[T].split('.').last

  inline def symbolName[T <: AnyKind]: String =
    def go(tpe: TypeInfo): String = tpe match
      case TypeInfo.Application(tpe, args) => go(tpe)
      case TypeInfo.FQN(name)              => name
      case TypeInfo.Wild                   => "?"
    go(TypeUtils.typeInfo[T])

  inline def typeInfo[T <: AnyKind]: TypeInfo = ${ TypeUtils.typeInfoMacro[T] }
  private def typeInfoMacro[T <: AnyKind: Type](using Quotes) =
    import quotes.reflect.*
    def go(t: TypeRepr): TypeInfo = t match
      case TypeLambda(x, y, app) => go(app)
      case AppliedType(a, b)     => TypeInfo.Application(go(a), b.map(go))
      case ParamRef(_, _)        => TypeInfo.Wild
      case TypeBounds(_, _)      => TypeInfo.Wild
      case TypeRef(t, name)      => TypeInfo.FQN(s"${t.show}.$name")
      case TermRef(t, name)      => TypeInfo.FQN(s"${t.show}.$name")
    Expr(go(TypeRepr.of[T].simplified))

  private implicit val typeInfoConv: ToExpr[TypeInfo] = new ToExpr[TypeInfo] {
    def apply(x: TypeInfo)(using Quotes): Expr[TypeInfo] = x match
      case TypeInfo.FQN(name) => '{ TypeInfo.FQN(${ Expr(name) }) }
      case TypeInfo.Wild      => '{ TypeInfo.Wild }
      case TypeInfo.Application(tpe, args) =>
        val tpeExpr  = apply(tpe)
        val argsExpr = Expr.ofList(args.map(apply))
        '{ TypeInfo.Application($tpeExpr, $argsExpr) }
  }
