package com.descope.sdk.auth;

import com.descope.exception.DescopeException;
import com.descope.model.auth.AuthenticationInfo;
import com.descope.model.enchantedlink.EnchantedLinkResponse;
import com.descope.model.magiclink.LoginOptions;
import com.descope.model.user.User;

public interface EnchantedLinkService {
  /**
   * Use to login a user based on an enchanted link that will be sent by email.
   *
   * @param loginId      - User login ID
   * @param uri          - Base URI
   * @param loginOptions - {@link LoginOptions LoginOptions}
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signIn(
      String loginId,
      String uri,
      LoginOptions loginOptions)
      throws DescopeException;

  /**
   * Use to create a new user based on the given loginID either email or a phone.
   *
   * @param loginId - User login ID
   * @param uri     - Base URI
   * @param user    - {@link User User}
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signUp(String loginId, String uri, User user)
      throws DescopeException;

  /**
   * Use to login in using loginID, if user does not exist, a new user will be created.
   *
   * @param loginId - User login ID
   * @param uri     - Base URI
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signUpOrIn(String loginId, String uri)
      throws DescopeException;

  /**
   * Use to get a session that was generated by SignIn/SignUp request.
   *
   * @param pendingRef - pending Ref
   * @return authentication info
   * @throws DescopeException - error upon failure
   */
  AuthenticationInfo getSession(String pendingRef) throws DescopeException;

  /**
   * Use to verify a SignIn/SignUp request, based on the enchanted link token generated.
   *
   * @param token - token to be validate
   * @throws DescopeException - error upon failure
   */
  void verify(String token) throws DescopeException;

  /**
   * Use to update email and validate via enchanted link.
   *
   * @param loginId - User login ID
   * @param email   - User email
   * @param uri     - Base URI
   * @return masked address where the link was sent (email)
   * @throws DescopeException - error upon failure
   */
  String updateUserEmail(String loginId, String email, String uri)
      throws DescopeException;

}
