package com.descope.sdk.auth;

import com.descope.exception.DescopeException;
import com.descope.model.auth.AuthenticationInfo;
import com.descope.model.auth.UpdateOptions;
import com.descope.model.enchantedlink.EnchantedLinkResponse;
import com.descope.model.magiclink.LoginOptions;
import com.descope.model.magiclink.SignUpOptions;
import com.descope.model.user.User;
import java.util.Map;

public interface EnchantedLinkService {
  /**
   * Use to login a user based on an enchanted link that will be sent by email.
   *
   * @param loginId      - User login ID
   * @param uri          - Base URI
   * @param token        - when doing step-up or mfa then we need current session token
   * @param loginOptions - {@link LoginOptions LoginOptions}
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signIn(
      String loginId,
      String uri,
      String token,
      LoginOptions loginOptions)
      throws DescopeException;

  /**
   * Use to create a new user based on the given loginID either email or a phone.
   *
   * @param loginId - User login ID
   * @param uri     - Base URI
   * @param user    - {@link User User}
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signUp(String loginId, String uri, User user)
      throws DescopeException;

  /**
   * Use to create a new user based on the given loginID either email or a phone.
   *
   * @param loginId - User login ID
   * @param uri     - Base URI
   * @param user    - {@link User User}
   * @param signupOptions - optional claims and template strings
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signUp(String loginId, String uri, User user, SignUpOptions signupOptions)
      throws DescopeException;

  /**
   * Use to login in using loginID, if user does not exist, a new user will be created.
   *
   * @param loginId - User login ID
   * @param uri     - Base URI
   * @return pendingRef, linkId and masked email
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse signUpOrIn(String loginId, String uri)
      throws DescopeException;

  /**
   * Use to get a session that was generated by SignIn/SignUp request.
   *
   * @param pendingRef - pending Ref
   * @return authentication info
   * @throws DescopeException - error upon failure
   */
  AuthenticationInfo getSession(String pendingRef) throws DescopeException;

  /**
   * Use to verify a SignIn/SignUp request, based on the enchanted link token generated.
   *
   * @param token - token to be validate
   * @throws DescopeException - error upon failure
   */
  void verify(String token) throws DescopeException;

  /**
   * Use to update email and validate via enchanted link.
   *
   * @param loginId - User login ID
   * @param email   - User email
   * @param uri     - Base URI
   * @param refreshToken - refresh token to perform the update
   * @param updateOptions - update options for the update
   * @return {@link EnchantedLinkResponse} including masked address where the link was sent (email),
   *         link to chose and link to retrieve new session from
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse updateUserEmail(String loginId, String email, String uri, String refreshToken,
      UpdateOptions updateOptions) throws DescopeException;

  /**
   * Use to update email and validate via enchanted link.
   *
   * @param loginId - User login ID
   * @param email   - User email
   * @param uri     - Base URI
   * @param refreshToken - refresh token to perform the update
   * @param updateOptions - update options for the update
   * @return {@link EnchantedLinkResponse} including masked address where the link was sent (email),
   *         link to chose and link to retrieve new session from
   * @throws DescopeException - error upon failure
   */
  EnchantedLinkResponse updateUserEmail(String loginId, String email, String uri, String refreshToken,
      UpdateOptions updateOptions, Map<String, String> templateOptions) throws DescopeException;
}
