/*
 * Decompiled with CFR 0.152.
 */
package com.dynamic.sql.datasource;

import com.dynamic.sql.context.SqlContextHelper;
import com.dynamic.sql.context.properties.SchemaProperties;
import com.dynamic.sql.context.properties.SqlContextProperties;
import com.dynamic.sql.datasource.DataSourceMapping;
import com.dynamic.sql.datasource.DataSourceMeta;
import com.dynamic.sql.datasource.DataSourceProvider;
import com.dynamic.sql.datasource.DataSourceScanner;
import com.dynamic.sql.datasource.connection.ConnectionHolder;
import com.dynamic.sql.enums.DbType;
import com.dynamic.sql.enums.SqlDialect;
import com.dynamic.sql.plugins.schema.DbSchemaMatcher;
import com.dynamic.sql.utils.StringUtils;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSourceUtils {
    private static final Logger log = LoggerFactory.getLogger(DataSourceUtils.class);

    private DataSourceUtils() {
    }

    public static void scanAndInitDataSource(SqlContextProperties sqlContextProperties) {
        String[] packagePath = sqlContextProperties.getScanDatabasePackage();
        if (packagePath == null || packagePath.length == 0) {
            throw new IllegalArgumentException("The package path to search must be provided");
        }
        ArrayList<DataSourceMapping> dataSources = new ArrayList<DataSourceMapping>();
        for (String path : packagePath) {
            log.trace("Scanning {}", (Object)path);
            List<DataSourceMapping> dataSource = DataSourceScanner.findDataSource(path);
            dataSources.addAll(dataSource);
            log.trace("{} results found", (Object)dataSource.size());
        }
        if (dataSources.isEmpty()) {
            log.warn("No data sources found!!!");
            return;
        }
        Map<String, List<DataSourceMapping>> groupByDataSourceName = dataSources.stream().collect(Collectors.groupingBy(DataSourceMapping::getDataSourceName));
        groupByDataSourceName.forEach((dataSourceName, dataSourceList) -> {
            if (dataSourceList.size() > 1) {
                throw new IllegalStateException("Found more than one data source: " + dataSourceName);
            }
        });
        List dataSourceNames = dataSources.stream().map(DataSourceMapping::getDataSourceName).collect(Collectors.toList());
        sqlContextProperties.getSchemaProperties().forEach(schemaProperties -> {
            if (!dataSourceNames.contains(schemaProperties.getDataSourceName())) {
                throw new IllegalArgumentException("Unable to match data source name: " + schemaProperties.getDataSourceName());
            }
        });
        for (DataSourceMapping dataSourceMapping : dataSources) {
            DataSourceUtils.checkAndSave(sqlContextProperties, dataSourceMapping);
        }
    }

    public static void checkAndSave(SqlContextProperties sqlContextProperties, DataSourceMapping dataSourceMapping) {
        log.debug("Test the data source connection to get the URL For '{}'. ", (Object)dataSourceMapping.getDataSourceName());
        Connection connection = null;
        SchemaProperties schemaProperties = DataSourceUtils.getSchemaProperties(sqlContextProperties, dataSourceMapping.getDataSourceName());
        try {
            connection = ConnectionHolder.getConnection(dataSourceMapping.getDataSource());
            DatabaseMetaData metaData = connection.getMetaData();
            DbType dbType = DataSourceUtils.matchDbType(metaData.getURL());
            String schema = DataSourceUtils.matchSchema(sqlContextProperties.getSchemaMatchers(), dbType, metaData.getURL());
            String version = metaData.getDatabaseProductVersion();
            schemaProperties.setDatabaseProductVersion(dbType, StringUtils.isEmpty(schemaProperties.getDatabaseProductVersion()) ? version : schemaProperties.getDatabaseProductVersion());
            SqlDialect sqlDialect = schemaProperties.getSqlDialect();
            if (sqlDialect == null && dbType.equals((Object)DbType.OTHER)) {
                log.error("Unsupported SQL dialect, If your database supports an existing SQL dialect, manually specify the dialect type.");
                throw new UnsupportedOperationException("Unsupported SQL dialect");
            }
            sqlDialect = sqlDialect == null ? SqlDialect.valueOf(dbType.name()) : sqlDialect;
            schemaProperties.setSqlDialect(sqlDialect);
            schemaProperties.setDataSourceName(dataSourceMapping.getDataSourceName());
            dataSourceMapping.setAllowDataSourceDefinitionOverriding(sqlContextProperties.isAllowDataSourceDefinitionOverriding());
            dataSourceMapping.setGlobalDefault(schemaProperties.isGlobalDefault());
            dataSourceMapping.setBindBasePackages(schemaProperties.getBindBasePackages());
            DataSourceUtils.initDataSource(dataSourceMapping, dbType, schema, version);
            SqlContextHelper.addSchemaProperties(sqlContextProperties);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to read meta information", e);
        }
        finally {
            ConnectionHolder.releaseConnection(dataSourceMapping.getDataSource(), connection);
        }
    }

    public static synchronized void initDataSource(DataSourceMapping dataSourceMapping, DbType dbType, String schema, String version) {
        log.debug("Initializing DataSource: {}, schema:{}, isGlobalDefault:{}, bindBasePackages:{}", new Object[]{dataSourceMapping.getDataSourceName(), schema, dataSourceMapping.isGlobalDefault(), dataSourceMapping.getBindBasePackages()});
        if (StringUtils.isBlank(dataSourceMapping.getDataSourceName())) {
            throw new IllegalArgumentException("The bean name must be provided");
        }
        if (dataSourceMapping.getDataSource() == null) {
            throw new IllegalArgumentException("The dataSource must be provided");
        }
        if (StringUtils.isBlank(schema)) {
            throw new IllegalArgumentException("The schema must be provided");
        }
        if (dbType == null) {
            throw new IllegalArgumentException("The dbType must be provided");
        }
        DataSourceMeta meta = new DataSourceMeta();
        meta.setSchema(schema);
        meta.setGlobalDefault(dataSourceMapping.isGlobalDefault());
        meta.setBindBasePackages(dataSourceMapping.getBindBasePackages());
        meta.setDataSource(dataSourceMapping.getDataSource());
        meta.setDbType(dbType);
        meta.setVersion(version);
        DataSourceProvider.saveDataSourceMeta(dataSourceMapping.isAllowDataSourceDefinitionOverriding(), dataSourceMapping.getDataSourceName(), meta);
        log.info("Initialized DataSource: {}", (Object)dataSourceMapping.getDataSourceName());
    }

    public static DbType matchDbType(String jdbcUrl) {
        if (jdbcUrl.startsWith("jdbc:mysql:")) {
            return DbType.MYSQL;
        }
        if (jdbcUrl.startsWith("jdbc:mariadb:")) {
            return DbType.MARIADB;
        }
        if (jdbcUrl.startsWith("jdbc:oracle:")) {
            return DbType.ORACLE;
        }
        return DbType.OTHER;
    }

    public static String matchSchema(Set<DbSchemaMatcher> dbSchemaMatchers, DbType dbType, String url) {
        for (DbSchemaMatcher schemaMatcher : dbSchemaMatchers) {
            if (!schemaMatcher.supports(dbType)) continue;
            String schema = schemaMatcher.matchSchema(url);
            if (StringUtils.isNotBlank(schema)) {
                return schema;
            }
            log.warn("'{}' supports the '{}' type but returns an empty string when parsing the database name", (Object)schemaMatcher.getClass().getCanonicalName(), (Object)dbType);
        }
        throw new IllegalArgumentException("Unsupported jdbc url: " + url);
    }

    private static SchemaProperties getSchemaProperties(SqlContextProperties sqlContextProperties, String dataSourceName) {
        for (SchemaProperties schemaProperty : sqlContextProperties.getSchemaProperties()) {
            if (!schemaProperty.getDataSourceName().equals(dataSourceName)) continue;
            return schemaProperty;
        }
        SchemaProperties schemaProperties = new SchemaProperties();
        sqlContextProperties.getSchemaProperties().add(schemaProperties);
        return schemaProperties;
    }
}

