package timber.log;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A {@link Tree Tree} for debug builds. Automatically infers the tag from the calling class.
 */
public class DebugTree extends Tree {

    private static final int MAX_LOG_LENGTH = 4000;
    private static final int CALL_STACK_INDEX = 8;
    private static final Pattern ANONYMOUS_CLASS = Pattern.compile("(\\$\\d+)+$");
    private final SimpleDateFormat SIMPLEDATEFORMAT = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS");

    @Override
    protected String getTag() {
        String tag = super.getTag();
        if (tag != null) return tag;

        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        if (stackTrace.length <= CALL_STACK_INDEX) {
            throw new IllegalStateException(
                    "Synthetic stacktrace didn't have enough elements: are you using proguard?");
        }
        return createStackElementTag(stackTrace[CALL_STACK_INDEX]);
    }

    /**
     * Extract the tag which should be used for the message from the {@code element}. By default
     * this will use the class name without any anonymous class suffixes (e.g., {@code Foo$1}
     * becomes {@code Foo}).
     * <p>
     * Note: This will not be called if a {@linkplain Timber#tag(String) manual tag} was specified.
     */
    protected String createStackElementTag(StackTraceElement element) {
        String tag = element.getClassName();
        Matcher m = ANONYMOUS_CLASS.matcher(tag);
        if (m.find()) {
            tag = m.replaceAll("");
        }
        return tag.substring(tag.lastIndexOf('.') + 1).replaceAll("\\$", ".");
    }

    /**
     * Break up {@code message} into maximum-length chunks (if needed) and send to either
     * {@link System#out stdout} or
     * {@link System#err stderr} for logging.
     * <p>
     * {@inheritDoc}
     */
    @Override
    protected void log(Level level, String tag, String message, Throwable t) {
        if (message.length() < MAX_LOG_LENGTH) {
            if (level.getLevelCode() <= Level.WARN.getLevelCode()) System.err.println(formatLine(level, tag, message));
            else System.out.println(formatLine(level, tag, message));
            return;
        }

        for (int i = 0, length = message.length(); i < length; i++) {
            int newline = message.indexOf('\n', i);
            newline = newline != -1 ? newline : length;
            do {
                int end = Math.min(newline, i + MAX_LOG_LENGTH);
                String part = message.substring(i, end);
                if (level.getLevelCode() <= Level.WARN.getLevelCode())
                    System.err.println(formatLine(level, tag, part));
                else
                    System.out.println(formatLine(level, tag, part));
                i = end;
            } while (i < newline);
        }

    }

    private String formatLine(Level level, String tag, String message) {
        return String.format("[%s][%s/%s]: %s", SIMPLEDATEFORMAT.format(new Date()), level.getLevelName().toUpperCase(), tag, message);
    }

}
