package com.github.azbh111.utils.java.date;


import com.github.azbh111.utils.java.datetime.DateTimeUtils;

import java.time.*;
import java.time.chrono.ChronoLocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.util.*;

/**
 *
 * 基于java8 time api的封装
 * java8 time api所有类都是不可变得,线程安全的
 *
 * @author pyz
 * @date 2019/3/12 12:19 AM
 */
public class DateUtils {
    public static final Map<String, DateTimeFormatter> standardDateFormatters = new LinkedHashMap<>();
    /**
     * 最开始的一天是星期几
     */
    public static final int ZERO_DAY_OF_WEEK = LocalDate.of(1970, 1, 1).getDayOfWeek().getValue();

    public static final Clock SYSTEM_DEFAULT_ZONE_CLOCK = Clock.systemDefaultZone();
    public static final Clock UTC_CLOCK = Clock.systemUTC();
    public static final ZoneId SYSTEM_DEFAULT_ZONE_ID = SYSTEM_DEFAULT_ZONE_CLOCK.getZone();
    public static final ZoneId UTC_ZONE_ID = UTC_CLOCK.getZone();

    static {
        standardDateFormatters.put("ISO_LOCAL_DATE", DateTimeFormatter.ISO_LOCAL_DATE);
        standardDateFormatters.put("ISO_DATE", DateTimeFormatter.ISO_DATE);
        standardDateFormatters.put("ISO_OFFSET_DATE", DateTimeFormatter.ISO_OFFSET_DATE);
        standardDateFormatters.put("ISO_ORDINAL_DATE", DateTimeFormatter.ISO_ORDINAL_DATE);
        standardDateFormatters.put("ISO_WEEK_DATE", DateTimeFormatter.ISO_WEEK_DATE);
        standardDateFormatters.put("BASIC_ISO_DATE", DateTimeFormatter.BASIC_ISO_DATE);
    }

    /**
     *
     * @param temporal
     * @return
     */
    public static LocalDate from(TemporalAccessor temporal) {
        if (temporal instanceof LocalDate) {
            return (LocalDate) temporal;
        } else if (temporal instanceof Instant) {
            return ((Instant) temporal).atZone(SYSTEM_DEFAULT_ZONE_ID).toLocalDate();
        }
        try {
            int year = temporal.get(ChronoField.YEAR);
            int month = temporal.get(ChronoField.MONTH_OF_YEAR);
            int day = temporal.get(ChronoField.DAY_OF_MONTH);
            return LocalDate.of(year, month, day);
        } catch (DateTimeException ex) {
            throw new DateTimeException("Unable to obtain LocalDateTime from TemporalAccessor: " +
                    temporal + " of type " + temporal.getClass().getName(), ex);
        }
    }

    public static Date toDate(LocalDate ld) {
        return Date.from(ld.atStartOfDay().atZone(SYSTEM_DEFAULT_ZONE_ID).toInstant());
    }

    public static LocalDate from(Date date) {
        return date.toInstant().atZone(SYSTEM_DEFAULT_ZONE_ID).toLocalDate();
    }

    public static LocalDate now() {
        return LocalDate.now();
    }

    /**
     * 日期从from按时按天递增,包含from,总共days天
     *
     * @param from 起始日期
     * @param days 天数
     * @return
     */
    public static List<LocalDate> flatDate(LocalDate from, int days) {
        List<LocalDate> dates = new ArrayList<>(days);
        for (int i = 0; i < days; i++) {
            dates.add(from.plusDays(i));
        }
        return dates;
    }

    public static LocalDate firstDayOfWeek() {
        return firstDayOfWeek(now());
    }

    public static LocalDate firstDayOfWeek(Date d) {
        return firstDayOfWeek(from(d));
    }

    public static LocalDate firstDayOfWeek(LocalDate dt) {
        int dayOfWeek = dt.getDayOfWeek().getValue();
        if (dayOfWeek > 1) {
            return dt.minusDays(dayOfWeek - 1);
        } else {
            return dt;
        }
    }

    public static LocalDate lastDayOfWeek() {
        return lastDayOfWeek(now());
    }

    public static LocalDate lastDayOfWeek(Date d) {
        return lastDayOfWeek(from(d));
    }

    public static LocalDate lastDayOfWeek(LocalDate d) {
        int dayOfWeek = d.getDayOfWeek().getValue();
        if (dayOfWeek < 7) {
            return d.plusDays(7 - dayOfWeek);
        } else {
            return d;
        }
    }

    public static LocalDate firstDayOfMonth() {
        return firstDayOfMonth(now());
    }

    public static LocalDate firstDayOfMonth(Date d) {
        return firstDayOfMonth(from(d));
    }

    public static LocalDate firstDayOfMonth(LocalDate dt) {
        return dt.with(TemporalAdjusters.firstDayOfMonth());
    }

    public static LocalDate lastDayOfMonth() {
        return lastDayOfMonth(now());
    }

    public static LocalDate lastDayOfMonth(Date d) {
        return lastDayOfMonth(from(d));
    }

    public static LocalDate lastDayOfMonth(LocalDate dt) {
        return dt.with(TemporalAdjusters.lastDayOfMonth());
    }

    public static LocalDate firstDayOfYear() {
        return firstDayOfYear(now());
    }

    public static LocalDate firstDayOfYear(Date d) {
        return firstDayOfYear(from(d));
    }

    public static LocalDate firstDayOfYear(LocalDate dt) {
        return dt.with(TemporalAdjusters.firstDayOfYear());
    }

    public static LocalDate lastDayOfYear() {
        return lastDayOfYear(now());
    }

    public static LocalDate lastDayOfYear(Date d) {
        return lastDayOfYear(from(d));
    }

    public static LocalDate lastDayOfYear(LocalDate dt) {
        return dt.with(TemporalAdjusters.lastDayOfYear());
    }


    public static long toEpochDay() {
        return toEpochDay(DateUtils.now());
    }

    public static long toEpochDay(Date d) {
        return toEpochDay(DateUtils.from(d));
    }

    public static long toEpochDay(ChronoLocalDate d) {
        return d.toEpochDay();
    }

    public static LocalDate of(int epochDay) {
        return LocalDate.ofEpochDay(epochDay);
    }

    public static int toEpochWeek() {
        return toEpochWeek(DateUtils.now());
    }

    public static int toEpochWeek(Date d) {
        return toEpochWeek(DateUtils.from(d));
    }

    public static int toEpochWeek(ChronoLocalDate ld) {
        return (int) ((ld.toEpochDay() + ZERO_DAY_OF_WEEK - 1) / 7);
    }

    public static LocalDate of(int epochWeek, int dayOfWeek) {
        return of(epochWeek * 7 - ZERO_DAY_OF_WEEK + dayOfWeek);
    }

    public static int dayOfWeek() {
        return dayOfWeek(LocalDate.now());
    }

    public static int dayOfWeek(LocalDate ld) {
        return ld.getDayOfWeek().getValue();
    }

    public static int dayOfMonth() {
        return dayOfMonth(now());
    }


    public static int dayOfMonth(LocalDate ld) {
        return ld.getDayOfMonth();
    }

    public static int monthOfYear() {
        return monthOfYear(now());
    }

    public static int monthOfYear(LocalDate ld) {
        return ld.getMonthValue();
    }

    public static boolean isLeapYear() {
        return isLeapYear(DateUtils.now());
    }

    public static boolean isLeapYear(Date d) {
        return isLeapYear(DateUtils.from(d));
    }

    public static boolean isLeapYear(ChronoLocalDate d) {
        return d.isLeapYear();
    }


    public static int lengthOfMoneth() {
        return lengthOfMoneth(DateUtils.now());
    }


    public static int lengthOfMoneth(Date d) {
        return lengthOfMoneth(DateUtils.from(d));
    }


    public static int lengthOfMoneth(ChronoLocalDate d) {
        return d.lengthOfMonth();
    }


    public static int lengthOfYear() {
        return lengthOfYear(DateUtils.now());
    }


    public static int lengthOfYear(Date d) {
        return lengthOfYear(DateUtils.from(d));
    }


    public static int lengthOfYear(ChronoLocalDate d) {
        return d.lengthOfYear();
    }

    public static LocalDate parse(String datetimeStr, String format) {
        DateTimeFormatter f = DateTimeUtils.getOrCreateDateTimeFormatter(format);
        return LocalDate.parse(datetimeStr, f);
    }


    /**
     * 用常用格式尝试解析
     * 若无法解析,抛出ParseException异常
     *
     * @param str
     * @return
     */
    public static LocalDate tryParse(String str) {
        TemporalAccessor t = null;
        LocalDate dt;
        for (DateTimeFormatter formatter : standardDateFormatters.values()) {
            try {
                t = formatter.parse(str, Instant::from);
                dt = from(t);
            } catch (Throwable e1) {
                try {
                    t = formatter.parse(str);
                    dt = from(t);
                } catch (DateTimeParseException e2) {
                    continue;
                }
            }
            return dt;
        }
        throw new DateParseException("Unable to parse " + str + " to LocalDate");
    }

    public static boolean isSameDay(Date d1, Date d2) {
        return from(d1).equals(from(d2));
    }

}
