/*
 * Copyright (c) 2014-2019, Deepspring Healthcare Inc. All rights reserved.
 */

package com.github.azbh111.utils.java.iterable;


import com.github.azbh111.utils.java.annotation.Nonnull;
import com.github.azbh111.utils.java.annotation.Nullable;
import com.github.azbh111.utils.java.exception.ExceptionUtils;
import com.github.azbh111.utils.java.iterable.model.DifferenceView;
import com.github.azbh111.utils.java.iterable.model.ObjectSumFunction;
import com.github.azbh111.utils.java.map.MapUtils;
import com.github.azbh111.utils.java.predicate.PredicateUtils;
import com.github.azbh111.utils.java.reflect.ReflectUtils;

import java.lang.reflect.Field;
import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * @author pyz
 * @date 2018/10/30 9:52 AM
 */
public class IterableUtils {

    /**
     * 按照给定的顺序排序
     * sortedKeys中没有的元素,会被过滤掉
     *
     * @param list       经过keyMapper映射,不能用重复的key
     * @param sortedKeys
     * @param keyMapper
     * @param <T>
     * @param <K>
     * @return 排序后的集合
     */
    public static <T, K> List<T> sortByKeys(@Nullable Iterable<T> list, @Nonnull Collection<K> sortedKeys, @Nonnull Function<T, K> keyMapper) {
        Map<K, T> valMap = toStream(list).collect(Collectors.toMap(keyMapper, Function.identity()));
        return sortedKeys.stream().map(i -> valMap.get(i)).filter(Objects::nonNull).collect(Collectors.toList());
    }

    /**
     * 将多个对象合计到一个对象里
     *
     * @param it
     * @param container
     * @param fieldFilter
     * @param sumFunction
     * @param <T>
     */
    public static <T> void sum(Iterable<T> it, T container, Predicate<Field> fieldFilter, ObjectSumFunction<T> sumFunction) {
        if (it == null) {
            return;
        }
        if (isNullOrEmpty(it)) {
            return;
        }
        T firstElement = getFirstElement(it);
        List<Field> fields = ReflectUtils.getAllFields(firstElement.getClass(), fieldFilter);
        for (Field field : fields) {
            field.setAccessible(true);
        }
        for (T t : it) {
            for (Field field : fields) {
                try {
                    Object appendValue = field.get(t);
                    Object currentValue = field.get(container);
                    Object newValue = sumFunction.sum(field, currentValue, t, appendValue);
                    field.set(container, newValue);
                } catch (IllegalAccessException e) {
                    ExceptionUtils.throwException(e);
                }
            }
        }
    }


    /**
     * 对Iterator进行计数
     * 会将该Iterator剩余元素全部遍历一次
     *
     * @param iterator
     * @param <T>
     * @return
     */
    public static <T> int count(Iterator<T> iterator) {
        if (iterator == null) {
            return 0;
        }
        int count = 0;
        while (iterator.hasNext()) {
            iterator.next();
            count++;
        }
        return count;
    }

    /**
     * 对Iterable进行计数
     *
     * @param iterable
     * @param <T>
     * @return
     */
    public static <T> int count(Iterable<T> iterable) {
        if (iterable == null) {
            return 0;
        }
        if (iterable instanceof Collection) {
            return ((Collection) iterable).size();
        }
        int count = 0;
        for (T t : iterable) {
            count++;
        }
        return count;
    }

    /**
     * 对比两个集合的差异
     *
     * @param leftList
     * @param rightList
     * @param <T>
     * @return
     */
    public static <T> DifferenceView<Set<T>> difference(Set<T> leftList, Set<T> rightList) {
        return difference(leftList, rightList, new HashSet<T>(), new HashSet<T>(), new HashSet<T>(), Function.identity());
    }

    /**
     * 对比两个集合的差异
     *
     * @param leftList
     * @param rightList
     * @param identityMapper
     * @param <T>
     * @param <K>
     * @return
     */
    public static <T, K> DifferenceView<Set<T>> difference(Set<T> leftList, Set<T> rightList, Function<T, K> identityMapper) {
        return difference(leftList, rightList, new HashSet<T>(), new HashSet<T>(), new HashSet<T>(), identityMapper);
    }

    /**
     * 对比两个集合的差异
     *
     * @param leftList
     * @param rightList
     * @param <T>
     * @param <K>
     * @return
     */
    public static <T, K> DifferenceView<List<T>> difference(List<T> leftList, List<T> rightList) {
        return difference(leftList, rightList, Function.identity());
    }

    /**
     * 对比两个集合的差异
     *
     * @param leftList
     * @param rightList
     * @param identityMapper
     * @param <T>
     * @param <K>
     * @return
     */
    public static <T, K> DifferenceView<List<T>> difference(List<T> leftList, List<T> rightList, Function<T, K> identityMapper) {
        return difference(leftList, rightList, new ArrayList<T>(), new ArrayList<T>(), new ArrayList<T>(), identityMapper);
    }

    private static <C extends Collection<T>, T, K> DifferenceView<C> difference(Collection<T> leftList, Collection<T> rightList, C left, C common, C right, Function<T, K> identityMapper) {
        leftList = leftList == null ? new ArrayList<>() : leftList;
        rightList = rightList == null ? new ArrayList<>() : rightList;
        Map<K, T> leftMap = new LinkedHashMap<>();
        for (T t : leftList) {
            leftMap.put(identityMapper.apply(t), t);
        }
        for (T t : rightList) {
            K k = identityMapper.apply(t);
            if (leftMap.containsKey(k)) {
                leftMap.remove(k);
                common.add(t);
                continue;
            } else {
                right.add(t);
            }
        }
        left.addAll(leftMap.values());
        return DifferenceView.<C>builder()
                .leftOnly(left)
                .common(common)
                .rightOnly(right)
                .build();
    }

    public static <T> void forEach(Consumer<T> consumer, Iterable<T>... its) {
        if (its == null) {
            return;
        }
        for (Iterable<T> it : its) {
            for (T t : it) {
                consumer.accept(t);
            }
        }
    }

    /**
     * 去重
     *
     * @param it
     * @param <T>
     */
    public static <T> void distinct(Iterable<T> it) {
        distinct(it, Function.identity());
    }

    /**
     * 去重
     *
     * @param it
     * @param distinctKey
     * @param <T>
     * @param <K>
     */
    public static <T, K> void distinct(Iterable<T> it, Function<T, K> distinctKey) {
        if (it == null) {
            return;
        }
        Iterator<T> iterator = it.iterator();
        Set<K> set = new HashSet<>();
        while (iterator.hasNext()) {
            if (!set.add(distinctKey.apply(iterator.next()))) {
                iterator.remove();
            }
        }
    }

    public static <T> boolean isNullOrEmpty(Iterable<? extends T> it) {
        return it == null || !it.iterator().hasNext();
    }

    public static <T> boolean isNotNullOrEmpty(Iterable<? extends T> it) {
        return it != null && it.iterator().hasNext();
    }

    /**
     * 获取最后一个元素
     *
     * @param it
     * @param <T>
     * @return
     */
    public static <T> T getLastElement(Iterable<T> it) {
        if (isNullOrEmpty(it)) {
            return null;
        }
        if (it instanceof List) {
            if (it instanceof Deque) {
                Deque<T> deque = (Deque<T>) it;
                return deque.getLast();
            } else {
                List<T> list = (List<T>) it;
                return list.get(list.size() - 1);
            }
        }
        T last = null;
        for (T t : it) {
            last = t;
        }
        return last;
    }

    /**
     * 返回集合第一个元素(无序集合返回值不固定)
     *
     * @param it
     * @param <T>
     * @return
     */
    public static <T> T getFirstElement(Iterable<T> it) {
        if (it == null) {
            return null;
        }
        Iterator<T> iterator = it.iterator();
        return iterator.hasNext() ? iterator.next() : null;
    }

    public static <T> List<T> filter(Iterable<T> it, Predicate<T> test) {
        return toStream(it)
                .filter(test)
                .collect(Collectors.toList());
    }

    public static <T> Optional<T> findFirst(Iterable<T> it, Predicate<T> test) {
        return toStream(it).filter(test).findFirst();
    }

    /**
     * 合计
     *
     * @param target
     * @param partMap
     * @param partSetter
     * @param <Item>
     * @param <Key>
     * @param <Part>
     */
    public static <Item, Key, Part> void aggregate(Map<Key, Item> target, Map<Key, Part> partMap, BiConsumer<Item, Part> partSetter) {
        for (Map.Entry<Key, Item> entry : target.entrySet()) {
            Part part = partMap.get(entry.getKey());
            if (part != null) {
                partSetter.accept(entry.getValue(), part);
            }
        }
        return;
    }

    /**
     * 合计
     *
     * @param target
     * @param parts
     * @param keyGetter
     * @param partSetter
     * @param <Item>
     * @param <Key>
     * @param <Part>
     */
    public static <Item, Key, Part> void aggregate(Map<Key, Item> target, Iterable<Part> parts, Function<Part, Key> keyGetter, BiConsumer<Item, Part> partSetter) {
        for (Part part : parts) {
            Key key = keyGetter.apply(part);
            Item item = target.get(key);
            if (item != null) {
                partSetter.accept(item, part);
            }
        }
        return;
    }

    /**
     * 合计
     *
     * @param target
     * @param partMap
     * @param keyGetter
     * @param partSetter
     * @param <Item>
     * @param <Key>
     * @param <Part>
     */
    public static <Item, Key, Part> void aggregate(Iterable<Item> target, Map<Key, Part> partMap, Function<Item, Key> keyGetter, BiConsumer<Item, Part> partSetter) {
        for (Item item : target) {
            Part part = partMap.get(keyGetter.apply(item));
            if (part == null) {
                continue;
            }
            partSetter.accept(item, part);
        }
        return;
    }

    // =====================groupingBy=========================
    public static <T, R> Map<R, List<T>> groupingBy(Iterable<? extends T> it, Function<? super T, ? extends R> keyMapper) {
        return groupingBy(it, keyMapper, Function.identity());
    }

    public static <T, R> Map<R, List<T>> groupingBy(Iterator<? extends T> it, Function<? super T, ? extends R> keyMapper) {
        return groupingBy(it, keyMapper, Function.identity());
    }

    public static <T, R> Map<R, List<T>> groupingBy(T[] it, Function<? super T, ? extends R> keyMapper) {
        return groupingBy(it, keyMapper, Function.identity());
    }

    public static <T, R> Map<R, List<T>> groupingBy(Stream<? extends T> it, Function<? super T, ? extends R> keyMapper) {
        return groupingBy(it, keyMapper, Function.identity());
    }

    public static <T, K, V> Map<K, List<V>> groupingBy(Iterable<? extends T> it, Function<? super T, ? extends K> keyMapper, Function<? super T, ? extends V> valueMapper) {
        return groupingBy(toStream(it), keyMapper, valueMapper);
    }

    public static <T, K, V> Map<K, List<V>> groupingBy(Iterator<? extends T> it, Function<? super T, ? extends K> keyMapper, Function<? super T, ? extends V> valueMapper) {
        return groupingBy(toStream(it), keyMapper, valueMapper);
    }

    public static <T, K, V> Map<K, List<V>> groupingBy(T[] it, Function<? super T, ? extends K> keyMapper, Function<? super T, ? extends V> valueMapper) {
        return groupingBy(toStream(it), keyMapper, valueMapper);
    }

    public static <T, K, V> Map<K, List<V>> groupingBy(Stream<? extends T> it, Function<? super T, ? extends K> keyMapper, Function<? super T, ? extends V> valueMapper) {
        return groupingBy(it, new HashMap<>(), keyMapper, valueMapper);
    }

    public static <T, K, V> Map<K, List<V>> groupingBy(Stream<? extends T> it, Map<K, List<V>> resultContainer, Function<? super T, ? extends K> keyMapper, Function<? super T, ? extends V> valueMapper) {
        it.forEach(t -> {
            // java stream api中的groupingBy要求key非空,这里同样
            MapUtils.getOrCreate(resultContainer,
                    Objects.requireNonNull(keyMapper.apply(t), "element cannot be mapped to a null key"),
                    ArrayList::new)
                    .add(valueMapper.apply(t));
        });
        return resultContainer;
    }

    // ====================groupingBy==========================

    // ====================collect==========================

    public static <T, R, A> R collect(T[] it, Collector<? super T, A, R> collector) {
        return collect(toStream(it), collector);
    }

    public static <T, R, A> R collect(Iterable<T> it, Collector<? super T, A, R> collector) {
        return collect(toStream(it), collector);
    }

    public static <T, R, A> R collect(Iterator<T> it, Collector<? super T, A, R> collector) {
        return collect(toStream(it), collector);
    }

    public static <T, R, A> R collect(Stream<? extends T> it, Collector<? super T, A, R> collector) {
        return it.collect(collector);
    }

    // ====================collect==========================


    // ====================toSet==========================
    public static <T> Set<T> toSet(Iterable<? extends T> it) {
        return toSet(it, Function.identity());
    }

    public static <T> Set<T> toSet(T[] it) {
        return toSet(it, Function.identity());
    }

    public static <T> Set<T> toSet(Stream<? extends T> it) {
        return toSet(it, Function.identity());
    }

    public static <T, R> Set<R> toSet(Iterable<? extends T> it, Function<? super T, ? extends R> func) {
        return toSet(toStream(it), func);
    }

    public static <T, R> Set<R> toSet(Iterable<? extends T> it, Function<? super T, ? extends R> func, Predicate<? super T> predicate) {
        return toSet(toStream(it), func, predicate);
    }

    public static <T, R> Set<R> toSet(T[] it, Function<? super T, ? extends R> func) {
        return toSet(toStream(it), func);
    }

    public static <T, R> Set<R> toSet(Stream<? extends T> it, Function<? super T, ? extends R> func) {
        return it.map(func).collect(Collectors.toSet());
    }

    public static <T, R> Set<R> toSet(Stream<? extends T> it, Function<? super T, ? extends R> func, Predicate<? super T> predicate) {
        return it.filter(predicate).map(func).collect(Collectors.toSet());
    }


    // ====================toSet==========================


    // ====================flatToList==========================
    public static <T, R> List<R> flatToList(Iterable<? extends T> it, Function<? super T, Iterable<R>> func) {
        return flatToList(toStream(it), func);
    }

    public static <T, R> List<R> flatToList(T[] it, Function<? super T, Iterable<R>> func) {
        return flatToList(toStream(it), func);
    }

    public static <T, R> List<R> flatToList(Stream<? extends T> it, Function<? super T, Iterable<R>> func) {
        return it
                .flatMap(item -> toStream(func.apply(item)))
                .collect(Collectors.toList());
    }
    // ====================flatToList==========================

    // ====================flatToSet==========================
    public static <T, R> Set<R> flatToSet(Iterable<? extends T> it, Function<? super T, Iterable<R>> func) {
        return flatToSet(toStream(it), func);
    }

    public static <T, R> Set<R> flatToSet(T[] it, Function<? super T, Iterable<R>> func) {
        return flatToSet(toStream(it), func);
    }

    public static <T, R> Set<R> flatToSet(Stream<? extends T> it, Function<? super T, Iterable<R>> func) {
        return it
                .flatMap(item -> toStream(func.apply(item)))
                .collect(Collectors.toSet());
    }
    // ====================flatToList==========================

    // ====================toList==========================
    public static <T> List<T> toList(Iterable<? extends T> it) {
        return toList(it, Function.identity());
    }

    public static <T> List<T> toList(T[] it) {
        return toList(it, Function.identity());
    }

    public static <T> List<T> toList(Stream<? extends T> it) {
        return toList(it, Function.identity());
    }

    public static <T, R> List<R> toList(Iterable<? extends T> it, Function<? super T, R> func) {
        return toList(toStream(it), func);
    }

    public static <T, R> List<R> toList(T[] it, Function<? super T, R> func) {
        return toList(toStream(it), func);
    }

    public static <T, R> List<R> toList(Stream<? extends T> it, Function<? super T, R> func) {
        return it
                .map(func)
                .collect(Collectors.toList());
    }
    // ====================toList==========================


    // ====================toDistinctedList==========================
    public static <T> List<T> toDistinctedList(Iterable<? extends T> it) {
        return toStream(it).distinct().collect(Collectors.toList());
    }

    public static <T, R> List<R> toDistinctedList(Iterable<? extends T> it, Function<? super T, R> func) {
        return toDistinctedList(toStream(it), func);
    }

    public static <T, R> List<R> toDistinctedList(Stream<? extends T> it, Function<? super T, R> func) {
        return it
                .map(func)
                .distinct()
                .collect(Collectors.toList());
    }

    public static <T, R> List<T> toDistinctedListBy(Iterable<? extends T> it, Function<? super T, R> func) {
        return toDistinctedListBy(toStream(it), func);
    }

    public static <T, R> List<T> toDistinctedListBy(Stream<? extends T> it, Function<? super T, R> func) {
        List<T> list = new ArrayList<>();
        Predicate<? super T> test = PredicateUtils.distinctBy(func);
        it.forEach(item -> {
            if (test.test(item)) {
                list.add(item);
            }
        });
        return list;
    }
    // ====================toDistinctedList==========================

    // ====================toMap==========================
    public static <T, R> Map<R, T> toMap(Iterable<? extends T> it, Function<? super T, ? extends R> keyMapper) {
        return toMap(it, keyMapper, Function.identity());
    }

    public static <T, R> Map<R, T> toMap(T[] it, Function<? super T, ? extends R> keyMapper) {
        return toMap(it, keyMapper, Function.identity());
    }

    public static <T, R> Map<R, T> toMap(Stream<? extends T> it, Function<? super T, ? extends R> keyMapper) {
        return toMap(it, keyMapper, Function.identity());
    }

    public static <T, R, X> Map<R, X> toMap(Iterable<? extends T> it, Function<? super T, ? extends R> keyMapper, Function<? super T, ? extends X> valueMapper) {
        return toMap(toStream0(it), keyMapper, valueMapper);
    }

    public static <T, R, X> Map<R, X> toMap(T[] it, Function<? super T, ? extends R> keyMapper, Function<? super T, ? extends X> valueMapper) {
        return toMap(toStream(it), keyMapper, valueMapper);
    }

    public static <T, R, X> Map<R, X> toMap(Stream<? extends T> it, Function<? super T, ? extends R> keyMapper, Function<? super T, ? extends X> valueMapper) {
        return it.collect(Collectors.toMap(keyMapper, valueMapper));
    }
    // ====================toMap==========================


    // ====================toStream==========================

    public static <T> Stream<T> toStream(Iterator<T> it, Iterator<T>... its) {
        Stream<T> stream = toStream0(it);
        if (its == null) {
            return stream;
        }
        for (Iterator<T> ts : its) {
            stream = Stream.concat(stream, toStream0(ts));
        }
        return stream;
    }

    public static <T> Stream<T> toStream(Iterable<T> it, Iterable<T>... its) {
        Stream<T> stream = toStream0(it);
        if (its == null) {
            return stream;
        }
        for (Iterable<T> ts : its) {
            stream = Stream.concat(stream, toStream0(ts));
        }
        return stream;
    }

    public static <T> Stream<T> toStream(T[] it, T[]... its) {
        Stream<T> stream = toStream0(it);
        if (its == null) {
            return stream;
        }
        for (T[] ts : its) {
            stream = Stream.concat(stream, toStream0(ts));
        }
        return stream;
    }

    public static <T> Stream<T> toStream(Stream<T> s, Stream<T>... ss) {
        Stream<T> stream = s == null ? Stream.empty() : s;
        for (Stream ts : ss) {
            stream = Stream.concat(stream, ts);
        }
        return stream;
    }

    private static <T> Stream<T> toStream0(Iterator<T> it) {
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(it, Spliterator.ORDERED), false);
    }

    private static <T> Stream<T> toStream0(Iterable<T> it) {
        if (it == null) {
            return Stream.empty();
        }
        return StreamSupport.stream(it.spliterator(), false);
    }

    private static <T> Stream<T> toStream0(T[] it) {
        if (it == null) {
            return Stream.empty();
        }
        return StreamSupport.stream(Arrays.spliterator(it), false);
    }


    // ====================max==========================
    public static <T extends Comparable> T max(Iterable<? extends T> it) {
        return max(it, Comparable::compareTo);
    }

    public static <T extends Comparable> T max(T[] it) {
        return max(it, Comparable::compareTo);
    }

    public static <T extends Comparable> T max(Stream<? extends T> it) {
        return max(it, Comparable::compareTo);
    }

    public static <T> T max(Iterable<T> it, Comparator<T> comparator) {
        return max(toStream(it), comparator);
    }

    public static <T> T max(T[] it, Comparator<T> comparator) {
        return max(toStream(it), comparator);
    }

    public static <T> T max(Stream<T> it, Comparator<T> comparator) {
        return it.max(comparator).orElse(null);
    }
    // ====================max==========================


    // ====================min==========================
    public static <T extends Comparable> T min(Iterable<? extends T> it) {
        return min(it, Comparable::compareTo);
    }

    public static <T extends Comparable> T min(T[] it) {
        return min(it, Comparable::compareTo);
    }

    public static <T extends Comparable> T min(Stream<? extends T> it) {
        return min(it, Comparable::compareTo);
    }

    public static <T> T min(Iterable<T> it, Comparator<T> comparator) {
        return min(toStream(it), comparator);
    }

    public static <T> T min(T[] it, Comparator<T> comparator) {
        return min(toStream(it), comparator);
    }

    public static <T> T min(Stream<T> it, Comparator<T> comparator) {
        return it.min(comparator).orElse(null);
    }

    // ====================min==========================
}
