package com.github.azbh111.utils.java.model;

import java.io.Serializable;
import java.util.function.Function;

/**
 * 可以用来封装带有失败状态的返回值
 *
 * @author pyz
 * @date 2019/6/1 3:12 PM
 */
public class Res<T> implements Cloneable, Serializable {
    private static final long serialVersionUID = 1067562144473066103L;

    public static int defaultFailStatus = 400;
    public static int defaultSuccessStatus = 200;
    private T data;
    private int status = defaultSuccessStatus;
    private String errorMsg;

    public static <T> Res<T> empty() {
        return new Res<>();
    }

    public static <T> Res<T> success(T data) {
        return new Res<T>().setData(data);
    }

    public static <T> Res<T> fail(int status) {
        return fail(null, status, null, new Object[0]);
    }

    public static <T> Res<T> fail(String errorMsg, Object... os) {
        return fail(null, defaultFailStatus, errorMsg, os);
    }

    public static <T> Res<T> fail(int status, String errorMsg, Object... os) {
        return fail(null, status, errorMsg, os);
    }

    public static <T> Res<T> fail(T data, int status, String errorMsg, Object... os) {
        return new Res<T>().setData(data).setStatus(status).errorMsg(errorMsg, os);
    }

    /**
     * 擦除泛型
     *
     * @return
     */
    public Res eraseGeneric() {
        return this;
    }

    public Res<T> apply(Res<T> res) {
        this.status = res.status;
        this.data = res.data;
        this.errorMsg = res.errorMsg;
        return this;
    }

    /**
     * 链式操作,方便多层检查
     *
     * @param mapper
     * @return
     */
    public Res<T> mapIfNecessary(Function<Res<T>, Res<T>> mapper) {
        if (isFail()) {
            return this;
        }
        Res<T> res = mapper.apply(this);
        if (res == null) {
            return this;
        }
        this.apply(res);
        return this;
    }

    public Res<T> success() {
        status = 200;
        return this;
    }

    public Res<T> fail() {
        status = defaultFailStatus;
        return this;
    }

    public boolean isSuccess() {
        return status == defaultSuccessStatus;
    }

    public boolean isFail() {
        return status != defaultSuccessStatus;
    }

    public T getData() {
        return data;
    }

    public Res<T> setData(T data) {
        this.data = data;
        return this;
    }

    public Res<T> data(T data) {
        this.data = data;
        return this;
    }

    public int getStatus() {
        return status;
    }

    public Res<T> setStatus(int status) {
        this.status = status;
        return this;
    }

    public Res<T> status(int status) {
        this.status = status;
        return this;
    }

    public String getErrorMsg() {
        return errorMsg;
    }

    public Res<T> setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public Res<T> errorMsg(String errorMsg, Object... os) {
        this.errorMsg = String.format(errorMsg, os);
        return this;
    }

    @Override
    public Res<T> clone() {
        Res<T> res = Res.empty();
        res.status = status;
        res.data = data;
        res.errorMsg = errorMsg;
        return res;
    }

    @Override
    public String toString() {
        return "Res{" +
                "data=" + data +
                ", status=" + status +
                ", errorMsg='" + errorMsg + '\'' +
                '}';
    }
}
