package com.github.azbh111.utils.java.reflect;

import com.github.azbh111.utils.java.reflect.model.FieldFilter;
import com.github.azbh111.utils.java.reflect.model.MethodFilter;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

/**
 * @author pyz
 * @date 2019/5/3 5:24 PM
 */
public class ReflectUtils {

    public static Annotation findAnnotation(AnnotatedElement annotatedElement, Class<? extends Annotation> annotationType) {
        Set<AnnotatedElement> set = new HashSet<>();
        return findAnnotation(annotatedElement, annotationType, set);
    }

    private static Annotation findAnnotation(AnnotatedElement annotatedElement, Class<? extends Annotation> annotationType, Set<AnnotatedElement> container) {
        if (!container.add(annotatedElement)) {
            return null;
        }
        Annotation ann = annotatedElement.getAnnotation(annotationType);
        if (ann != null) {
            return ann;
        }
        for (Annotation annotation : annotatedElement.getAnnotations()) {
            ann = findAnnotation(annotation.getClass(), annotationType, container);
            if (ann != null) {
                return ann;
            }
        }
        if (annotatedElement instanceof Class) {
            Class type = (Class) annotatedElement;
            ann = findAnnotation(type.getSuperclass(), annotationType, container);
            if (ann != null) {
                return ann;
            }
            for (Class aClass : type.getInterfaces()) {
                ann = findAnnotation(aClass, annotationType, container);
                if (ann != null) {
                    return ann;
                }
            }
        }
        return null;
    }

    public static List<Field> getAllFields(Class clazz) {
        return getAllFields(clazz, FieldFilter.create());
    }

    public static Field getField(Class clazz, String name) throws NoSuchFieldException {
        return getField(clazz, FieldFilter.create().setName(name));
    }

    public static Field getField(Class clazz, FieldFilter filter) throws NoSuchFieldException {
        List<Field> list = getAllFields(clazz, filter);
        if (list == null || list.isEmpty()) {
            throw new NoSuchFieldException("can not find field in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        if (filter.getIndex() != null) {
            if (filter.getIndex() < list.size()) {
                return list.get(filter.getIndex());
            } else {
                throw new NoSuchFieldException("can not find field in " + clazz.getName() + " with filter:" + String.valueOf(filter));
            }
        }
        if (list.size() > 1) {
            throw new NoSuchFieldException("find " + list.size() + " fields in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        return list.get(0);
    }

    public static Field getField(Class clazz, Predicate<Field> filter) throws NoSuchFieldException {
        if (filter instanceof FieldFilter) {
            return getField(clazz, (FieldFilter) filter);
        }
        List<Field> list = getAllFields(clazz, filter);
        if (list == null || list.isEmpty()) {
            throw new NoSuchFieldException("can not find field in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        if (list.size() > 1) {
            throw new NoSuchFieldException("find " + list.size() + " fields in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        return list.get(0);
    }

    public static List<Field> getAllFields(Class clazz, Predicate<Field> filter) {
        List<Field> fs = new ArrayList<>();
        Class now = clazz;
        while (now != null && now != Object.class) {
            for (Field field : now.getDeclaredFields()) {
                if (!filter.test(field)) {
                    continue;
                }
                fs.add(field);
            }
            now = now.getSuperclass();
        }
        return fs;
    }

    public static List<Method> getAllMethods(Class clazz) {
        return getAllMethods(clazz, MethodFilter.create());
    }

    public static Method getMethod(Class clazz, String name) throws NoSuchMethodException {
        return getMethod(clazz, MethodFilter.create().setName(name));
    }

    public static Method getMethod(Class clazz, MethodFilter filter) throws NoSuchMethodException {
        List<Method> list = getAllMethods(clazz, filter);
        if (list == null || list.isEmpty()) {
            throw new NoSuchMethodException("can not find method in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        if (filter.getIndex() != null) {
            if (filter.getIndex() < list.size()) {
                return list.get(filter.getIndex());
            } else {
                throw new NoSuchMethodException("can not find method in " + clazz.getName() + " with filter:" + String.valueOf(filter));
            }
        }
        if (list.size() > 1) {
            throw new NoSuchMethodException("find " + list.size() + " methods in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        return list.get(0);
    }

    public static Method getMethod(Class clazz, Predicate<Method> filter) throws NoSuchMethodException {
        if (filter instanceof MethodFilter) {
            return getMethod(clazz, (MethodFilter) filter);
        }
        List<Method> list = getAllMethods(clazz, filter);
        if (list == null || list.isEmpty()) {
            throw new NoSuchMethodException("can not find method in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        if (list.size() > 1) {
            throw new NoSuchMethodException("find " + list.size() + " methods in " + clazz.getName() + " with filter:" + String.valueOf(filter));
        }
        return list.get(0);
    }

    public static List<Method> getAllMethods(Class clazz, Predicate<Method> filter) {
        List<Method> fs = new ArrayList<>();
        Class now = clazz;
        while (now != null && now != Object.class) {
            for (Method method : now.getDeclaredMethods()) {
                if (!filter.test(method)) {
                    continue;
                }
                fs.add(method);
            }
            now = now.getSuperclass();
        }
        return fs;
    }
}
