package com.github.azbh111.utils.java.io;

import com.github.azbh111.utils.java.lang.LazyField;
import com.github.azbh111.utils.java.promise.Promise;

import java.io.*;
import java.net.Socket;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * @author pyz
 * @date 2019/4/29 9:01 PM
 */
public class IOUtils {
    private static final int EOF = -1;
    private static final int DEFAULT_BUFFER_SIZE = 1024 * 4;
    private static final LazyField<ThreadPoolExecutor> executor = LazyField.from(() -> new ThreadPoolExecutor(1, 1024, 10, TimeUnit.SECONDS, new LinkedBlockingDeque<>()));

    /**
     * 从 input 复制数据到 output
     *
     * @param input
     * @param output
     * @param buffer 缓存
     * @return
     * @throws IOException
     */
    public static long copy(InputStream input, OutputStream output, byte[] buffer) throws IOException {
        long count = 0;
        int n = 0;
        while (EOF != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    /**
     * 从 input 复制数据到 output
     *
     * @param input
     * @param output
     * @return
     * @throws IOException
     */
    public static long copy(InputStream input, OutputStream output) throws IOException {
        return copy(input, output, new byte[DEFAULT_BUFFER_SIZE]);
    }

    /**
     * 从 input 复制数据到 output
     *
     * @param input
     * @param output
     * @param buffer 缓存
     * @return Promise<Long>
     * @throws IOException
     */
    public static Promise copyAsync(InputStream input, OutputStream output, byte[] buffer) throws IOException {
        Promise p = new Promise();
        executor.get().execute(() -> {
            try {
                long count = 0;
                int n = 0;
                while (EOF != (n = input.read(buffer))) {
                    output.write(buffer, 0, n);
                    count += n;
                }
                p.resolve(count);
            } catch (Throwable x) {
                p.reject(x);
            }
        });
        return p;
    }

    /**
     * 从 input 复制数据到 output
     *
     * @param input
     * @param output
     * @return Promise<Long>
     * @throws IOException
     */
    public static Promise copyAsync(InputStream input, OutputStream output) throws IOException {
        return copyAsync(input, output, new byte[DEFAULT_BUFFER_SIZE]);
    }

    public static byte[] readBytes(URL input) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readBytes(is);
        }
    }

    public static byte[] readBytes(File input) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readBytes(is);
        }
    }

    public static byte[] readBytes(Socket input) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readBytes(is);
        }
    }

    public static byte[] readBytes(InputStream input) throws IOException {
        ByteArrayOutputStream res = new ByteArrayOutputStream();
        byte[] cache = new byte[128];
        int size;
        while ((size = input.read(cache)) != -1) {
            res.write(cache, 0, size);
        }
        return res.toByteArray();
    }

    public static String readString(URL input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readString(new InputStreamReader(is, charset));
        }
    }

    public static String readString(Socket input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readString(new InputStreamReader(is, charset));
        }
    }

    public static String readString(File input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readString(new InputStreamReader(is, charset));
        }
    }

    public static String readString(InputStream input, Charset charset) throws IOException {
        return readString(new InputStreamReader(input, charset));
    }

    public static String readString(Reader reader) throws IOException {
        StringBuilder sb = new StringBuilder();
        char[] cache = new char[128];
        int size;
        while ((size = reader.read(cache)) != -1) {
            sb.append(cache, 0, size);
        }
        return sb.toString();
    }

    public static List<String> readLines(Socket input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readLines(new InputStreamReader(is, charset));
        }
    }

    public static List<String> readLines(URL input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readLines(new InputStreamReader(is, charset));
        }
    }

    public static List<String> readLines(File input, Charset charset) throws IOException {
        try (InputStream is = openInputStream(input)) {
            return readLines(new InputStreamReader(is, charset));
        }
    }


    public static List<String> readLines(InputStream input, Charset charset) throws IOException {
        return readLines(new InputStreamReader(input, charset));
    }

    public static List<String> readLines(Reader reader) throws IOException {
        BufferedReader br = toBufferedReader(reader);
        List<String> list = new ArrayList<>();
        String line = null;
        while ((line = br.readLine()) != null) {
            list.add(line);
        }
        return list;
    }

    public static void readLineLarge(Socket input, Charset charset, Consumer<String> consumer) throws IOException {
        try (InputStream is = openInputStream(input)) {
            readLineLarge(new InputStreamReader(is, charset), consumer);
        }
    }

    public static void readLineLarge(URL input, Charset charset, Consumer<String> consumer) throws IOException {
        try (InputStream is = openInputStream(input)) {
            readLineLarge(new InputStreamReader(is, charset), consumer);
        }
    }

    public static void readLineLarge(File input, Charset charset, Consumer<String> consumer) throws IOException {
        try (InputStream is = openInputStream(input)) {
            readLineLarge(new InputStreamReader(is, charset), consumer);
        }
    }

    public static void readLineLarge(InputStream input, Charset charset, Consumer<String> consumer) throws IOException {
        readLineLarge(new InputStreamReader(input, charset), consumer);
    }

    public static void readLineLarge(Reader reader, Consumer<String> consumer) throws IOException {
        BufferedReader br = toBufferedReader(reader);
        String line = null;
        while ((line = br.readLine()) != null) {
            consumer.accept(line);
        }
    }

    private static BufferedReader toBufferedReader(Reader reader) {
        BufferedReader br;
        if (reader == null || reader instanceof BufferedReader) {
            br = (BufferedReader) reader;
        } else {
            br = new BufferedReader(reader);
        }
        return br;
    }

    public static InputStream openInputStream(String filePath) throws IOException {
        return filePath == null ||filePath.isEmpty()? null : openInputStream(new File(filePath));
    }

    public static InputStream openInputStream(URL input) throws IOException {
        return input == null ? null : input.openStream();
    }

    public static InputStream openInputStream(File input) throws IOException {
        return input == null ? null : new FileInputStream(input);
    }

    public static InputStream openInputStream(Path input) throws IOException {
        return input == null ? null : Files.newInputStream(input);
    }

    public static InputStream openInputStream(Socket input) throws IOException {
        return input == null ? null : input.getInputStream();
    }

    public static OutputStream openOutputStream(String filePath) throws IOException {
        return filePath == null || filePath.isEmpty()? null : openOutputStream(new File(filePath));
    }

    public static OutputStream openOutputStream(URL output) throws IOException {
        return output == null ? null : output.openConnection().getOutputStream();
    }

    public static OutputStream openOutputStream(File output) throws IOException {
        return output == null ? null : openOutputStream(output.toPath());
    }

    public static OutputStream openOutputStream(Path output) throws IOException {
        if (output == null) {
            return null;
        }
        if (!Files.exists(output)) {
            Files.createFile(output);
        }
        return Files.newOutputStream(output);
    }

    public static OutputStream openOutputStream(Socket output) throws IOException {
        return output == null ? null : output.getOutputStream();
    }

}
