package com.github.azbh111.utils.java.lang;

import lombok.Getter;
import lombok.ToString;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author: zyp
 * @date: 2021/3/25 10:07
 */
@ToString
public class Stopwatch {
    @Getter
    private final String name;
    private long start;
    private final List<String> steps = new ArrayList<>();
    private final Map<String, long[]> timeMap = new HashMap<>();

    private Stopwatch(String name) {
        this.name = name;
        this.start = System.currentTimeMillis();
    }

    public static Stopwatch create(String name) {
        return new Stopwatch(name);
    }

    public static Stopwatch create() {
        return new Stopwatch("stopwatch");
    }

    public Stopwatch start(String... steps) {
        return startAndLog(noopStartLogger, steps);
    }

    public Stopwatch startAndLog(String... steps) {
        return startAndLog(soutStartLogger, steps);
    }

    public Stopwatch startAndLog(StartLogger logger, String... steps) {
        if (steps == null || steps.length == 0) {
            return this;
        }
        long now = System.currentTimeMillis();
        for (String step : steps) {
            this.steps.add(step);
            timeMap.put(step, new long[]{now, 0});
            logger.log(name, step, now);
        }
        return this;
    }

    public Stopwatch stop(String... steps) {
        return stopAndLog(noopStopLogger, steps);
    }

    public Stopwatch stopAndLog(String... steps) {
        return stopAndLog(soutStopLogger, steps);
    }

    public Stopwatch stopAndLog(StopLogger logger, String... steps) {
        if (steps == null || steps.length == 0) {
            return this;
        }
        long now = System.currentTimeMillis();
        for (String step : steps) {
            long[] pair = timeMap.computeIfAbsent(step, k -> new long[]{start, 0});
            pair[1] = now;
            logger.log(name, step, now, now - pair[0]);
        }
        return this;
    }

    public void reset() {
        this.start = System.currentTimeMillis();
        this.steps.clear();
        this.timeMap.clear();
    }

    public long getTime() {
        return System.currentTimeMillis() - start;
    }

    private static final StopLogger soutStopLogger = (name, step, now, cost) -> System.out.println(String.format("stopwatch: %s stop at %s. cost %s ms.", step, now, cost));
    private static final StopLogger noopStopLogger = (name, step, now, cost) -> {
    };
    private static final StartLogger soutStartLogger = (name, step, now) -> System.out.println(String.format("stopwatch: %s start at %s.", step, now));
    private static final StartLogger noopStartLogger = (name, step, now) -> {
    };


    public static interface StartLogger {
        public void log(String name, String step, long now);
    }

    public static interface StopLogger {
        public void log(String name, String step, long now, long cost);
    }
}
