package com.github.azbh111.utils.java.rsa;

import com.github.azbh111.utils.java.annotation.Nonnull;
import com.github.azbh111.utils.java.annotation.Nullable;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * 封装rsa 加密/解密/签名/验证签名 等操作
 *
 * @author: zyp
 * @date: 2020/11/13 16:06
 */
public class RSAUtils {
    static final String KEY_ALGORITHM = "RSA";

    /**
     * 为null时无法进行签名
     *
     * @param signMethod
     * @return
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, int keySize) throws NoSuchAlgorithmException {
        return RSAPair.create(signMethod, keySize);
    }

    /**
     * 为null时无法进行签名
     *
     * @param signMethod
     * @param publicKeyHex
     * @param privateKeyHex
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, @Nonnull String publicKeyHex, @Nonnull String privateKeyHex) throws InvalidKeySpecException, NoSuchAlgorithmException {
        return RSAPair.create(signMethod, publicKeyHex, privateKeyHex);
    }

    /**
     * @param signMethod 为null时，无法进行签名
     * @param publicKey
     * @param privateKey
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, @Nonnull PublicKey publicKey, @Nonnull PrivateKey privateKey) {
        return RSAPair.create(signMethod, publicKey, privateKey);
    }

    /**
     * 创建私钥
     *
     * @param privateKeyBytes
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     */
    public static PrivateKey createPrivateKey(@Nonnull byte[] privateKeyBytes) throws NoSuchAlgorithmException, InvalidKeySpecException {
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        return keyFactory.generatePrivate(keySpec);
    }

    /**
     * 创建公钥
     *
     * @param publicKeyBytes
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     */
    public static PublicKey createPublicKey(@Nonnull byte[] publicKeyBytes) throws NoSuchAlgorithmException, InvalidKeySpecException {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * 公钥加密
     *
     * @param publicKeyBytes 公钥
     * @param data           明文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] publicEncrypt(@Nonnull byte[] publicKeyBytes, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException {
        return encrypt(createPublicKey(publicKeyBytes), data);
    }

    /**
     * 私钥加密
     *
     * @param privatKeyBytes 私钥
     * @param data           明文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] privateEncrypt(@Nonnull byte[] privatKeyBytes, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException {
        return encrypt(createPrivateKey(privatKeyBytes), data);
    }

    /**
     * 公钥解密
     *
     * @param publicKeyBytes 公钥
     * @param data           密文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] publicDencrypt(@Nonnull byte[] publicKeyBytes, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException {
        return dencrypt(createPublicKey(publicKeyBytes), data);
    }


    /**
     * 私钥解密
     *
     * @param privatKeyBytes 私钥
     * @param data           密文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] privateDencrypt(@Nonnull byte[] privatKeyBytes, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException {
        return dencrypt(createPrivateKey(privatKeyBytes), data);
    }

    /**
     * 加密
     *
     * @param key  密钥
     * @param data 密文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] encrypt(@Nonnull Key key, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return doFinal(Cipher.ENCRYPT_MODE, key, data);
    }

    /**
     * 1. 使用公钥对私钥加密的数据进行解密
     * 2. 使用私钥对公钥加密的数据进行解密
     *
     * @param key  密钥
     * @param data 密文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidKeySpecException
     */
    public static byte[] dencrypt(@Nonnull Key key, @Nonnull byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return doFinal(Cipher.DECRYPT_MODE, key, data);
    }

    private static byte[] doFinal(int cipherMode, Key key, byte[] data) throws NoSuchPaddingException, NoSuchAlgorithmException, BadPaddingException, IllegalBlockSizeException, InvalidKeyException {
        Cipher cipher = Cipher.getInstance("RSA");
        cipher.init(cipherMode, key);
        return cipher.doFinal(data);
    }

    /**
     * 私钥签名
     *
     * @param signMethod      签名方法
     * @param privateKeyBytes 私钥
     * @param data            明文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws SignatureException
     * @throws InvalidKeySpecException
     */
    public static byte[] sign(@Nonnull RSASignMethodEnum signMethod, @Nonnull byte[] privateKeyBytes, @Nonnull byte[] data) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException, InvalidKeySpecException {
        return sign(signMethod, createPrivateKey(privateKeyBytes), data);
    }

    /**
     * 私钥签名
     *
     * @param signMethod      签名方法
     * @param privateKeyBytes 私钥
     * @param data            明文
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws SignatureException
     * @throws InvalidKeySpecException
     */
    public static byte[] sign(@Nonnull RSASignMethodEnum signMethod, @Nonnull PrivateKey privateKey, @Nonnull byte[] data) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        Signature signature = Signature.getInstance(signMethod.name());
        signature.initSign(privateKey);
        signature.update(data);
        return signature.sign();
    }

    /**
     * 公钥验证签名
     *
     * @param signMethod     签名方法
     * @param publicKeyBytes 公钥
     * @param data           明文
     * @param sign           签名
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws SignatureException
     * @throws InvalidKeySpecException
     */
    public static boolean verifySign(@Nonnull RSASignMethodEnum signMethod, @Nonnull byte[] publicKeyBytes, @Nonnull byte[] data, @Nonnull byte[] sign) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException, InvalidKeySpecException {
        return verifySign(signMethod, createPublicKey(publicKeyBytes), data, sign);
    }

    /**
     * 公钥验证签名
     *
     * @param signMethod     签名方法
     * @param publicKeyBytes 公钥
     * @param data           明文
     * @param sign           签名
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws SignatureException
     * @throws InvalidKeySpecException
     */
    public static boolean verifySign(@Nonnull RSASignMethodEnum signMethod, @Nonnull PublicKey publicKey, @Nonnull byte[] data, @Nonnull byte[] sign) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        Signature signature = Signature.getInstance(signMethod.name());
        signature.initVerify(publicKey);
        signature.update(data);
        return signature.verify(sign);
    }

}
