package cn.binarywang.wx.miniapp.api;

import java.io.File;

import cn.binarywang.wx.miniapp.bean.WxMaJscode2SessionResult;
import cn.binarywang.wx.miniapp.config.WxMaConfig;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.common.util.http.MediaUploadRequestExecutor;
import me.chanjar.weixin.common.util.http.RequestExecutor;
import me.chanjar.weixin.common.util.http.RequestHttp;

/**
 * @author <a href="https://github.com/binarywang">Binary Wang</a>
 */
public interface WxMaService {
  /**
   * 获取access_token.
   */
  String GET_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s";

  String JSCODE_TO_SESSION_URL = "https://api.weixin.qq.com/sns/jscode2session";

  String IMG_SEC_CHECK_URL = "https://api.weixin.qq.com/wxa/img_sec_check";

  /**
   * <pre>
   * 校验一张图片是否含有违法违规内容.
   * 应用场景举例：1）图片智能鉴黄：涉及拍照的工具类应用(如美拍，识图类应用)用户拍照上传检测；电商类商品上架图片检测；媒体类用户文章里的图片检测等；2）敏感人脸识别：用户头像；媒体类用户文章里的图片检测；社交类用户上传的图片检测等。频率限制：单个 appId 调用上限为 1000 次/分钟，100,000 次/天
   * 详情请见: https://developers.weixin.qq.com/miniprogram/dev/api/imgSecCheck.html
   * </pre>
   */
  boolean imgSecCheck(File file) throws WxErrorException;

  /**
   * 获取登录后的session信息.
   *
   * @param jsCode 登录时获取的 code
   */
  WxMaJscode2SessionResult jsCode2SessionInfo(String jsCode) throws WxErrorException;

  /**
   * <pre>
   * 验证消息的确来自微信服务器.
   * 详情请见: http://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421135319&token=&lang=zh_CN
   * </pre>
   */
  boolean checkSignature(String timestamp, String nonce, String signature);

  /**
   * 获取access_token, 不强制刷新access_token.
   *
   * @see #getAccessToken(boolean)
   */
  String getAccessToken() throws WxErrorException;

  /**
   * <pre>
   * 获取access_token，本方法线程安全.
   * 且在多线程同时刷新时只刷新一次，避免超出2000次/日的调用次数上限
   *
   * 另：本service的所有方法都会在access_token过期是调用此方法
   *
   * 程序员在非必要情况下尽量不要主动调用此方法
   *
   * 详情请见: http://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140183&token=&lang=zh_CN
   * </pre>
   *
   * @param forceRefresh 强制刷新
   */
  String getAccessToken(boolean forceRefresh) throws WxErrorException;

  /**
   * 当本Service没有实现某个API的时候，可以用这个，针对所有微信API中的GET请求.
   */
  String get(String url, String queryParam) throws WxErrorException;

  /**
   * 当本Service没有实现某个API的时候，可以用这个，针对所有微信API中的POST请求.
   */
  String post(String url, String postData) throws WxErrorException;

  /**
   * <pre>
   * Service没有实现某个API的时候，可以用这个，
   * 比{@link #get}和{@link #post}方法更灵活，可以自己构造RequestExecutor用来处理不同的参数和不同的返回类型。
   * 可以参考，{@link MediaUploadRequestExecutor}的实现方法
   * </pre>
   */
  <T, E> T execute(RequestExecutor<T, E> executor, String uri, E data) throws WxErrorException;

  /**
   * <pre>
   * 设置当微信系统响应系统繁忙时，要等待多少 retrySleepMillis(ms) * 2^(重试次数 - 1) 再发起重试.
   * 默认：1000ms
   * </pre>
   */
  void setRetrySleepMillis(int retrySleepMillis);

  /**
   * <pre>
   * 设置当微信系统响应系统繁忙时，最大重试次数.
   * 默认：5次
   * </pre>
   */
  void setMaxRetryTimes(int maxRetryTimes);

  /**
   * 获取WxMaConfig 对象.
   *
   * @return WxMaConfig
   */
  WxMaConfig getWxMaConfig();

  /**
   * 注入 {@link WxMaConfig} 的实现.
   */
  void setWxMaConfig(WxMaConfig wxConfigProvider);

  /**
   * 返回消息（客服消息和模版消息）发送接口方法实现类，以方便调用其各个接口.
   *
   * @return WxMaMsgService
   */
  WxMaMsgService getMsgService();

  /**
   * 返回素材相关接口方法的实现类对象，以方便调用其各个接口.
   *
   * @return WxMaMediaService
   */
  WxMaMediaService getMediaService();

  /**
   * 返回用户相关接口方法的实现类对象，以方便调用其各个接口.
   *
   * @return WxMaUserService
   */
  WxMaUserService getUserService();

  /**
   * 返回二维码相关接口方法的实现类对象，以方便调用其各个接口.
   *
   * @return WxMaQrcodeService
   */
  WxMaQrcodeService getQrcodeService();

  /**
   * 返回模板配置相关接口方法的实现类对象, 以方便调用其各个接口.
   *
   * @return WxMaTemplateService
   */
  WxMaTemplateService getTemplateService();

  /**
   * 数据分析相关查询服务.
   *
   * @return WxMaAnalysisService
   */
  WxMaAnalysisService getAnalysisService();

  /**
   * 返回代码操作相关的 API.
   *
   * @return WxMaCodeService
   */
  WxMaCodeService getCodeService();

  /**
   * 返回jsapi操作相关的 API服务类对象.
   *
   * @return WxMaJsapiService
   */
  WxMaJsapiService getJsapiService();

  /**
   * 小程序修改服务器地址、成员管理 API.
   *
   * @return WxMaSettingService
   */
  WxMaSettingService getSettingService();

  /**
   * 返回分享相关查询服务.
   * @return WxMaShareService
   */
  WxMaShareService getShareService();

  /**
   * 返回维新运动相关接口服务对象.
   * @return WxMaShareService
   */
  WxMaRunService getRunService();

  /**
   * 初始化http请求对象.
   */
  void initHttp();

  /**
   * 请求http请求相关信息.
   */
  RequestHttp getRequestHttp();


}
