/*
 * jDialects, a tiny SQL dialect tool
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later. See
 * the lgpl.txt file in the root directory or
 * <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package com.github.drinkjava2.jdialects;

import java.util.HashMap;
import java.util.Map;

import com.github.drinkjava2.jdialects.utils.StrUtils;

/**
 * Collect reserved words of all databases, keywords not included
 * 
 * @author Yong Zhu 
 * @since 1.0.1
 */
@SuppressWarnings("all")
public class ReservedDBWords {

	private static final Map<String, String> RESERVED_WORDS = new HashMap<String, String>();

	private ReservedDBWords() {
		// hide default constructor
	}

	private static void addWord(String word, String databaseName) {
		String value = RESERVED_WORDS.get(word.toUpperCase());
		if (value == null || value.length() == 0)
			RESERVED_WORDS.put(word.toUpperCase(), databaseName);
		else {
			if (value.indexOf(databaseName) < 0)
				RESERVED_WORDS.put(word.toUpperCase(), value + "/" + databaseName);
		}
	}

	/**
	 * Check if is a dialect reserved word of ANSI-SQL reserved word
	 * 
	 * @return false:not reserved word. true:reserved by dialect or ANSI-SQL
	 */
	public static boolean isReservedWord(Dialect dialect, String word) {
		if (!isReservedWord(word))
			return false;
		String fitDatabases = RESERVED_WORDS.get(word.toUpperCase()).toUpperCase();
		if (fitDatabases.contains("ANSI"))
			return true;
		String dia = dialect.toString().replace("Dialect", "").toUpperCase();
		if (dia.length() >= 4)
			dia = dia.substring(0, 4);// only compare first 4 letters
		return (fitDatabases.contains(dia));
	}

	/**
	 * Check if is a reserved word of any database
	 */
	public static boolean isReservedWord(String word) {
		return !StrUtils.isEmpty(word) && RESERVED_WORDS.containsKey(word.toUpperCase());
	}

	/**
	 * Return database name of given reserved word
	 */
	public static String reservedForDB(String word) {
		return RESERVED_WORDS.get(word.toUpperCase());
	}

	/**
	 * Below Collect reserved words of most database families
	 */
	static {
		// =========================ANSI SQL===================================
		// From http://developer.mimer.se/validator/sql-reserved-words.tml
		addWord("ABSOLUTE", "ANSI-SQL");
		addWord("ACTION", "ANSI-SQL");
		addWord("ADD", "ANSI-SQL");
		addWord("AFTER", "ANSI-SQL");
		addWord("ALL", "ANSI-SQL");
		addWord("ALLOCATE", "ANSI-SQL");
		addWord("ALTER", "ANSI-SQL");
		addWord("AND", "ANSI-SQL");
		addWord("ANY", "ANSI-SQL");
		addWord("ARE", "ANSI-SQL");
		addWord("ARRAY", "ANSI-SQL");
		addWord("AS", "ANSI-SQL");
		addWord("ASC", "ANSI-SQL");
		addWord("ASENSITIVE", "ANSI-SQL");
		addWord("ASSERTION", "ANSI-SQL");
		addWord("ASYMMETRIC", "ANSI-SQL");
		addWord("AT", "ANSI-SQL");
		addWord("ATOMIC", "ANSI-SQL");
		addWord("AUTHORIZATION", "ANSI-SQL");
		addWord("AVG", "ANSI-SQL");
		addWord("BEFORE", "ANSI-SQL");
		addWord("BEGIN", "ANSI-SQL");
		addWord("BETWEEN", "ANSI-SQL");
		addWord("BIGINT", "ANSI-SQL");
		addWord("BINARY", "ANSI-SQL");
		addWord("BIT", "ANSI-SQL");
		addWord("BIT_LENGTH", "ANSI-SQL");
		addWord("BLOB", "ANSI-SQL");
		addWord("BOOLEAN", "ANSI-SQL");
		addWord("BOTH", "ANSI-SQL");
		addWord("BREADTH", "ANSI-SQL");
		addWord("BY", "ANSI-SQL");
		addWord("CALL", "ANSI-SQL");
		addWord("CALLED", "ANSI-SQL");
		addWord("CASCADE", "ANSI-SQL");
		addWord("CASCADED", "ANSI-SQL");
		addWord("CASE", "ANSI-SQL");
		addWord("CAST", "ANSI-SQL");
		addWord("CATALOG", "ANSI-SQL");
		addWord("CHAR", "ANSI-SQL");
		addWord("CHAR_LENGTH", "ANSI-SQL");
		addWord("CHARACTER", "ANSI-SQL");
		addWord("CHARACTER_LENGTH", "ANSI-SQL");
		addWord("CHECK", "ANSI-SQL");
		addWord("CLOB", "ANSI-SQL");
		addWord("CLOSE", "ANSI-SQL");
		addWord("COALESCE", "ANSI-SQL");
		addWord("COLLATE", "ANSI-SQL");
		addWord("COLLATION", "ANSI-SQL");
		addWord("COLUMN", "ANSI-SQL");
		addWord("COMMIT", "ANSI-SQL");
		addWord("CONDITION", "ANSI-SQL");
		addWord("CONNECT", "ANSI-SQL");
		addWord("CONNECTION", "ANSI-SQL");
		addWord("CONSTRAINT", "ANSI-SQL");
		addWord("CONSTRAINTS", "ANSI-SQL");
		addWord("CONSTRUCTOR", "ANSI-SQL");
		addWord("CONTAINS", "ANSI-SQL");
		addWord("CONTINUE", "ANSI-SQL");
		addWord("CONVERT", "ANSI-SQL");
		addWord("CORRESPONDING", "ANSI-SQL");
		addWord("COUNT", "ANSI-SQL");
		addWord("CREATE", "ANSI-SQL");
		addWord("CROSS", "ANSI-SQL");
		addWord("CUBE", "ANSI-SQL");
		addWord("CURRENT", "ANSI-SQL");
		addWord("CURRENT_DATE", "ANSI-SQL");
		addWord("CURRENT_DEFAULT_TRANSFORM_GROUP", "ANSI-SQL");
		addWord("CURRENT_PATH", "ANSI-SQL");
		addWord("CURRENT_ROLE", "ANSI-SQL");
		addWord("CURRENT_TIME", "ANSI-SQL");
		addWord("CURRENT_TIMESTAMP", "ANSI-SQL");
		addWord("CURRENT_TRANSFORM_GROUP_FOR_TYPE", "ANSI-SQL");
		addWord("CURRENT_USER", "ANSI-SQL");
		addWord("CURSOR", "ANSI-SQL");
		addWord("CYCLE", "ANSI-SQL");
		addWord("DATA", "ANSI-SQL");
		addWord("DATE", "ANSI-SQL");
		addWord("DAY", "ANSI-SQL");
		addWord("DEALLOCATE", "ANSI-SQL");
		addWord("DEC", "ANSI-SQL");
		addWord("DECIMAL", "ANSI-SQL");
		addWord("DECLARE", "ANSI-SQL");
		addWord("DEFAULT", "ANSI-SQL");
		addWord("DEFERRABLE", "ANSI-SQL");
		addWord("DEFERRED", "ANSI-SQL");
		addWord("DELETE", "ANSI-SQL");
		addWord("DEPTH", "ANSI-SQL");
		addWord("DEREF", "ANSI-SQL");
		addWord("DESC", "ANSI-SQL");
		addWord("DESCRIBE", "ANSI-SQL");
		addWord("DESCRIPTOR", "ANSI-SQL");
		addWord("DETERMINISTIC", "ANSI-SQL");
		addWord("DIAGNOSTICS", "ANSI-SQL");
		addWord("DISCONNECT", "ANSI-SQL");
		addWord("DISTINCT", "ANSI-SQL");
		addWord("DO", "ANSI-SQL");
		addWord("DOMAIN", "ANSI-SQL");
		addWord("DOUBLE", "ANSI-SQL");
		addWord("DROP", "ANSI-SQL");
		addWord("DYNAMIC", "ANSI-SQL");
		addWord("EACH", "ANSI-SQL");
		addWord("ELEMENT", "ANSI-SQL");
		addWord("ELSE", "ANSI-SQL");
		addWord("ELSEIF", "ANSI-SQL");
		addWord("END", "ANSI-SQL");
		addWord("EQUALS", "ANSI-SQL");
		addWord("ESCAPE", "ANSI-SQL");
		addWord("EXCEPT", "ANSI-SQL");
		addWord("EXCEPTION", "ANSI-SQL");
		addWord("EXEC", "ANSI-SQL");
		addWord("EXECUTE", "ANSI-SQL");
		addWord("EXISTS", "ANSI-SQL");
		addWord("EXIT", "ANSI-SQL");
		addWord("EXTERNAL", "ANSI-SQL");
		addWord("EXTRACT", "ANSI-SQL");
		addWord("FETCH", "ANSI-SQL");
		addWord("FILTER", "ANSI-SQL");
		addWord("FIRST", "ANSI-SQL");
		addWord("FLOAT", "ANSI-SQL");
		addWord("FOR", "ANSI-SQL");
		addWord("FOREIGN", "ANSI-SQL");
		addWord("FOUND", "ANSI-SQL");
		addWord("FREE", "ANSI-SQL");
		addWord("FROM", "ANSI-SQL");
		addWord("FULL", "ANSI-SQL");
		addWord("FUNCTION", "ANSI-SQL");
		addWord("GENERAL", "ANSI-SQL");
		addWord("GET", "ANSI-SQL");
		addWord("GLOBAL", "ANSI-SQL");
		addWord("GO", "ANSI-SQL");
		addWord("GOTO", "ANSI-SQL");
		addWord("GRANT", "ANSI-SQL");
		addWord("GROUP", "ANSI-SQL");
		addWord("GROUPING", "ANSI-SQL");
		addWord("HANDLER", "ANSI-SQL");
		addWord("HAVING", "ANSI-SQL");
		addWord("HOLD", "ANSI-SQL");
		addWord("HOUR", "ANSI-SQL");
		addWord("IDENTITY", "ANSI-SQL");
		addWord("IF", "ANSI-SQL");
		addWord("IMMEDIATE", "ANSI-SQL");
		addWord("IN", "ANSI-SQL");
		addWord("INDICATOR", "ANSI-SQL");
		addWord("INITIALLY", "ANSI-SQL");
		addWord("INNER", "ANSI-SQL");
		addWord("INOUT", "ANSI-SQL");
		addWord("INPUT", "ANSI-SQL");
		addWord("INSENSITIVE", "ANSI-SQL");
		addWord("INSERT", "ANSI-SQL");
		addWord("INT", "ANSI-SQL");
		addWord("INTEGER", "ANSI-SQL");
		addWord("INTERSECT", "ANSI-SQL");
		addWord("INTERVAL", "ANSI-SQL");
		addWord("INTO", "ANSI-SQL");
		addWord("IS", "ANSI-SQL");
		addWord("ISOLATION", "ANSI-SQL");
		addWord("ITERATE", "ANSI-SQL");
		addWord("JOIN", "ANSI-SQL");
		addWord("KEY", "ANSI-SQL");
		addWord("LANGUAGE", "ANSI-SQL");
		addWord("LARGE", "ANSI-SQL");
		addWord("LAST", "ANSI-SQL");
		addWord("LATERAL", "ANSI-SQL");
		addWord("LEADING", "ANSI-SQL");
		addWord("LEAVE", "ANSI-SQL");
		addWord("LEFT", "ANSI-SQL");
		addWord("LEVEL", "ANSI-SQL");
		addWord("LIKE", "ANSI-SQL");
		addWord("LOCAL", "ANSI-SQL");
		addWord("LOCALTIME", "ANSI-SQL");
		addWord("LOCALTIMESTAMP", "ANSI-SQL");
		addWord("LOCATOR", "ANSI-SQL");
		addWord("LOOP", "ANSI-SQL");
		addWord("LOWER", "ANSI-SQL");
		addWord("MAP", "ANSI-SQL");
		addWord("MATCH", "ANSI-SQL");
		addWord("MAX", "ANSI-SQL");
		addWord("MEMBER", "ANSI-SQL");
		addWord("MERGE", "ANSI-SQL");
		addWord("METHOD", "ANSI-SQL");
		addWord("MIN", "ANSI-SQL");
		addWord("MINUTE", "ANSI-SQL");
		addWord("MODIFIES", "ANSI-SQL");
		addWord("MODULE", "ANSI-SQL");
		addWord("MONTH", "ANSI-SQL");
		addWord("MULTISET", "ANSI-SQL");
		addWord("NAMES", "ANSI-SQL");
		addWord("NATIONAL", "ANSI-SQL");
		addWord("NATURAL", "ANSI-SQL");
		addWord("NCHAR", "ANSI-SQL");
		addWord("NCLOB", "ANSI-SQL");
		addWord("NEW", "ANSI-SQL");
		addWord("NEXT", "ANSI-SQL");
		addWord("NO", "ANSI-SQL");
		addWord("NONE", "ANSI-SQL");
		addWord("NOT", "ANSI-SQL");
		addWord("NULL", "ANSI-SQL");
		addWord("NULLIF", "ANSI-SQL");
		addWord("NUMERIC", "ANSI-SQL");
		addWord("OBJECT", "ANSI-SQL");
		addWord("OCTET_LENGTH", "ANSI-SQL");
		addWord("OF", "ANSI-SQL");
		addWord("OLD", "ANSI-SQL");
		addWord("ON", "ANSI-SQL");
		addWord("ONLY", "ANSI-SQL");
		addWord("OPEN", "ANSI-SQL");
		addWord("OPTION", "ANSI-SQL");
		addWord("OR", "ANSI-SQL");
		addWord("ORDER", "ANSI-SQL");
		addWord("ORDINALITY", "ANSI-SQL");
		addWord("OUT", "ANSI-SQL");
		addWord("OUTER", "ANSI-SQL");
		addWord("OUTPUT", "ANSI-SQL");
		addWord("OVER", "ANSI-SQL");
		addWord("OVERLAPS", "ANSI-SQL");
		addWord("PAD", "ANSI-SQL");
		addWord("PARAMETER", "ANSI-SQL");
		addWord("PARTIAL", "ANSI-SQL");
		addWord("PARTITION", "ANSI-SQL");
		addWord("PATH", "ANSI-SQL");
		addWord("POSITION", "ANSI-SQL");
		addWord("PRECISION", "ANSI-SQL");
		addWord("PREPARE", "ANSI-SQL");
		addWord("PRESERVE", "ANSI-SQL");
		addWord("PRIMARY", "ANSI-SQL");
		addWord("PRIOR", "ANSI-SQL");
		addWord("PRIVILEGES", "ANSI-SQL");
		addWord("PROCEDURE", "ANSI-SQL");
		addWord("PUBLIC", "ANSI-SQL");
		addWord("RANGE", "ANSI-SQL");
		addWord("READ", "ANSI-SQL");
		addWord("READS", "ANSI-SQL");
		addWord("REAL", "ANSI-SQL");
		addWord("RECURSIVE", "ANSI-SQL");
		addWord("REF", "ANSI-SQL");
		addWord("REFERENCES", "ANSI-SQL");
		addWord("REFERENCING", "ANSI-SQL");
		addWord("RELATIVE", "ANSI-SQL");
		addWord("RELEASE", "ANSI-SQL");
		addWord("REPEAT", "ANSI-SQL");
		addWord("RESIGNAL", "ANSI-SQL");
		addWord("RESTRICT", "ANSI-SQL");
		addWord("RESULT", "ANSI-SQL");
		addWord("RETURN", "ANSI-SQL");
		addWord("RETURNS", "ANSI-SQL");
		addWord("REVOKE", "ANSI-SQL");
		addWord("RIGHT", "ANSI-SQL");
		addWord("ROLE", "ANSI-SQL");
		addWord("ROLLBACK", "ANSI-SQL");
		addWord("ROLLUP", "ANSI-SQL");
		addWord("ROUTINE", "ANSI-SQL");
		addWord("ROW", "ANSI-SQL");
		addWord("ROWS", "ANSI-SQL");
		addWord("SAVEPOINT", "ANSI-SQL");
		addWord("SCHEMA", "ANSI-SQL");
		addWord("SCOPE", "ANSI-SQL");
		addWord("SCROLL", "ANSI-SQL");
		addWord("SEARCH", "ANSI-SQL");
		addWord("SECOND", "ANSI-SQL");
		addWord("SECTION", "ANSI-SQL");
		addWord("SELECT", "ANSI-SQL");
		addWord("SENSITIVE", "ANSI-SQL");
		addWord("SESSION", "ANSI-SQL");
		addWord("SESSION_USER", "ANSI-SQL");
		addWord("SET", "ANSI-SQL");
		addWord("SETS", "ANSI-SQL");
		addWord("SIGNAL", "ANSI-SQL");
		addWord("SIMILAR", "ANSI-SQL");
		addWord("SIZE", "ANSI-SQL");
		addWord("SMALLINT", "ANSI-SQL");
		addWord("SOME", "ANSI-SQL");
		addWord("SPACE", "ANSI-SQL");
		addWord("SPECIFIC", "ANSI-SQL");
		addWord("SPECIFICTYPE", "ANSI-SQL");
		addWord("SQL", "ANSI-SQL");
		addWord("SQLCODE", "ANSI-SQL");
		addWord("SQLERROR", "ANSI-SQL");
		addWord("SQLEXCEPTION", "ANSI-SQL");
		addWord("SQLSTATE", "ANSI-SQL");
		addWord("SQLWARNING", "ANSI-SQL");
		addWord("START", "ANSI-SQL");
		addWord("STATE", "ANSI-SQL");
		addWord("STATIC", "ANSI-SQL");
		addWord("SUBMULTISET", "ANSI-SQL");
		addWord("SUBSTRING", "ANSI-SQL");
		addWord("SUM", "ANSI-SQL");
		addWord("SYMMETRIC", "ANSI-SQL");
		addWord("SYSTEM", "ANSI-SQL");
		addWord("SYSTEM_USER", "ANSI-SQL");
		addWord("TABLE", "ANSI-SQL");
		addWord("TABLESAMPLE", "ANSI-SQL");
		addWord("TEMPORARY", "ANSI-SQL");
		addWord("THEN", "ANSI-SQL");
		addWord("TIME", "ANSI-SQL");
		addWord("TIMESTAMP", "ANSI-SQL");
		addWord("TIMEZONE_HOUR", "ANSI-SQL");
		addWord("TIMEZONE_MINUTE", "ANSI-SQL");
		addWord("TO", "ANSI-SQL");
		addWord("TRAILING", "ANSI-SQL");
		addWord("TRANSACTION", "ANSI-SQL");
		addWord("TRANSLATE", "ANSI-SQL");
		addWord("TRANSLATION", "ANSI-SQL");
		addWord("TREAT", "ANSI-SQL");
		addWord("TRIGGER", "ANSI-SQL");
		addWord("TRIM", "ANSI-SQL");
		addWord("UNDER", "ANSI-SQL");
		addWord("UNDO", "ANSI-SQL");
		addWord("UNION", "ANSI-SQL");
		addWord("UNIQUE", "ANSI-SQL");
		addWord("UNKNOWN", "ANSI-SQL");
		addWord("UNNEST", "ANSI-SQL");
		addWord("UNTIL", "ANSI-SQL");
		addWord("UPDATE", "ANSI-SQL");
		addWord("UPPER", "ANSI-SQL");
		addWord("USAGE", "ANSI-SQL");
		addWord("USER", "ANSI-SQL");
		addWord("USING", "ANSI-SQL");
		addWord("VALUE", "ANSI-SQL");
		addWord("VALUES", "ANSI-SQL");
		addWord("VARCHAR", "ANSI-SQL");
		addWord("VARYING", "ANSI-SQL");
		addWord("VIEW", "ANSI-SQL");
		addWord("WHEN", "ANSI-SQL");
		addWord("WHENEVER", "ANSI-SQL");
		addWord("WHERE", "ANSI-SQL");
		addWord("WHILE", "ANSI-SQL");
		addWord("WINDOW", "ANSI-SQL");
		addWord("WITH", "ANSI-SQL");
		addWord("WITHIN", "ANSI-SQL");
		addWord("WITHOUT", "ANSI-SQL");
		addWord("WORK", "ANSI-SQL");
		addWord("WRITE", "ANSI-SQL");
		addWord("YEAR", "ANSI-SQL");
		addWord("ZONE", "ANSI-SQL");
		addWord("FALSE", "ANSI-SQL");
		addWord("TRUE", "ANSI-SQL");
	}

	static {
		// =========================MySQL===================================
		// From https://dev.mysql.com/doc/refman/5.7/en/keywords.html
		addWord("ACCESSIBLE", "MySQL");
		addWord("ADD", "MySQL");
		addWord("ALL", "MySQL");
		addWord("ALTER", "MySQL");
		addWord("ANALYZE", "MySQL");
		addWord("AND", "MySQL");
		addWord("AS", "MySQL");
		addWord("ASC", "MySQL");
		addWord("ASENSITIVE", "MySQL");
		addWord("BEFORE", "MySQL");
		addWord("BETWEEN", "MySQL");
		addWord("BIGINT", "MySQL");
		addWord("BINARY", "MySQL");
		addWord("BLOB", "MySQL");
		addWord("BOTH", "MySQL");
		addWord("BY", "MySQL");
		addWord("CALL", "MySQL");
		addWord("CASCADE", "MySQL");
		addWord("CASE", "MySQL");
		addWord("CHANGE", "MySQL");
		addWord("CHAR", "MySQL");
		addWord("CHARACTER", "MySQL");
		addWord("CHECK", "MySQL");
		addWord("COLLATE", "MySQL");
		addWord("COLUMN", "MySQL");
		addWord("CONDITION", "MySQL");
		addWord("CONSTRAINT", "MySQL");
		addWord("CONTINUE", "MySQL");
		addWord("CONVERT", "MySQL");
		addWord("CREATE", "MySQL");
		addWord("CROSS", "MySQL");
		addWord("CURRENT_DATE", "MySQL");
		addWord("CURRENT_TIME", "MySQL");
		addWord("CURRENT_TIMESTAMP", "MySQL");
		addWord("CURRENT_USER", "MySQL");
		addWord("CURSOR", "MySQL");
		addWord("DATABASE", "MySQL");
		addWord("DATABASES", "MySQL");
		addWord("DAY_HOUR", "MySQL");
		addWord("DAY_MICROSECOND", "MySQL");
		addWord("DAY_MINUTE", "MySQL");
		addWord("DAY_SECOND", "MySQL");
		addWord("DEC", "MySQL");
		addWord("DECIMAL", "MySQL");
		addWord("DECLARE", "MySQL");
		addWord("DEFAULT", "MySQL");
		addWord("DELAYED", "MySQL");
		addWord("DELETE", "MySQL");
		addWord("DESC", "MySQL");
		addWord("DESCRIBE", "MySQL");
		addWord("DETERMINISTIC", "MySQL");
		addWord("DISTINCT", "MySQL");
		addWord("DISTINCTROW", "MySQL");
		addWord("DIV", "MySQL");
		addWord("DOUBLE", "MySQL");
		addWord("DROP", "MySQL");
		addWord("DUAL", "MySQL");
		addWord("EACH", "MySQL");
		addWord("ELSE", "MySQL");
		addWord("ELSEIF", "MySQL");
		addWord("ENCLOSED", "MySQL");
		addWord("ESCAPED", "MySQL");
		addWord("EXISTS", "MySQL");
		addWord("EXIT", "MySQL");
		addWord("EXPLAIN", "MySQL");
		addWord("FALSE", "MySQL");
		addWord("FETCH", "MySQL");
		addWord("FLOAT", "MySQL");
		addWord("FLOAT4", "MySQL");
		addWord("FLOAT8", "MySQL");
		addWord("FOR", "MySQL");
		addWord("FORCE", "MySQL");
		addWord("FOREIGN", "MySQL");
		addWord("FROM", "MySQL");
		addWord("FULLTEXT", "MySQL");
		addWord("GENERATED[i]", "MySQL");
		addWord("GET", "MySQL");
		addWord("GRANT", "MySQL");
		addWord("GROUP", "MySQL");
		addWord("HAVING", "MySQL");
		addWord("HIGH_PRIORITY", "MySQL");
		addWord("HOUR_MICROSECOND", "MySQL");
		addWord("HOUR_MINUTE", "MySQL");
		addWord("HOUR_SECOND", "MySQL");
		addWord("IF", "MySQL");
		addWord("IGNORE", "MySQL");
		addWord("IN", "MySQL");
		addWord("INDEX", "MySQL");
		addWord("INFILE", "MySQL");
		addWord("INNER", "MySQL");
		addWord("INOUT", "MySQL");
		addWord("INSENSITIVE", "MySQL");
		addWord("INSERT", "MySQL");
		addWord("INT", "MySQL");
		addWord("INT1", "MySQL");
		addWord("INT2", "MySQL");
		addWord("INT3", "MySQL");
		addWord("INT4", "MySQL");
		addWord("INT8", "MySQL");
		addWord("INTEGER", "MySQL");
		addWord("INTERVAL", "MySQL");
		addWord("INTO", "MySQL");
		addWord("IO_AFTER_GTIDS", "MySQL");
		addWord("IO_BEFORE_GTIDS", "MySQL");
		addWord("IS", "MySQL");
		addWord("ITERATE", "MySQL");
		addWord("JOIN", "MySQL");
		addWord("KEY", "MySQL");
		addWord("KEYS", "MySQL");
		addWord("KILL", "MySQL");
		addWord("LEADING", "MySQL");
		addWord("LEAVE", "MySQL");
		addWord("LEFT", "MySQL");
		addWord("LIKE", "MySQL");
		addWord("LIMIT", "MySQL");
		addWord("LINEAR", "MySQL");
		addWord("LINES", "MySQL");
		addWord("LOAD", "MySQL");
		addWord("LOCALTIME", "MySQL");
		addWord("LOCALTIMESTAMP", "MySQL");
		addWord("LOCK", "MySQL");
		addWord("LONG", "MySQL");
		addWord("LONGBLOB", "MySQL");
		addWord("LONGTEXT", "MySQL");
		addWord("LOOP", "MySQL");
		addWord("LOW_PRIORITY", "MySQL");
		addWord("MASTER_BIND", "MySQL");
		addWord("MASTER_SSL_VERIFY_SERVER_CERT", "MySQL");
		addWord("MATCH", "MySQL");
		addWord("MAXVALUE", "MySQL");
		addWord("MEDIUMBLOB", "MySQL");
		addWord("MEDIUMINT", "MySQL");
		addWord("MEDIUMTEXT", "MySQL");
		addWord("MIDDLEINT", "MySQL");
		addWord("MINUTE_MICROSECOND", "MySQL");
		addWord("MINUTE_SECOND", "MySQL");
		addWord("MOD", "MySQL");
		addWord("MODIFIES", "MySQL");
		addWord("NATURAL", "MySQL");
		addWord("NOT", "MySQL");
		addWord("NO_WRITE_TO_BINLOG", "MySQL");
		addWord("NULL", "MySQL");
		addWord("NUMERIC", "MySQL");
		addWord("ON", "MySQL");
		addWord("OPTIMIZE", "MySQL");
		addWord("OPTIMIZER_COSTS[r]", "MySQL");
		addWord("OPTION", "MySQL");
		addWord("OPTIONALLY", "MySQL");
		addWord("OR", "MySQL");
		addWord("ORDER", "MySQL");
		addWord("OUT", "MySQL");
		addWord("OUTER", "MySQL");
		addWord("OUTFILE", "MySQL");
		addWord("PARTITION", "MySQL");
		addWord("PRECISION", "MySQL");
		addWord("PRIMARY", "MySQL");
		addWord("PROCEDURE", "MySQL");
		addWord("PURGE", "MySQL");
		addWord("RANGE", "MySQL");
		addWord("READ", "MySQL");
		addWord("READS", "MySQL");
		addWord("READ_WRITE", "MySQL");
		addWord("REAL", "MySQL");
		addWord("REFERENCES", "MySQL");
		addWord("REGEXP", "MySQL");
		addWord("RELEASE", "MySQL");
		addWord("RENAME", "MySQL");
		addWord("REPEAT", "MySQL");
		addWord("REPLACE", "MySQL");
		addWord("REQUIRE", "MySQL");
		addWord("RESIGNAL", "MySQL");
		addWord("RESTRICT", "MySQL");
		addWord("RETURN", "MySQL");
		addWord("REVOKE", "MySQL");
		addWord("RIGHT", "MySQL");
		addWord("RLIKE", "MySQL");
		addWord("SCHEMA", "MySQL");
		addWord("SCHEMAS", "MySQL");
		addWord("SECOND_MICROSECOND", "MySQL");
		addWord("SELECT", "MySQL");
		addWord("SENSITIVE", "MySQL");
		addWord("SEPARATOR", "MySQL");
		addWord("SET", "MySQL");
		addWord("SHOW", "MySQL");
		addWord("SIGNAL", "MySQL");
		addWord("SMALLINT", "MySQL");
		addWord("SPATIAL", "MySQL");
		addWord("SPECIFIC", "MySQL");
		addWord("SQL", "MySQL");
		addWord("SQLEXCEPTION", "MySQL");
		addWord("SQLSTATE", "MySQL");
		addWord("SQLWARNING", "MySQL");
		addWord("SQL_BIG_RESULT", "MySQL");
		addWord("SQL_CALC_FOUND_ROWS", "MySQL");
		addWord("SQL_SMALL_RESULT", "MySQL");
		addWord("SSL", "MySQL");
		addWord("STARTING", "MySQL");
		addWord("STORED[ac]", "MySQL");
		addWord("STRAIGHT_JOIN", "MySQL");
		addWord("TABLE", "MySQL");
		addWord("TERMINATED", "MySQL");
		addWord("THEN", "MySQL");
		addWord("TINYBLOB", "MySQL");
		addWord("TINYINT", "MySQL");
		addWord("TINYTEXT", "MySQL");
		addWord("TO", "MySQL");
		addWord("TRAILING", "MySQL");
		addWord("TRIGGER", "MySQL");
		addWord("TRUE", "MySQL");
		addWord("UNDO", "MySQL");
		addWord("UNION", "MySQL");
		addWord("UNIQUE", "MySQL");
		addWord("UNLOCK", "MySQL");
		addWord("UNSIGNED", "MySQL");
		addWord("UPDATE", "MySQL");
		addWord("USAGE", "MySQL");
		addWord("USE", "MySQL");
		addWord("USING", "MySQL");
		addWord("UTC_DATE", "MySQL");
		addWord("UTC_TIME", "MySQL");
		addWord("UTC_TIMESTAMP", "MySQL");
		addWord("VALUES", "MySQL");
		addWord("VARBINARY", "MySQL");
		addWord("VARCHAR", "MySQL");
		addWord("VARCHARACTER", "MySQL");
		addWord("VARYING", "MySQL");
		addWord("VIRTUAL[ae]", "MySQL");
		addWord("WHEN", "MySQL");
		addWord("WHERE", "MySQL");
		addWord("WHILE", "MySQL");
		addWord("WITH", "MySQL");
		addWord("WRITE", "MySQL");
		addWord("XOR", "MySQL");
		addWord("YEAR_MONTH", "MySQL");
		addWord("ZEROFILL", "MySQL");
	}

	static {
		// ================SQLSERVER===================================
		// from
		// https://docs.microsoft.com/en-us/sql/t-sql/language-elements/reserved-keywords-transact-sql

		addWord("ADD", "SQLServer");
		addWord("ALL", "SQLServer");
		addWord("ALTER", "SQLServer");
		addWord("AND", "SQLServer");
		addWord("ANY", "SQLServer");
		addWord("AS", "SQLServer");
		addWord("ASC", "SQLServer");
		addWord("AUTHORIZATION", "SQLServer");
		addWord("BACKUP", "SQLServer");
		addWord("BEGIN", "SQLServer");
		addWord("BETWEEN", "SQLServer");
		addWord("BREAK", "SQLServer");
		addWord("BROWSE", "SQLServer");
		addWord("BULK", "SQLServer");
		addWord("BY", "SQLServer");
		addWord("CASCADE", "SQLServer");
		addWord("CASE", "SQLServer");
		addWord("CHECK", "SQLServer");
		addWord("CHECKPOINT", "SQLServer");
		addWord("CLOSE", "SQLServer");
		addWord("CLUSTERED", "SQLServer");
		addWord("COALESCE", "SQLServer");
		addWord("COLLATE", "SQLServer");
		addWord("COLUMN", "SQLServer");
		addWord("COMMIT", "SQLServer");
		addWord("COMPUTE", "SQLServer");
		addWord("CONSTRAINT", "SQLServer");
		addWord("CONTAINS", "SQLServer");
		addWord("CONTAINSTABLE", "SQLServer");
		addWord("CONTINUE", "SQLServer");
		addWord("CONVERT", "SQLServer");
		addWord("CREATE", "SQLServer");
		addWord("CROSS", "SQLServer");
		addWord("CURRENT", "SQLServer");
		addWord("CURRENT_DATE", "SQLServer");
		addWord("CURRENT_TIME", "SQLServer");
		addWord("CURRENT_TIMESTAMP", "SQLServer");
		addWord("CURRENT_USER", "SQLServer");
		addWord("CURSOR", "SQLServer");
		addWord("DATABASE", "SQLServer");
		addWord("DBCC", "SQLServer");
		addWord("DEALLOCATE", "SQLServer");
		addWord("DECLARE", "SQLServer");
		addWord("DEFAULT", "SQLServer");
		addWord("DELETE", "SQLServer");
		addWord("DENY", "SQLServer");
		addWord("DESC", "SQLServer");
		addWord("DISK", "SQLServer");
		addWord("DISTINCT", "SQLServer");
		addWord("DISTRIBUTED", "SQLServer");
		addWord("DOUBLE", "SQLServer");
		addWord("DROP", "SQLServer");
		addWord("DUMP", "SQLServer");
		addWord("ELSE", "SQLServer");
		addWord("END", "SQLServer");
		addWord("ERRLVL", "SQLServer");
		addWord("ESCAPE", "SQLServer");
		addWord("EXCEPT", "SQLServer");
		addWord("EXEC", "SQLServer");
		addWord("EXECUTE", "SQLServer");
		addWord("EXISTS", "SQLServer");
		addWord("EXIT", "SQLServer");
		addWord("EXTERNAL", "SQLServer");
		addWord("FETCH", "SQLServer");
		addWord("FILE", "SQLServer");
		addWord("FILLFACTOR", "SQLServer");
		addWord("FOR", "SQLServer");
		addWord("FOREIGN", "SQLServer");
		addWord("FREETEXT", "SQLServer");
		addWord("FREETEXTTABLE", "SQLServer");
		addWord("FROM", "SQLServer");
		addWord("FULL", "SQLServer");
		addWord("FUNCTION", "SQLServer");
		addWord("GOTO", "SQLServer");
		addWord("GRANT", "SQLServer");
		addWord("GROUP", "SQLServer");
		addWord("HAVING", "SQLServer");
		addWord("HOLDLOCK", "SQLServer");
		addWord("IDENTITY", "SQLServer");
		addWord("IDENTITY_INSERT", "SQLServer");
		addWord("IDENTITYCOL", "SQLServer");
		addWord("IF", "SQLServer");
		addWord("IN", "SQLServer");
		addWord("INDEX", "SQLServer");
		addWord("INNER", "SQLServer");
		addWord("INSERT", "SQLServer");
		addWord("INTERSECT", "SQLServer");
		addWord("INTO", "SQLServer");
		addWord("IS", "SQLServer");
		addWord("JOIN", "SQLServer");
		addWord("KEY", "SQLServer");
		addWord("KILL", "SQLServer");
		addWord("LEFT", "SQLServer");
		addWord("LIKE", "SQLServer");
		addWord("LINENO", "SQLServer");
		addWord("LOAD", "SQLServer");
		addWord("MERGE", "SQLServer");
		addWord("NATIONAL", "SQLServer");
		addWord("NOCHECK", "SQLServer");
		addWord("NONCLUSTERED", "SQLServer");
		addWord("NOT", "SQLServer");
		addWord("NULL", "SQLServer");
		addWord("NULLIF", "SQLServer");
		addWord("OF", "SQLServer");
		addWord("OFF", "SQLServer");
		addWord("OFFSETS", "SQLServer");
		addWord("ON", "SQLServer");
		addWord("OPEN", "SQLServer");
		addWord("OPENDATASOURCE", "SQLServer");
		addWord("OPENQUERY", "SQLServer");
		addWord("OPENROWSET", "SQLServer");
		addWord("OPENXML", "SQLServer");
		addWord("OPTION", "SQLServer");
		addWord("OR", "SQLServer");
		addWord("ORDER", "SQLServer");
		addWord("OUTER", "SQLServer");
		addWord("OVER", "SQLServer");
		addWord("PERCENT", "SQLServer");
		addWord("PIVOT", "SQLServer");
		addWord("PLAN", "SQLServer");
		addWord("PRECISION", "SQLServer");
		addWord("PRIMARY", "SQLServer");
		addWord("PRINT", "SQLServer");
		addWord("PROC", "SQLServer");
		addWord("PROCEDURE", "SQLServer");
		addWord("PUBLIC", "SQLServer");
		addWord("RAISERROR", "SQLServer");
		addWord("READ", "SQLServer");
		addWord("READTEXT", "SQLServer");
		addWord("RECONFIGURE", "SQLServer");
		addWord("REFERENCES", "SQLServer");
		addWord("REPLICATION", "SQLServer");
		addWord("RESTORE", "SQLServer");
		addWord("RESTRICT", "SQLServer");
		addWord("RETURN", "SQLServer");
		addWord("REVERT", "SQLServer");
		addWord("REVOKE", "SQLServer");
		addWord("RIGHT", "SQLServer");
		addWord("ROLLBACK", "SQLServer");
		addWord("ROWCOUNT", "SQLServer");
		addWord("ROWGUIDCOL", "SQLServer");
		addWord("RULE", "SQLServer");
		addWord("SAVE", "SQLServer");
		addWord("SCHEMA", "SQLServer");
		addWord("SECURITYAUDIT", "SQLServer");
		addWord("SELECT", "SQLServer");
		addWord("SEMANTICKEYPHRASETABLE", "SQLServer");
		addWord("SEMANTICSIMILARITYDETAILSTABLE", "SQLServer");
		addWord("SEMANTICSIMILARITYTABLE", "SQLServer");
		addWord("SESSION_USER", "SQLServer");
		addWord("SET", "SQLServer");
		addWord("SETUSER", "SQLServer");
		addWord("SHUTDOWN", "SQLServer");
		addWord("SOME", "SQLServer");
		addWord("STATISTICS", "SQLServer");
		addWord("SYSTEM_USER", "SQLServer");
		addWord("TABLE", "SQLServer");
		addWord("TABLESAMPLE", "SQLServer");
		addWord("TEXTSIZE", "SQLServer");
		addWord("THEN", "SQLServer");
		addWord("TO", "SQLServer");
		addWord("TOP", "SQLServer");
		addWord("TRAN", "SQLServer");
		addWord("TRANSACTION", "SQLServer");
		addWord("TRIGGER", "SQLServer");
		addWord("TRUNCATE", "SQLServer");
		addWord("TRY_CONVERT", "SQLServer");
		addWord("TSEQUAL", "SQLServer");
		addWord("UNION", "SQLServer");
		addWord("UNIQUE", "SQLServer");
		addWord("UNPIVOT", "SQLServer");
		addWord("UPDATE", "SQLServer");
		addWord("UPDATETEXT", "SQLServer");
		addWord("USE", "SQLServer");
		addWord("USER", "SQLServer");
		addWord("VALUES", "SQLServer");
		addWord("VARYING", "SQLServer");
		addWord("VIEW", "SQLServer");
		addWord("WAITFOR", "SQLServer");
		addWord("WHEN", "SQLServer");
		addWord("WHERE", "SQLServer");
		addWord("WHILE", "SQLServer");
		addWord("WITH", "SQLServer");
		addWord("WITHINGROUP", "SQLServer");
		addWord("WRITETEXT", "SQLServer");
	}

	static {
		// =============ORACLE=============================
		// From
		// http://docs.oracle.com/cd/B28359_01/appdev.111/b31231/appb.htm#CJHIIICD
		 addWord("ACCESS","Oracle");
		 addWord("ADD","Oracle");
		 addWord("ALL","Oracle");
		 addWord("ALTER","Oracle");
		 addWord("AND","Oracle");
		 addWord("ANY","Oracle");
		 addWord("ARRAYLEN","Oracle");
		 addWord("AS","Oracle");
		 addWord("ASC","Oracle");
		 addWord("AUDIT","Oracle");
		 addWord("BETWEEN","Oracle");
		 addWord("BY","Oracle");
		 addWord("CHAR","Oracle");
		 addWord("CHECK","Oracle");
		 addWord("CLUSTER","Oracle");
		 addWord("COLUMN","Oracle");
		 addWord("COMMENT","Oracle");
		 addWord("COMPRESS","Oracle");
		 addWord("CONNECT","Oracle");
		 addWord("CREATE","Oracle");
		 addWord("CURRENT","Oracle");
		 addWord("DATE","Oracle");
		 addWord("DECIMAL","Oracle");
		 addWord("DEFAULT","Oracle");
		 addWord("DELETE","Oracle");
		 addWord("DESC","Oracle");
		 addWord("DISTINCT","Oracle");
		 addWord("DROP","Oracle");
		 addWord("ELSE","Oracle");
		 addWord("EXCLUSIVE","Oracle");
		 addWord("EXISTS","Oracle");
		 addWord("FILE","Oracle");
		 addWord("FLOAT","Oracle");
		 addWord("FOR","Oracle");
		 addWord("FROM","Oracle");
		 addWord("GRANT","Oracle");
		 addWord("GROUP","Oracle");
		 addWord("HAVING","Oracle");
		 addWord("IDENTIFIED","Oracle");
		 addWord("IMMEDIATE","Oracle");
		 addWord("IN","Oracle");
		 addWord("INCREMENT","Oracle");
		 addWord("INDEX","Oracle");
		 addWord("INITIAL","Oracle");
		 addWord("INSERT","Oracle");
		 addWord("INTEGER","Oracle");
		 addWord("INTERSECT","Oracle");
		 addWord("INTO","Oracle");
		 addWord("IS","Oracle");
		 addWord("LEVEL","Oracle");
		 addWord("LIKE","Oracle");
		 addWord("LOCK","Oracle");
		 addWord("LONG","Oracle");
		 addWord("MAXEXTENTS","Oracle");
		 addWord("MINUS","Oracle");
		 addWord("MODE","Oracle");
		 addWord("MODIFY","Oracle");
		 addWord("NOAUDIT","Oracle");
		 addWord("NOCOMPRESS","Oracle");
		 addWord("NOT","Oracle");
		 addWord("NOTFOUND","Oracle");
		 addWord("NOWAIT","Oracle");
		 addWord("NULL","Oracle");
		 addWord("NUMBER","Oracle");
		 addWord("OF","Oracle");
		 addWord("OFFLINE","Oracle");
		 addWord("ON","Oracle");
		 addWord("ONLINE","Oracle");
		 addWord("OPTION","Oracle");
		 addWord("OR","Oracle");
		 addWord("ORDER","Oracle");
		 addWord("PCTFREE","Oracle");
		 addWord("PRIOR","Oracle");
		 addWord("PRIVILEGES","Oracle");
		 addWord("PUBLIC","Oracle");
		 addWord("RAW","Oracle");
		 addWord("RENAME","Oracle");
		 addWord("RESOURCE","Oracle");
		 addWord("REVOKE","Oracle");
		 addWord("ROW","Oracle");
		 addWord("ROWID","Oracle");
		 addWord("ROWLABEL","Oracle");
		 addWord("ROWNUM","Oracle");
		 addWord("ROWS","Oracle");
		 addWord("SELECT","Oracle");
		 addWord("SESSION","Oracle");
		 addWord("SET","Oracle");
		 addWord("SHARE","Oracle");
		 addWord("SIZE","Oracle");
		 addWord("SMALLINT","Oracle");
		 addWord("SQLBUF","Oracle");
		 addWord("START","Oracle");
		 addWord("SUCCESSFUL","Oracle");
		 addWord("SYNONYM","Oracle");
		 addWord("SYSDATE","Oracle");
		 addWord("TABLE","Oracle");
		 addWord("THEN","Oracle");
		 addWord("TO","Oracle");
		 addWord("TRIGGER","Oracle");
		 addWord("UID","Oracle");
		 addWord("UNION","Oracle");
		 addWord("UNIQUE","Oracle");
		 addWord("UPDATE","Oracle");
		 addWord("USER","Oracle");
		 addWord("VALIDATE","Oracle");
		 addWord("VALUES","Oracle");
		 addWord("VARCHAR","Oracle");
		 addWord("VARCHAR2","Oracle");
		 addWord("VIEW","Oracle");
		 addWord("WHENEVER","Oracle");
		 addWord("WHERE","Oracle");
		 addWord("WITH","Oracle");

	}

	static {
		// =============PostgreSQL=============================
		// From
		// https://www.postgresql.org/docs/7.3/static/sql-keywords-appendix.html
		addWord("ALL", "PostgreSQL");
		addWord("ANALYSE", "PostgreSQL");
		addWord("ANALYZE", "PostgreSQL");
		addWord("AND", "PostgreSQL");
		addWord("ANY", "PostgreSQL");
		addWord("AS", "PostgreSQL");
		addWord("ASC", "PostgreSQL");
		addWord("AUTHORIZATION", "PostgreSQL");
		addWord("BETWEEN", "PostgreSQL");
		addWord("BINARY", "PostgreSQL");
		addWord("BOTH", "PostgreSQL");
		addWord("CASE", "PostgreSQL");
		addWord("CAST", "PostgreSQL");
		addWord("CHECK", "PostgreSQL");
		addWord("COLLATE", "PostgreSQL");
		addWord("COLUMN", "PostgreSQL");
		addWord("CONSTRAINT", "PostgreSQL");
		addWord("CREATE", "PostgreSQL");
		addWord("CROSS", "PostgreSQL");
		addWord("CURRENT_DATE", "PostgreSQL");
		addWord("CURRENT_TIME", "PostgreSQL");
		addWord("CURRENT_TIMESTAMP", "PostgreSQL");
		addWord("CURRENT_USER", "PostgreSQL");
		addWord("DEFAULT", "PostgreSQL");
		addWord("DEFERRABLE", "PostgreSQL");
		addWord("DESC", "PostgreSQL");
		addWord("DISTINCT", "PostgreSQL");
		addWord("DO", "PostgreSQL");
		addWord("ELSE", "PostgreSQL");
		addWord("END", "PostgreSQL");
		addWord("EXCEPT", "PostgreSQL");
		addWord("FOR", "PostgreSQL");
		addWord("FOREIGN", "PostgreSQL");
		addWord("FREEZE", "PostgreSQL");
		addWord("FROM", "PostgreSQL");
		addWord("FULL", "PostgreSQL");
		addWord("GRANT", "PostgreSQL");
		addWord("GROUP", "PostgreSQL");
		addWord("HAVING", "PostgreSQL");
		addWord("ILIKE", "PostgreSQL");
		addWord("IN", "PostgreSQL");
		addWord("INITIALLY", "PostgreSQL");
		addWord("INNER", "PostgreSQL");
		addWord("INTERSECT", "PostgreSQL");
		addWord("INTO", "PostgreSQL");
		addWord("IS", "PostgreSQL");
		addWord("ISNULL", "PostgreSQL");
		addWord("JOIN", "PostgreSQL");
		addWord("LEADING", "PostgreSQL");
		addWord("LEFT", "PostgreSQL");
		addWord("LIKE", "PostgreSQL");
		addWord("LIMIT", "PostgreSQL");
		addWord("LOCALTIME", "PostgreSQL");
		addWord("LOCALTIMESTAMP", "PostgreSQL");
		addWord("NATURAL", "PostgreSQL");
		addWord("NEW", "PostgreSQL");
		addWord("NOT", "PostgreSQL");
		addWord("NOTNULL", "PostgreSQL");
		addWord("NULL", "PostgreSQL");
		addWord("OFF", "PostgreSQL");
		addWord("OFFSET", "PostgreSQL");
		addWord("OLD", "PostgreSQL");
		addWord("ON", "PostgreSQL");
		addWord("ONLY", "PostgreSQL");
		addWord("OR", "PostgreSQL");
		addWord("ORDER", "PostgreSQL");
		addWord("OUTER", "PostgreSQL");
		addWord("OVERLAPS", "PostgreSQL");
		addWord("PLACING", "PostgreSQL");
		addWord("PRIMARY", "PostgreSQL");
		addWord("REFERENCES", "PostgreSQL");
		addWord("RIGHT", "PostgreSQL");
		addWord("SELECT", "PostgreSQL");
		addWord("SESSION_USER", "PostgreSQL");
		addWord("SIMILAR", "PostgreSQL");
		addWord("SOME", "PostgreSQL");
		addWord("TABLE", "PostgreSQL");
		addWord("THEN", "PostgreSQL");
		addWord("TO", "PostgreSQL");
		addWord("TRAILING", "PostgreSQL");
		addWord("UNION", "PostgreSQL");
		addWord("UNIQUE", "PostgreSQL");
		addWord("USER", "PostgreSQL");
		addWord("USING", "PostgreSQL");
		addWord("VERBOSE", "PostgreSQL");
		addWord("WHEN", "PostgreSQL");
		addWord("WHERE", "PostgreSQL");
		addWord("FALSE", "PostgreSQL");
		addWord("TRUE", "PostgreSQL");
	}

	static {
		// =============DB2=============================
		// From
		// https://www.ibm.com/support/knowledgecenter/en/SSEPGG_9.7.0/com.ibm.db2.luw.sql.ref.doc/doc/r0001095.html
		addWord("ACTIVATE", "DB2");
		addWord("ADD", "DB2");
		addWord("AFTER", "DB2");
		addWord("ALIAS", "DB2");
		addWord("ALL", "DB2");
		addWord("ALLOCATE", "DB2");
		addWord("ALLOW", "DB2");
		addWord("ALTER", "DB2");
		addWord("AND", "DB2");
		addWord("ANY", "DB2");
		addWord("AS", "DB2");
		addWord("ASENSITIVE", "DB2");
		addWord("ASSOCIATE", "DB2");
		addWord("ASUTIME", "DB2");
		addWord("AT", "DB2");
		addWord("ATTRIBUTES", "DB2");
		addWord("AUDIT", "DB2");
		addWord("AUTHORIZATION", "DB2");
		addWord("AUX", "DB2");
		addWord("AUXILIARY", "DB2");
		addWord("BEFORE", "DB2");
		addWord("BEGIN", "DB2");
		addWord("BETWEEN", "DB2");
		addWord("BINARY", "DB2");
		addWord("BUFFERPOOL", "DB2");
		addWord("BY", "DB2");
		addWord("CACHE", "DB2");
		addWord("CALL", "DB2");
		addWord("CALLED", "DB2");
		addWord("CAPTURE", "DB2");
		addWord("CARDINALITY", "DB2");
		addWord("CASCADED", "DB2");
		addWord("CASE", "DB2");
		addWord("CAST", "DB2");
		addWord("CCSID", "DB2");
		addWord("CHAR", "DB2");
		addWord("CHARACTER", "DB2");
		addWord("CHECK", "DB2");
		addWord("CLONE", "DB2");
		addWord("CLOSE", "DB2");
		addWord("CLUSTER", "DB2");
		addWord("COLLECTION", "DB2");
		addWord("COLLID", "DB2");
		addWord("COLUMN", "DB2");
		addWord("COMMENT", "DB2");
		addWord("COMMIT", "DB2");
		addWord("CONCAT", "DB2");
		addWord("CONDITION", "DB2");
		addWord("CONNECT", "DB2");
		addWord("CONNECTION", "DB2");
		addWord("CONSTRAINT", "DB2");
		addWord("CONTAINS", "DB2");
		addWord("CONTINUE", "DB2");
		addWord("COUNT", "DB2");
		addWord("COUNT_BIG", "DB2");
		addWord("CREATE", "DB2");
		addWord("CROSS", "DB2");
		addWord("CURRENT", "DB2");
		addWord("CURRENT_DATE", "DB2");
		addWord("CURRENT_LC_CTYPE", "DB2");
		addWord("CURRENT_PATH", "DB2");
		addWord("CURRENT_SCHEMA", "DB2");
		addWord("CURRENT_SERVER", "DB2");
		addWord("CURRENT_TIME", "DB2");
		addWord("CURRENT_TIMESTAMP", "DB2");
		addWord("CURRENT_TIMEZONE", "DB2");
		addWord("CURRENT_USER", "DB2");
		addWord("CURSOR", "DB2");
		addWord("CYCLE", "DB2");
		addWord("DATA", "DB2");
		addWord("DATABASE", "DB2");
		addWord("DATAPARTITIONNAME", "DB2");
		addWord("DATAPARTITIONNUM", "DB2");
		addWord("DATE", "DB2");
		addWord("DAY", "DB2");
		addWord("DAYS", "DB2");
		addWord("DB2GENERAL", "DB2");
		addWord("DB2GENRL", "DB2");
		addWord("DB2SQL", "DB2");
		addWord("DBINFO", "DB2");
		addWord("DBPARTITIONNAME", "DB2");
		addWord("DBPARTITIONNUM", "DB2");
		addWord("DEALLOCATE", "DB2");
		addWord("DECLARE", "DB2");
		addWord("DEFAULT", "DB2");
		addWord("DEFAULTS", "DB2");
		addWord("DEFINITION", "DB2");
		addWord("DELETE", "DB2");
		addWord("DENSE_RANK", "DB2");
		addWord("DENSERANK", "DB2");
		addWord("DESCRIBE", "DB2");
		addWord("DESCRIPTOR", "DB2");
		addWord("DETERMINISTIC", "DB2");
		addWord("DIAGNOSTICS", "DB2");
		addWord("DISABLE", "DB2");
		addWord("DISALLOW", "DB2");
		addWord("DISCONNECT", "DB2");
		addWord("DISTINCT", "DB2");
		addWord("DO", "DB2");
		addWord("DOCUMENT", "DB2");
		addWord("DOUBLE", "DB2");
		addWord("DROP", "DB2");
		addWord("DSSIZE", "DB2");
		addWord("DYNAMIC", "DB2");
		addWord("EACH", "DB2");
		addWord("EDITPROC", "DB2");
		addWord("ELSE", "DB2");
		addWord("ELSEIF", "DB2");
		addWord("ENABLE", "DB2");
		addWord("ENCODING", "DB2");
		addWord("ENCRYPTION", "DB2");
		addWord("END", "DB2");
		addWord("END-EXEC", "DB2");
		addWord("ENDING", "DB2");
		addWord("ERASE", "DB2");
		addWord("ESCAPE", "DB2");
		addWord("EVERY", "DB2");
		addWord("EXCEPT", "DB2");
		addWord("EXCEPTION", "DB2");
		addWord("EXCLUDING", "DB2");
		addWord("EXCLUSIVE", "DB2");
		addWord("EXECUTE", "DB2");
		addWord("EXISTS", "DB2");
		addWord("EXIT", "DB2");
		addWord("EXPLAIN", "DB2");
		addWord("EXTENDED", "DB2");
		addWord("EXTERNAL", "DB2");
		addWord("EXTRACT", "DB2");
		addWord("FENCED", "DB2");
		addWord("FETCH", "DB2");
		addWord("FIELDPROC", "DB2");
		addWord("FILE", "DB2");
		addWord("FINAL", "DB2");
		addWord("FOR", "DB2");
		addWord("FOREIGN", "DB2");
		addWord("FREE", "DB2");
		addWord("FROM", "DB2");
		addWord("FULL", "DB2");
		addWord("FUNCTION", "DB2");
		addWord("GENERAL", "DB2");
		addWord("GENERATED", "DB2");
		addWord("GET", "DB2");
		addWord("GLOBAL", "DB2");
		addWord("GO", "DB2");
		addWord("GOTO", "DB2");
		addWord("GRANT", "DB2");
		addWord("GRAPHIC", "DB2");
		addWord("GROUP", "DB2");
		addWord("HANDLER", "DB2");
		addWord("HASH", "DB2");
		addWord("HASHED_VALUE", "DB2");
		addWord("HAVING", "DB2");
		addWord("HINT", "DB2");
		addWord("HOLD", "DB2");
		addWord("HOUR", "DB2");
		addWord("HOURS", "DB2");
		addWord("IDENTITY", "DB2");
		addWord("IF", "DB2");
		addWord("IMMEDIATE", "DB2");
		addWord("IN", "DB2");
		addWord("INCLUDING", "DB2");
		addWord("INCLUSIVE", "DB2");
		addWord("INCREMENT", "DB2");
		addWord("INDEX", "DB2");
		addWord("INDICATOR", "DB2");
		addWord("INDICATORS", "DB2");
		addWord("INF", "DB2");
		addWord("INFINITY", "DB2");
		addWord("INHERIT", "DB2");
		addWord("INNER", "DB2");
		addWord("INOUT", "DB2");
		addWord("INSENSITIVE", "DB2");
		addWord("INSERT", "DB2");
		addWord("INTEGRITY", "DB2");
		addWord("INTERSECT", "DB2");
		addWord("INTO", "DB2");
		addWord("IS", "DB2");
		addWord("ISOBID", "DB2");
		addWord("ISOLATION", "DB2");
		addWord("ITERATE", "DB2");
		addWord("JAR", "DB2");
		addWord("JAVA", "DB2");
		addWord("JOIN", "DB2");
		addWord("KEEP", "DB2");
		addWord("KEY", "DB2");
		addWord("LABEL", "DB2");
		addWord("LANGUAGE", "DB2");
		addWord("LATERAL", "DB2");
		addWord("LC_CTYPE", "DB2");
		addWord("LEAVE", "DB2");
		addWord("LEFT", "DB2");
		addWord("LIKE", "DB2");
		addWord("LIMIT", "DB2");
		addWord("LINKTYPE", "DB2");
		addWord("LOCAL", "DB2");
		addWord("LOCALDATE", "DB2");
		addWord("LOCALE", "DB2");
		addWord("LOCALTIME", "DB2");
		addWord("LOCALTIMESTAMP", "DB2");
		addWord("LOCATOR", "DB2");
		addWord("LOCATORS", "DB2");
		addWord("LOCK", "DB2");
		addWord("LOCKMAX", "DB2");
		addWord("LOCKSIZE", "DB2");
		addWord("LONG", "DB2");
		addWord("LOOP", "DB2");
		addWord("MAINTAINED", "DB2");
		addWord("MATERIALIZED", "DB2");
		addWord("MAXVALUE", "DB2");
		addWord("MICROSECOND", "DB2");
		addWord("MICROSECONDS", "DB2");
		addWord("MINUTE", "DB2");
		addWord("MINUTES", "DB2");
		addWord("MINVALUE", "DB2");
		addWord("MODE", "DB2");
		addWord("MODIFIES", "DB2");
		addWord("MONTH", "DB2");
		addWord("MONTHS", "DB2");
		addWord("NAN", "DB2");
		addWord("NEW", "DB2");
		addWord("NEW_TABLE", "DB2");
		addWord("NEXTVAL", "DB2");
		addWord("NO", "DB2");
		addWord("NOCACHE", "DB2");
		addWord("NOCYCLE", "DB2");
		addWord("NODENAME", "DB2");
		addWord("NODENUMBER", "DB2");
		addWord("NOMAXVALUE", "DB2");
		addWord("NOMINVALUE", "DB2");
		addWord("NONE", "DB2");
		addWord("NOORDER", "DB2");
		addWord("NORMALIZED", "DB2");
		addWord("NOT", "DB2");
		addWord("NULL", "DB2");
		addWord("NULLS", "DB2");
		addWord("NUMPARTS", "DB2");
		addWord("OBID", "DB2");
		addWord("OF", "DB2");
		addWord("OFFSET", "DB2");
		addWord("OLD", "DB2");
		addWord("OLD_TABLE", "DB2");
		addWord("ON", "DB2");
		addWord("OPEN", "DB2");
		addWord("OPTIMIZATION", "DB2");
		addWord("OPTIMIZE", "DB2");
		addWord("OPTION", "DB2");
		addWord("OR", "DB2");
		addWord("ORDER", "DB2");
		addWord("OUT", "DB2");
		addWord("OUTER", "DB2");
		addWord("OVER", "DB2");
		addWord("OVERRIDING", "DB2");
		addWord("PACKAGE", "DB2");
		addWord("PADDED", "DB2");
		addWord("PAGESIZE", "DB2");
		addWord("PARAMETER", "DB2");
		addWord("PART", "DB2");
		addWord("PARTITION", "DB2");
		addWord("PARTITIONED", "DB2");
		addWord("PARTITIONING", "DB2");
		addWord("PARTITIONS", "DB2");
		addWord("PASSWORD", "DB2");
		addWord("PATH", "DB2");
		addWord("PIECESIZE", "DB2");
		addWord("PLAN", "DB2");
		addWord("POSITION", "DB2");
		addWord("PRECISION", "DB2");
		addWord("PREPARE", "DB2");
		addWord("PREVVAL", "DB2");
		addWord("PRIMARY", "DB2");
		addWord("PRIQTY", "DB2");
		addWord("PRIVILEGES", "DB2");
		addWord("PROCEDURE", "DB2");
		addWord("PROGRAM", "DB2");
		addWord("PSID", "DB2");
		addWord("PUBLIC", "DB2");
		addWord("QUERY", "DB2");
		addWord("QUERYNO", "DB2");
		addWord("RANGE", "DB2");
		addWord("RANK", "DB2");
		addWord("READ", "DB2");
		addWord("READS", "DB2");
		addWord("RECOVERY", "DB2");
		addWord("REFERENCES", "DB2");
		addWord("REFERENCING", "DB2");
		addWord("REFRESH", "DB2");
		addWord("RELEASE", "DB2");
		addWord("RENAME", "DB2");
		addWord("REPEAT", "DB2");
		addWord("RESET", "DB2");
		addWord("RESIGNAL", "DB2");
		addWord("RESTART", "DB2");
		addWord("RESTRICT", "DB2");
		addWord("RESULT", "DB2");
		addWord("RESULT_SET_LOCATOR", "DB2");
		addWord("RETURN", "DB2");
		addWord("RETURNS", "DB2");
		addWord("REVOKE", "DB2");
		addWord("RIGHT", "DB2");
		addWord("ROLE", "DB2");
		addWord("ROLLBACK", "DB2");
		addWord("ROUND_CEILING", "DB2");
		addWord("ROUND_DOWN", "DB2");
		addWord("ROUND_FLOOR", "DB2");
		addWord("ROUND_HALF_DOWN", "DB2");
		addWord("ROUND_HALF_EVEN", "DB2");
		addWord("ROUND_HALF_UP", "DB2");
		addWord("ROUND_UP", "DB2");
		addWord("ROUTINE", "DB2");
		addWord("ROW", "DB2");
		addWord("ROW_NUMBER", "DB2");
		addWord("ROWNUMBER", "DB2");
		addWord("ROWS", "DB2");
		addWord("ROWSET", "DB2");
		addWord("RRN", "DB2");
		addWord("RUN", "DB2");
		addWord("SAVEPOINT", "DB2");
		addWord("SCHEMA", "DB2");
		addWord("SCRATCHPAD", "DB2");
		addWord("SCROLL", "DB2");
		addWord("SEARCH", "DB2");
		addWord("SECOND", "DB2");
		addWord("SECONDS", "DB2");
		addWord("SECQTY", "DB2");
		addWord("SECURITY", "DB2");
		addWord("SELECT", "DB2");
		addWord("SENSITIVE", "DB2");
		addWord("SEQUENCE", "DB2");
		addWord("SESSION", "DB2");
		addWord("SESSION_USER", "DB2");
		addWord("SET", "DB2");
		addWord("SIGNAL", "DB2");
		addWord("SIMPLE", "DB2");
		addWord("SNAN", "DB2");
		addWord("SOME", "DB2");
		addWord("SOURCE", "DB2");
		addWord("SPECIFIC", "DB2");
		addWord("SQL", "DB2");
		addWord("SQLID", "DB2");
		addWord("STACKED", "DB2");
		addWord("STANDARD", "DB2");
		addWord("START", "DB2");
		addWord("STARTING", "DB2");
		addWord("STATEMENT", "DB2");
		addWord("STATIC", "DB2");
		addWord("STATMENT", "DB2");
		addWord("STAY", "DB2");
		addWord("STOGROUP", "DB2");
		addWord("STORES", "DB2");
		addWord("STYLE", "DB2");
		addWord("SUBSTRING", "DB2");
		addWord("SUMMARY", "DB2");
		addWord("SYNONYM", "DB2");
		addWord("SYSFUN", "DB2");
		addWord("SYSIBM", "DB2");
		addWord("SYSPROC", "DB2");
		addWord("SYSTEM", "DB2");
		addWord("SYSTEM_USER", "DB2");
		addWord("TABLE", "DB2");
		addWord("TABLESPACE", "DB2");
		addWord("THEN", "DB2");
		addWord("TIME", "DB2");
		addWord("TIMESTAMP", "DB2");
		addWord("TO", "DB2");
		addWord("TRANSACTION", "DB2");
		addWord("TRIGGER", "DB2");
		addWord("TRIM", "DB2");
		addWord("TRUNCATE", "DB2");
		addWord("TYPE", "DB2");
		addWord("UNDO", "DB2");
		addWord("UNION", "DB2");
		addWord("UNIQUE", "DB2");
		addWord("UNTIL", "DB2");
		addWord("UPDATE", "DB2");
		addWord("USAGE", "DB2");
		addWord("USER", "DB2");
		addWord("USING", "DB2");
		addWord("VALIDPROC", "DB2");
		addWord("VALUE", "DB2");
		addWord("VALUES", "DB2");
		addWord("VARIABLE", "DB2");
		addWord("VARIANT", "DB2");
		addWord("VCAT", "DB2");
		addWord("VERSION", "DB2");
		addWord("VIEW", "DB2");
		addWord("VOLATILE", "DB2");
		addWord("VOLUMES", "DB2");
		addWord("WHEN", "DB2");
		addWord("WHENEVER", "DB2");
		addWord("WHERE", "DB2");
		addWord("WHILE", "DB2");
		addWord("WITH", "DB2");
		addWord("WITHOUT", "DB2");
		addWord("WLM", "DB2");
		addWord("WRITE", "DB2");
		addWord("XMLELEMENT", "DB2");
		addWord("XMLEXISTS", "DB2");
		addWord("XMLNAMESPACES", "DB2");
		addWord("YEAR", "DB2");
		addWord("YEARS", "DB2");
	}

	static {
		// =============Access=============================
		// From
		// https://msdn.microsoft.com/en-us/library/bb208875(v=office.12).aspx
		addWord("ABSOLUTE", "Access");
		addWord("ANY", "Access");
		addWord("ADD", "Access");
		addWord("ARE", "Access");
		addWord("ADMINDB", "Access");
		addWord("AS", "Access");
		addWord("ALL", "Access");
		addWord("ASC", "Access");
		addWord("Alphanumeric", "Access");
		addWord("ASSERTION", "Access");
		addWord("ALTER", "Access");
		addWord("AUTHORIZATION", "Access");
		addWord("ALTER TABLE", "Access");
		addWord("AUTOINCREMENT", "Access");
		addWord("And", "Access");
		addWord("Avg", "Access");
		addWord("BEGIN", "Access");
		addWord("COLLATION", "Access");
		addWord("Between", "Access");
		addWord("COLUMN", "Access");
		addWord("BINARY", "Access");
		addWord("COMMIT", "Access");
		addWord("BIT", "Access");
		addWord("COMP", "Access");
		addWord("COMPRESSION", "Access");
		addWord("BIT_LENGTH", "Access");
		addWord("CONNECT", "Access");
		addWord("BOOLEAN", "Access");
		addWord("CONNECTION", "Access");
		addWord("BOTH", "Access");
		addWord("CONSTRAINT", "Access");
		addWord("CONSTRAINTS", "Access");
		addWord("BY", "Access");
		addWord("CONTAINER", "Access");
		addWord("BYTE", "Access");
		addWord("CONTAINS", "Access");
		addWord("CASCADE", "Access");
		addWord("CONVERT", "Access");
		addWord("CATALOG", "Access");
		addWord("Count", "Access");
		addWord("CHAR", "Access");
		addWord("CHARACTER", "Access");
		addWord("COUNTER", "Access");
		addWord("CHAR_LENGTH", "Access");
		addWord("CREATE", "Access");
		addWord("CHARACTER_LENGTH", "Access");
		addWord("CURRENCY", "Access");
		addWord("CHECK", "Access");
		addWord("CURRENT_DATE", "Access");
		addWord("CLOSE", "Access");
		addWord("CURRENT_TIME", "Access");
		addWord("CLUSTERED", "Access");
		addWord("CURRENT_TIMESTAMP", "Access");
		addWord("COALESCE", "Access");
		addWord("CURRENT_USER", "Access");
		addWord("COLLATE", "Access");
		addWord("CURSOR", "Access");
		addWord("DATABASE", "Access");
		addWord("DISALLOW", "Access");
		addWord("DATE", "Access");
		addWord("DISCONNECT", "Access");
		addWord("DATETIME", "Access");
		addWord("DISTINCT", "Access");
		addWord("DAY", "Access");
		addWord("DISTINCTROW", "Access");
		addWord("DEC", "Access");
		addWord("DECIMAL", "Access");
		addWord("DOMAIN", "Access");
		addWord("DECLARE", "Access");
		addWord("DOUBLE", "Access");
		addWord("DELETE", "Access");
		addWord("DROP", "Access");
		addWord("DESC", "Access");
		addWord("Eqv", "Access");
		addWord("FOREIGN", "Access");
		addWord("EXCLUSIVECONNECT", "Access");
		addWord("FROM", "Access");
		addWord("EXEC", "Access");
		addWord("EXECUTE", "Access");
		addWord("FROM Clause", "Access");
		addWord("EXISTS", "Access");
		addWord("GENERAL", "Access");
		addWord("EXTRACT", "Access");
		addWord("GRANT", "Access");
		addWord("FALSE", "Access");
		addWord("GROUP", "Access");
		addWord("FETCH", "Access");
		addWord("GUID", "Access");
		addWord("FIRST", "Access");
		addWord("HAVING", "Access");
		addWord("FLOAT", "Access");
		addWord("FLOAT8", "Access");
		addWord("HOUR", "Access");
		addWord("FLOAT4", "Access");
		addWord("IDENTITY", "Access");
		addWord("INPUT", "Access");
		addWord("IEEEDOUBLE", "Access");
		addWord("INSENSITIVE", "Access");
		addWord("IEEESINGLE", "Access");
		addWord("INSERT", "Access");
		addWord("IGNORE", "Access");
		addWord("INSERT INTO", "Access");
		addWord("IMAGE", "Access");
		addWord("INT", "Access");
		addWord("INTEGER", "Access");
		addWord("INTEGER4", "Access");
		addWord("Imp", "Access");
		addWord("INTEGER1", "Access");
		addWord("In", "Access");
		addWord("INTEGER2", "Access");
		addWord("INTERVAL", "Access");
		addWord("INDEX", "Access");
		addWord("INTO", "Access");
		addWord("INDEXCREATEDB", "Access");
		addWord("Is", "Access");
		addWord("INNER", "Access");
		addWord("ISOLATION", "Access");
		addWord("JOIN", "Access");
		addWord("LONGTEXT", "Access");
		addWord("KEY", "Access");
		addWord("LOWER", "Access");
		addWord("LANGUAGE", "Access");
		addWord("MATCH", "Access");
		addWord("LAST", "Access");
		addWord("Max", "Access");
		addWord("LEFT", "Access");
		addWord("MEMO", "Access");
		addWord("Level", "Access");
		addWord("Min", "Access");
		addWord("Like", "Access");
		addWord("MINUTE", "Access");
		addWord("LOGICAL", "Access");
		addWord("LOGICAL1", "Access");
		addWord("Mod", "Access");
		addWord("LONG", "Access");
		addWord("MONEY", "Access");
		addWord("LONGBINARY", "Access");
		addWord("MONTH", "Access");
		addWord("LONGCHAR", "Access");
		addWord("NATIONAL", "Access");
		addWord("Outer", "Access");
		addWord("NCHAR", "Access");
		addWord("OUTPUT", "Access");
		addWord("NONCLUSTERED", "Access");
		addWord("OWNERACCESS", "Access");
		addWord("Not", "Access");
		addWord("PAD", "Access");
		addWord("NTEXT", "Access");
		addWord("PARAMETERS", "Access");
		addWord("NULL", "Access");
		addWord("PARTIAL", "Access");
		addWord("NUMBER", "Access");
		addWord("PASSWORD", "Access");
		addWord("NUMERIC", "Access");
		addWord("PERCENT", "Access");
		addWord("NVARCHAR", "Access");
		addWord("PIVOT", "Access");
		addWord("OCTET_LENGTH", "Access");
		addWord("POSITION", "Access");
		addWord("OLEOBJECT", "Access");
		addWord("PRECISION", "Access");
		addWord("ON", "Access");
		addWord("PREPARE", "Access");
		addWord("OPEN", "Access");
		addWord("PRIMARY", "Access");
		addWord("OPTION", "Access");
		addWord("PRIVILEGES", "Access");
		addWord("Or", "Access");
		addWord("PROC", "Access");
		addWord("PROCEDURE", "Access");
		addWord("ORDER", "Access");
		addWord("PUBLIC", "Access");
		addWord("REAL", "Access");
		addWord("SMALLDATETIME", "Access");
		addWord("REFERENCES", "Access");
		addWord("SMALLINT", "Access");
		addWord("RESTRICT", "Access");
		addWord("SMALLMONEY", "Access");
		addWord("REVOKE", "Access");
		addWord("SOME", "Access");
		addWord("RIGHT", "Access");
		addWord("SPACE", "Access");
		addWord("ROLLBACK", "Access");
		addWord("SQL", "Access");
		addWord("SCHEMA", "Access");
		addWord("SQLCODE", "Access");
		addWord("SQLERROR", "Access");
		addWord("SQLSTATE", "Access");
		addWord("SECOND", "Access");
		addWord("StDev", "Access");
		addWord("SELECT", "Access");
		addWord("StDevP", "Access");
		addWord("SELECTSCHEMA", "Access");
		addWord("STRING", "Access");
		addWord("SELECTSECURITY", "Access");
		addWord("SUBSTRING", "Access");
		addWord("SET", "Access");
		addWord("Sum", "Access");
		addWord("SHORT", "Access");
		addWord("SYSNAME", "Access");
		addWord("SINGLE", "Access");
		addWord("SYSTEM_USER", "Access");
		addWord("SIZE", "Access");
		addWord("TABLE", "Access");
		addWord("UPDATEOWNER", "Access");
		addWord("TableID", "Access");
		addWord("UPDATESECURITY", "Access");
		addWord("TEMPORARY", "Access");
		addWord("UPPER", "Access");
		addWord("TEXT", "Access");
		addWord("USAGE", "Access");
		addWord("TIME", "Access");
		addWord("USER", "Access");
		addWord("TIMESTAMP", "Access");
		addWord("USING", "Access");
		addWord("TIMEZONE_HOUR", "Access");
		addWord("VALUE", "Access");
		addWord("TIMEZONE_MINUTE", "Access");
		addWord("VALUES", "Access");
		addWord("TINYINT", "Access");
		addWord("Var", "Access");
		addWord("TO", "Access");
		addWord("VARBINARY", "Access");
		addWord("TOP", "Access");
		addWord("VARCHAR", "Access");
		addWord("TRAILING", "Access");
		addWord("VarP", "Access");
		addWord("TRANSACTION", "Access");
		addWord("VARYING", "Access");
		addWord("TRANSFORM", "Access");
		addWord("VIEW", "Access");
		addWord("TRANSLATE", "Access");
		addWord("WHEN", "Access");
		addWord("TRANSLATION", "Access");
		addWord("WHENEVER", "Access");
		addWord("TRIM", "Access");
		addWord("WHERE", "Access");
		addWord("TRUE", "Access");
		addWord("WITH", "Access");
		addWord("UNION", "Access");
		addWord("WORK", "Access");
		addWord("UNIQUE", "Access");
		addWord("Xor", "Access");
		addWord("UNIQUEIDENTIFIER", "Access");
		addWord("YEAR", "Access");
		addWord("UNKNOWN", "Access");
		addWord("YESNO", "Access");
		addWord("UPDATE", "Access");
		addWord("ZONE", "Access");
		addWord("UPDATEIDENTITY", "Access");
	}
	static {
		// =============SQLite=============================
		// From
		// http://www.sqlite.org/lang_keywords.html
		addWord("ABORT", "SQLite");
		addWord("ACTION", "SQLite");
		addWord("ADD", "SQLite");
		addWord("AFTER", "SQLite");
		addWord("ALL", "SQLite");
		addWord("ALTER", "SQLite");
		addWord("ANALYZE", "SQLite");
		addWord("AND", "SQLite");
		addWord("AS", "SQLite");
		addWord("ASC", "SQLite");
		addWord("ATTACH", "SQLite");
		addWord("AUTOINCREMENT", "SQLite");
		addWord("BEFORE", "SQLite");
		addWord("BEGIN", "SQLite");
		addWord("BETWEEN", "SQLite");
		addWord("BY", "SQLite");
		addWord("CASCADE", "SQLite");
		addWord("CASE", "SQLite");
		addWord("CAST", "SQLite");
		addWord("CHECK", "SQLite");
		addWord("COLLATE", "SQLite");
		addWord("COLUMN", "SQLite");
		addWord("COMMIT", "SQLite");
		addWord("CONFLICT", "SQLite");
		addWord("CONSTRAINT", "SQLite");
		addWord("CREATE", "SQLite");
		addWord("CROSS", "SQLite");
		addWord("CURRENT_DATE", "SQLite");
		addWord("CURRENT_TIME", "SQLite");
		addWord("CURRENT_TIMESTAMP", "SQLite");
		addWord("DATABASE", "SQLite");
		addWord("DEFAULT", "SQLite");
		addWord("DEFERRABLE", "SQLite");
		addWord("DEFERRED", "SQLite");
		addWord("DELETE", "SQLite");
		addWord("DESC", "SQLite");
		addWord("DETACH", "SQLite");
		addWord("DISTINCT", "SQLite");
		addWord("DROP", "SQLite");
		addWord("EACH", "SQLite");
		addWord("ELSE", "SQLite");
		addWord("END", "SQLite");
		addWord("ESCAPE", "SQLite");
		addWord("EXCEPT", "SQLite");
		addWord("EXCLUSIVE", "SQLite");
		addWord("EXISTS", "SQLite");
		addWord("EXPLAIN", "SQLite");
		addWord("FAIL", "SQLite");
		addWord("FOR", "SQLite");
		addWord("FOREIGN", "SQLite");
		addWord("FROM", "SQLite");
		addWord("FULL", "SQLite");
		addWord("GLOB", "SQLite");
		addWord("GROUP", "SQLite");
		addWord("HAVING", "SQLite");
		addWord("IF", "SQLite");
		addWord("IGNORE", "SQLite");
		addWord("IMMEDIATE", "SQLite");
		addWord("IN", "SQLite");
		addWord("INDEX", "SQLite");
		addWord("INDEXED", "SQLite");
		addWord("INITIALLY", "SQLite");
		addWord("INNER", "SQLite");
		addWord("INSERT", "SQLite");
		addWord("INSTEAD", "SQLite");
		addWord("INTERSECT", "SQLite");
		addWord("INTO", "SQLite");
		addWord("IS", "SQLite");
		addWord("ISNULL", "SQLite");
		addWord("JOIN", "SQLite");
		addWord("KEY", "SQLite");
		addWord("LEFT", "SQLite");
		addWord("LIKE", "SQLite");
		addWord("LIMIT", "SQLite");
		addWord("MATCH", "SQLite");
		addWord("NATURAL", "SQLite");
		addWord("NO", "SQLite");
		addWord("NOT", "SQLite");
		addWord("NOTNULL", "SQLite");
		addWord("NULL", "SQLite");
		addWord("OF", "SQLite");
		addWord("OFFSET", "SQLite");
		addWord("ON", "SQLite");
		addWord("OR", "SQLite");
		addWord("ORDER", "SQLite");
		addWord("OUTER", "SQLite");
		addWord("PLAN", "SQLite");
		addWord("PRAGMA", "SQLite");
		addWord("PRIMARY", "SQLite");
		addWord("QUERY", "SQLite");
		addWord("RAISE", "SQLite");
		addWord("RECURSIVE", "SQLite");
		addWord("REFERENCES", "SQLite");
		addWord("REGEXP", "SQLite");
		addWord("REINDEX", "SQLite");
		addWord("RELEASE", "SQLite");
		addWord("RENAME", "SQLite");
		addWord("REPLACE", "SQLite");
		addWord("RESTRICT", "SQLite");
		addWord("RIGHT", "SQLite");
		addWord("ROLLBACK", "SQLite");
		addWord("ROW", "SQLite");
		addWord("SAVEPOINT", "SQLite");
		addWord("SELECT", "SQLite");
		addWord("SET", "SQLite");
		addWord("TABLE", "SQLite");
		addWord("TEMP", "SQLite");
		addWord("TEMPORARY", "SQLite");
		addWord("THEN", "SQLite");
		addWord("TO", "SQLite");
		addWord("TRANSACTION", "SQLite");
		addWord("TRIGGER", "SQLite");
		addWord("UNION", "SQLite");
		addWord("UNIQUE", "SQLite");
		addWord("UPDATE", "SQLite");
		addWord("USING", "SQLite");
		addWord("VACUUM", "SQLite");
		addWord("VALUES", "SQLite");
		addWord("VIEW", "SQLite");
		addWord("VIRTUAL", "SQLite");
		addWord("WHEN", "SQLite");
		addWord("WHERE", "SQLite");
		addWord("WITH", "SQLite");
		addWord("WITHOUT", "SQLite");
	}

	static {
		// =============Teradata=============================
		// From
		// http://walton.uark.edu/enterprise/downloads/Teradata/Teradata_Database_Reserved.pdf
		addWord("ABORT", "Teradata");
		addWord("ABORTSESSION", "Teradata");
		addWord("ABS", "Teradata");
		addWord("ACCESS_LOCK", "Teradata");
		addWord("ACCOUNT", "Teradata");
		addWord("ACOS", "Teradata");
		addWord("ACOSH", "Teradata");
		addWord("ADD", "Teradata");
		addWord("ADD_MONTHS", "Teradata");
		addWord("ADMIN", "Teradata");
		addWord("AFTER", "Teradata");
		addWord("AGGREGATE", "Teradata");
		addWord("ALIAS", "Teradata");
		addWord("ALL", "Teradata");
		addWord("ALTER", "Teradata");
		addWord("AMP", "Teradata");
		addWord("AND", "Teradata");
		addWord("ANSIDATE", "Teradata");
		addWord("ANY", "Teradata");
		addWord("AS", "Teradata");
		addWord("ASC", "Teradata");
		addWord("ASIN", "Teradata");
		addWord("ASINH", "Teradata");
		addWord("AT", "Teradata");
		addWord("ATAN", "Teradata");
		addWord("ATAN2", "Teradata");
		addWord("ATANH", "Teradata");
		addWord("ATOMIC", "Teradata");
		addWord("AUTHORIZATION", "Teradata");
		addWord("AVE", "Teradata");
		addWord("AVERAGE", "Teradata");
		addWord("AVG", "Teradata");
		addWord("BEFORE", "Teradata");
		addWord("BEGIN", "Teradata");
		addWord("BETWEEN", "Teradata");
		addWord("BLOB", "Teradata");
		addWord("BOTH", "Teradata");
		addWord("BT", "Teradata");
		addWord("BUT", "Teradata");
		addWord("BY", "Teradata");
		addWord("BYTE", "Teradata");
		addWord("BYTEINT", "Teradata");
		addWord("BYTES", "Teradata");
		addWord("CALL", "Teradata");
		addWord("CASE", "Teradata");
		addWord("CASE_N", "Teradata");
		addWord("CASESPECIFIC", "Teradata");
		addWord("CAST", "Teradata");
		addWord("CD", "Teradata");
		addWord("CHAR", "Teradata");
		addWord("CHAR_LENGTH", "Teradata");
		addWord("CHAR2HEXINT", "Teradata");
		addWord("CHARACTER", "Teradata");
		addWord("CHARACTER_LENGTH", "Teradata");
		addWord("CHARACTERS", "Teradata");
		addWord("CHARS", "Teradata");
		addWord("CHECK", "Teradata");
		addWord("CHECKPOINT", "Teradata");
		addWord("CLASS", "Teradata");
		addWord("CLOSE", "Teradata");
		addWord("CLUSTER", "Teradata");
		addWord("CM", "Teradata");
		addWord("COALESCE", "Teradata");
		addWord("COLLATION", "Teradata");
		addWord("COLLECT", "Teradata");
		addWord("COLUMN", "Teradata");
		addWord("COMMENT", "Teradata");
		addWord("COMMIT", "Teradata");
		addWord("COMPRESS", "Teradata");
		addWord("CONSTRAINT", "Teradata");
		addWord("CONTINUE", "Teradata");
		addWord("CONVERT_TABLE_HEADER", "Teradata");
		addWord("CORR", "Teradata");
		addWord("COS", "Teradata");
		addWord("COSH", "Teradata");
		addWord("COUNT", "Teradata");
		addWord("COVAR_POP", "Teradata");
		addWord("COVAR_SAMP", "Teradata");
		addWord("CREATE", "Teradata");
		addWord("CROSS", "Teradata");
		addWord("CS", "Teradata");
		addWord("CSUM", "Teradata");
		addWord("CT", "Teradata");
		addWord("CURRENT", "Teradata");
		addWord("CURRENT_DATE", "Teradata");
		addWord("CURRENT_TIME", "Teradata");
		addWord("CURRENT_TIMESTAMP", "Teradata");
		addWord("CURSOR", "Teradata");
		addWord("CV", "Teradata");
		addWord("CYCLE", "Teradata");
		addWord("DATABASE", "Teradata");
		addWord("DATABLOCKSIZE", "Teradata");
		addWord("DATE", "Teradata");
		addWord("DATEFORM", "Teradata");
		addWord("DAY", "Teradata");
		addWord("DEC", "Teradata");
		addWord("DECIMAL", "Teradata");
		addWord("DECLARE", "Teradata");
		addWord("DEFAULT", "Teradata");
		addWord("DEGREES", "Teradata");
		addWord("DEL", "Teradata");
		addWord("DELETE", "Teradata");
		addWord("DESC", "Teradata");
		addWord("DESCRIPTOR", "Teradata");
		addWord("DETERMINISTIC", "Teradata");
		addWord("DIAGNOSTIC", "Teradata");
		addWord("DISABLED", "Teradata");
		addWord("DISTINCT", "Teradata");
		addWord("DO", "Teradata");
		addWord("DOUBLE", "Teradata");
		addWord("DROP", "Teradata");
		addWord("DUAL", "Teradata");
		addWord("DUMP", "Teradata");
		addWord("EACH", "Teradata");
		addWord("ECHO", "Teradata");
		addWord("ELSE", "Teradata");
		addWord("ELSEIF", "Teradata");
		addWord("ENABLED", "Teradata");
		addWord("END", "Teradata");
		addWord("EQ", "Teradata");
		addWord("ERROR", "Teradata");
		addWord("ERRORFILES", "Teradata");
		addWord("ERRORTABLES", "Teradata");
		addWord("ESCAPE", "Teradata");
		addWord("ET", "Teradata");
		addWord("EXCEPT", "Teradata");
		addWord("EXEC", "Teradata");
		addWord("EXECUTE", "Teradata");
		addWord("EXISTS", "Teradata");
		addWord("EXIT", "Teradata");
		addWord("EXP", "Teradata");
		addWord("EXPLAIN", "Teradata");
		addWord("EXTERNAL", "Teradata");
		addWord("EXTRACT", "Teradata");
		addWord("FALLBACK", "Teradata");
		addWord("FASTEXPORT", "Teradata");
		addWord("FETCH", "Teradata");
		addWord("FIRST", "Teradata");
		addWord("FLOAT", "Teradata");
		addWord("FOR", "Teradata");
		addWord("FOREIGN", "Teradata");
		addWord("FORMAT", "Teradata");
		addWord("FOUND", "Teradata");
		addWord("FREESPACE", "Teradata");
		addWord("FROM", "Teradata");
		addWord("FULL", "Teradata");
		addWord("FUNCTION", "Teradata");
		addWord("GE", "Teradata");
		addWord("GENERATED", "Teradata");
		addWord("GIVE", "Teradata");
		addWord("GO", "Teradata");
		addWord("GOTO", "Teradata");
		addWord("GRANT", "Teradata");
		addWord("GRAPHIC", "Teradata");
		addWord("GROUP", "Teradata");
		addWord("GT", "Teradata");
		addWord("HANDLER", "Teradata");
		addWord("HASH", "Teradata");
		addWord("HASHAMP", "Teradata");
		addWord("HASHBAKAMP", "Teradata");
		addWord("HASHBUCKET", "Teradata");
		addWord("HASHROW", "Teradata");
		addWord("HAVING", "Teradata");
		addWord("HELP", "Teradata");
		addWord("HOUR", "Teradata");
		addWord("IDENTITY", "Teradata");
		addWord("IF", "Teradata");
		addWord("IMMEDIATE", "Teradata");
		addWord("IN", "Teradata");
		addWord("INCONSISTENT", "Teradata");
		addWord("INDEX", "Teradata");
		addWord("INDICATOR", "Teradata");
		addWord("INITIATE", "Teradata");
		addWord("INNER", "Teradata");
		addWord("INOUT", "Teradata");
		addWord("INPUT", "Teradata");
		addWord("INS", "Teradata");
		addWord("INSERT", "Teradata");
		addWord("INSTEAD", "Teradata");
		addWord("INT", "Teradata");
		addWord("INTEGER", "Teradata");
		addWord("INTEGERDATE", "Teradata");
		addWord("INTERSECT", "Teradata");
		addWord("INTERVAL", "Teradata");
		addWord("INTO", "Teradata");
		addWord("IS", "Teradata");
		addWord("ITERATE", "Teradata");
		addWord("JOIN", "Teradata");
		addWord("JOURNAL", "Teradata");
		addWord("KEY", "Teradata");
		addWord("KURTOSIS", "Teradata");
		addWord("LANGUAGE", "Teradata");
		addWord("LE", "Teradata");
		addWord("LEADING", "Teradata");
		addWord("LEAVE", "Teradata");
		addWord("LEFT", "Teradata");
		addWord("LIKE", "Teradata");
		addWord("LIMIT", "Teradata");
		addWord("LN", "Teradata");
		addWord("LOADING", "Teradata");
		addWord("LOCAL", "Teradata");
		addWord("LOCATOR", "Teradata");
		addWord("LOCK", "Teradata");
		addWord("LOCKING", "Teradata");
		addWord("LOG", "Teradata");
		addWord("LOGGING", "Teradata");
		addWord("LOGON", "Teradata");
		addWord("LONG", "Teradata");
		addWord("LOOP", "Teradata");
		addWord("LOWER", "Teradata");
		addWord("LT", "Teradata");
		addWord("MACRO", "Teradata");
		addWord("MAVG", "Teradata");
		addWord("MAX", "Teradata");
		addWord("MAXIMUM", "Teradata");
		addWord("MCHARACTERS", "Teradata");
		addWord("MDIFF", "Teradata");
		addWord("MERGE", "Teradata");
		addWord("MIN", "Teradata");
		addWord("MINDEX", "Teradata");
		addWord("MINIMUM", "Teradata");
		addWord("MINUS", "Teradata");
		addWord("MINUTE", "Teradata");
		addWord("MLINREG", "Teradata");
		addWord("MLOAD", "Teradata");
		addWord("MOD", "Teradata");
		addWord("MODE", "Teradata");
		addWord("MODIFY", "Teradata");
		addWord("MONITOR", "Teradata");
		addWord("MONRESOURCE", "Teradata");
		addWord("MONSESSION", "Teradata");
		addWord("MONTH", "Teradata");
		addWord("MSUBSTR", "Teradata");
		addWord("MSUM", "Teradata");
		addWord("MULTISET", "Teradata");
		addWord("NAMED", "Teradata");
		addWord("NATURAL", "Teradata");
		addWord("NE", "Teradata");
		addWord("NEW", "Teradata");
		addWord("NEW_TABLE", "Teradata");
		addWord("NEXT", "Teradata");
		addWord("NO", "Teradata");
		addWord("NONE", "Teradata");
		addWord("NOT", "Teradata");
		addWord("NOWAIT", "Teradata");
		addWord("NULL", "Teradata");
		addWord("NULLIF", "Teradata");
		addWord("NULLIFZERO", "Teradata");
		addWord("NUMERIC", "Teradata");
		addWord("OBJECTS", "Teradata");
		addWord("OCTET_LENGTH", "Teradata");
		addWord("OF", "Teradata");
		addWord("OFF", "Teradata");
		addWord("OLD", "Teradata");
		addWord("OLD_TABLE", "Teradata");
		addWord("ON", "Teradata");
		addWord("ONLY", "Teradata");
		addWord("OPEN", "Teradata");
		addWord("OPTION", "Teradata");
		addWord("OR", "Teradata");
		addWord("ORDER", "Teradata");
		addWord("OUT", "Teradata");
		addWord("OUTER", "Teradata");
		addWord("OVER", "Teradata");
		addWord("OVERLAPS", "Teradata");
		addWord("OVERRIDE", "Teradata");
		addWord("PARAMETER", "Teradata");
		addWord("PASSWORD", "Teradata");
		addWord("PERCENT", "Teradata");
		addWord("PERCENT_RANK", "Teradata");
		addWord("PERM", "Teradata");
		addWord("PERMANENT", "Teradata");
		addWord("POSITION", "Teradata");
		addWord("PRECISION", "Teradata");
		addWord("PREPARE", "Teradata");
		addWord("PRESERVE", "Teradata");
		addWord("PRIMARY", "Teradata");
		addWord("PRIVATE", "Teradata");
		addWord("PRIVILEGES", "Teradata");
		addWord("PROCEDURE", "Teradata");
		addWord("PROFILE", "Teradata");
		addWord("PROPORTIONAL", "Teradata");
		addWord("PROTECTION", "Teradata");
		addWord("PUBLIC", "Teradata");
		addWord("QUALIFIED", "Teradata");
		addWord("QUALIFY", "Teradata");
		addWord("QUANTILE", "Teradata");
		addWord("RADIANS", "Teradata");
		addWord("RANDOM", "Teradata");
		addWord("RANGE_N", "Teradata");
		addWord("RANK", "Teradata");
		addWord("REAL", "Teradata");
		addWord("REFERENCES", "Teradata");
		addWord("REFERENCING", "Teradata");
		addWord("REGR_AVGX", "Teradata");
		addWord("REGR_AVGY", "Teradata");
		addWord("REGR_COUNT", "Teradata");
		addWord("REGR_INTERCEPT", "Teradata");
		addWord("REGR_R2", "Teradata");
		addWord("REGR_SLOPE", "Teradata");
		addWord("REGR_SXX", "Teradata");
		addWord("REGR_SXY", "Teradata");
		addWord("REGR_SYY", "Teradata");
		addWord("RELEASE", "Teradata");
		addWord("RENAME", "Teradata");
		addWord("REPEAT", "Teradata");
		addWord("REPLACE", "Teradata");
		addWord("REPLICATION", "Teradata");
		addWord("REPOVERRIDE", "Teradata");
		addWord("REQUEST", "Teradata");
		addWord("RESTART", "Teradata");
		addWord("RESTORE", "Teradata");
		addWord("RESUME", "Teradata");
		addWord("RET", "Teradata");
		addWord("RETRIEVE", "Teradata");
		addWord("RETURNS", "Teradata");
		addWord("REVALIDATE", "Teradata");
		addWord("REVOKE", "Teradata");
		addWord("RIGHT", "Teradata");
		addWord("RIGHTS", "Teradata");
		addWord("ROLE", "Teradata");
		addWord("ROLLBACK", "Teradata");
		addWord("ROLLFORWARD", "Teradata");
		addWord("ROW", "Teradata");
		addWord("ROW_NUMBER", "Teradata");
		addWord("ROWID", "Teradata");
		addWord("ROWS", "Teradata");
		addWord("SAMPLE", "Teradata");
		addWord("SAMPLEID", "Teradata");
		addWord("SCROLL", "Teradata");
		addWord("SECOND", "Teradata");
		addWord("SEL", "Teradata");
		addWord("SELECT", "Teradata");
		addWord("SESSION", "Teradata");
		addWord("SET", "Teradata");
		addWord("SETRESRATE", "Teradata");
		addWord("SETSESSRATE", "Teradata");
		addWord("SHOW", "Teradata");
		addWord("SIMPLE", "Teradata");
		addWord("SIN", "Teradata");
		addWord("SINH", "Teradata");
		addWord("SKEW", "Teradata");
		addWord("SMALLINT", "Teradata");
		addWord("SOME", "Teradata");
		addWord("SOUNDEX", "Teradata");
		addWord("SPECIFIC", "Teradata");
		addWord("SPOOL", "Teradata");
		addWord("SQL", "Teradata");
		addWord("SQLEXCEPTION", "Teradata");
		addWord("SQLTEXT", "Teradata");
		addWord("SQLWARNING", "Teradata");
		addWord("SQRT", "Teradata");
		addWord("SS", "Teradata");
		addWord("START", "Teradata");
		addWord("STARTUP", "Teradata");
	}

	static {
		// =============HANA=============================
		// From
		// http://help-legacy.sap.com/saphelp_hanaplatform/helpdata/en/28/bcd6af3eb6437892719f7c27a8a285/content.htm
		addWord("ALL", "Hana");
		addWord("ALTER", "Hana");
		addWord("AS", "Hana");
		addWord("BEFORE", "Hana");
		addWord("BEGIN", "Hana");
		addWord("BOTH", "Hana");
		addWord("CASE", "Hana");
		addWord("CHAR", "Hana");
		addWord("CONDITION", "Hana");
		addWord("CONNECT", "Hana");
		addWord("CROSS", "Hana");
		addWord("CUBE", "Hana");
		addWord("CURRENT_CONNECTION", "Hana");
		addWord("CURRENT_DATE", "Hana");
		addWord("CURRENT_SCHEMA", "Hana");
		addWord("CURRENT_TIME", "Hana");
		addWord("CURRENT_TIMESTAMP", "Hana");
		addWord("CURRENT_TRANSACTION_ISOLATION_LEVEL", "Hana");
		addWord("CURRENT_USER", "Hana");
		addWord("CURRENT_UTCDATE", "Hana");
		addWord("CURRENT_UTCTIME", "Hana");
		addWord("CURRENT_UTCTIMESTAMP", "Hana");
		addWord("CURRVAL", "Hana");
		addWord("CURSOR", "Hana");
		addWord("DECLARE", "Hana");
		addWord("DISTINCT", "Hana");
		addWord("ELSE", "Hana");
		addWord("ELSEIF", "Hana");
		addWord("END", "Hana");
		addWord("EXCEPT", "Hana");
		addWord("EXCEPTION", "Hana");
		addWord("EXEC", "Hana");
		addWord("FALSE", "Hana");
		addWord("FOR", "Hana");
		addWord("FROM", "Hana");
		addWord("FULL", "Hana");
		addWord("GROUP", "Hana");
		addWord("HAVING", "Hana");
		addWord("IF", "Hana");
		addWord("IN", "Hana");
		addWord("INNER", "Hana");
		addWord("INOUT", "Hana");
		addWord("INTERSECT", "Hana");
		addWord("INTO", "Hana");
		addWord("IS", "Hana");
		addWord("JOIN", "Hana");
		addWord("LEADING", "Hana");
		addWord("LEFT", "Hana");
		addWord("LIMIT", "Hana");
		addWord("LOOP", "Hana");
		addWord("MINUS", "Hana");
		addWord("NATURAL", "Hana");
		addWord("NCHAR", "Hana");
		addWord("NEXTVAL", "Hana");
		addWord("NULL", "Hana");
		addWord("ON", "Hana");
		addWord("ORDER", "Hana");
		addWord("OUT", "Hana");
		addWord("PRIOR", "Hana");
		addWord("RETURN", "Hana");
		addWord("RETURNS", "Hana");
		addWord("REVERSE", "Hana");
		addWord("RIGHT", "Hana");
		addWord("ROLLUP", "Hana");
		addWord("ROWID", "Hana");
		addWord("SELECT", "Hana");
		addWord("SESSION_USER", "Hana");
		addWord("SET", "Hana");
		addWord("SQL", "Hana");
		addWord("START", "Hana");
		addWord("SYSUUID", "Hana");
		addWord("TABLESAMPLE", "Hana");
		addWord("TOP", "Hana");
		addWord("TRAILING", "Hana");
		addWord("TRUE", "Hana");
		addWord("UNION", "Hana");
		addWord("UNKNOWN", "Hana");
		addWord("USING", "Hana");
		addWord("UTCTIMESTAMP", "Hana");
		addWord("VALUES", "Hana");
		addWord("WHEN", "Hana");
		addWord("WHERE", "Hana");
		addWord("WHILE", "Hana");
		addWord("WITH", "Hana");
	}

	static {
		// =============Maria=============================
		// From
		// https://mariadb.com/kb/en/mariadb/reserved-words/
		addWord("ACCESSIBLE", "Maria");
		addWord("ADD", "Maria");
		addWord("ALL", "Maria");
		addWord("ALTER", "Maria");
		addWord("ANALYZE", "Maria");
		addWord("AND", "Maria");
		addWord("AS", "Maria");
		addWord("ASC", "Maria");
		addWord("ASENSITIVE", "Maria");
		addWord("BEFORE", "Maria");
		addWord("BETWEEN", "Maria");
		addWord("BIGINT", "Maria");
		addWord("BINARY", "Maria");
		addWord("BLOB", "Maria");
		addWord("BOTH", "Maria");
		addWord("BY", "Maria");
		addWord("CALL", "Maria");
		addWord("CASCADE", "Maria");
		addWord("CASE", "Maria");
		addWord("CHANGE", "Maria");
		addWord("CHAR", "Maria");
		addWord("CHARACTER", "Maria");
		addWord("CHECK", "Maria");
		addWord("COLLATE", "Maria");
		addWord("COLUMN", "Maria");
		addWord("CONDITION", "Maria");
		addWord("CONSTRAINT", "Maria");
		addWord("CONTINUE", "Maria");
		addWord("CONVERT", "Maria");
		addWord("CREATE", "Maria");
		addWord("CROSS", "Maria");
		addWord("CURRENT_DATE", "Maria");
		addWord("CURRENT_TIME", "Maria");
		addWord("CURRENT_TIMESTAMP", "Maria");
		addWord("CURRENT_USER", "Maria");
		addWord("CURSOR", "Maria");
		addWord("DATABASE", "Maria");
		addWord("DATABASES", "Maria");
		addWord("DAY_HOUR", "Maria");
		addWord("DAY_MICROSECOND", "Maria");
		addWord("DAY_MINUTE", "Maria");
		addWord("DAY_SECOND", "Maria");
		addWord("DEC", "Maria");
		addWord("DECIMAL", "Maria");
		addWord("DECLARE", "Maria");
		addWord("DEFAULT", "Maria");
		addWord("DELAYED", "Maria");
		addWord("DELETE", "Maria");
		addWord("DESC", "Maria");
		addWord("DESCRIBE", "Maria");
		addWord("DETERMINISTIC", "Maria");
		addWord("DISTINCT", "Maria");
		addWord("DISTINCTROW", "Maria");
		addWord("DIV", "Maria");
		addWord("DOUBLE", "Maria");
		addWord("DROP", "Maria");
		addWord("DUAL", "Maria");
		addWord("EACH", "Maria");
		addWord("ELSE", "Maria");
		addWord("ELSEIF", "Maria");
		addWord("ENCLOSED", "Maria");
		addWord("ESCAPED", "Maria");
		addWord("EXISTS", "Maria");
		addWord("EXIT", "Maria");
		addWord("EXPLAIN", "Maria");
		addWord("FETCH", "Maria");
		addWord("FLOAT", "Maria");
		addWord("FLOAT4", "Maria");
		addWord("FLOAT8", "Maria");
		addWord("FOR", "Maria");
		addWord("FORCE", "Maria");
		addWord("FOREIGN", "Maria");
		addWord("FROM", "Maria");
		addWord("FULLTEXT", "Maria");
		addWord("GENERAL", "Maria");
		addWord("GRANT", "Maria");
		addWord("GROUP", "Maria");
		addWord("HAVING", "Maria");
		addWord("HIGH_PRIORITY", "Maria");
		addWord("HOUR_MICROSECOND", "Maria");
		addWord("HOUR_MINUTE", "Maria");
		addWord("HOUR_SECOND", "Maria");
		addWord("IF", "Maria");
		addWord("IGNORE", "Maria");
		addWord("IGNORE_SERVER_IDS", "Maria");
		addWord("IN", "Maria");
		addWord("INDEX", "Maria");
		addWord("INFILE", "Maria");
		addWord("INNER", "Maria");
		addWord("INOUT", "Maria");
		addWord("INSENSITIVE", "Maria");
		addWord("INSERT", "Maria");
		addWord("INT", "Maria");
		addWord("INT1", "Maria");
		addWord("INT2", "Maria");
		addWord("INT3", "Maria");
		addWord("INT4", "Maria");
		addWord("INT8", "Maria");
		addWord("INTEGER", "Maria");
		addWord("INTERVAL", "Maria");
		addWord("INTO", "Maria");
		addWord("IS", "Maria");
		addWord("ITERATE", "Maria");
		addWord("JOIN", "Maria");
		addWord("KEY", "Maria");
		addWord("KEYS", "Maria");
		addWord("KILL", "Maria");
		addWord("LEADING", "Maria");
		addWord("LEAVE", "Maria");
		addWord("LEFT", "Maria");
		addWord("LIKE", "Maria");
		addWord("LIMIT", "Maria");
		addWord("LINEAR", "Maria");
		addWord("LINES", "Maria");
		addWord("LOAD", "Maria");
		addWord("LOCALTIME", "Maria");
		addWord("LOCALTIMESTAMP", "Maria");
		addWord("LOCK", "Maria");
		addWord("LONG", "Maria");
		addWord("LONGBLOB", "Maria");
		addWord("LONGTEXT", "Maria");
		addWord("LOOP", "Maria");
		addWord("LOW_PRIORITY", "Maria");
		addWord("MASTER_HEARTBEAT_PERIOD", "Maria");
		addWord("MASTER_SSL_VERIFY_SERVER_CERT", "Maria");
		addWord("MATCH", "Maria");
		addWord("MAXVALUE", "Maria");
		addWord("MEDIUMBLOB", "Maria");
		addWord("MEDIUMINT", "Maria");
		addWord("MEDIUMTEXT", "Maria");
		addWord("MIDDLEINT", "Maria");
		addWord("MINUTE_MICROSECOND", "Maria");
		addWord("MINUTE_SECOND", "Maria");
		addWord("MOD", "Maria");
		addWord("MODIFIES", "Maria");
		addWord("NATURAL", "Maria");
		addWord("NO_WRITE_TO_BINLOG", "Maria");
		addWord("NOT", "Maria");
		addWord("NULL", "Maria");
		addWord("NUMERIC", "Maria");
		addWord("ON", "Maria");
		addWord("OPTIMIZE", "Maria");
		addWord("OPTION", "Maria");
		addWord("OPTIONALLY", "Maria");
		addWord("OR", "Maria");
		addWord("ORDER", "Maria");
		addWord("OUT", "Maria");
		addWord("OUTER", "Maria");
		addWord("OUTFILE", "Maria");
		addWord("PARTITION", "Maria");
		addWord("PRECISION", "Maria");
		addWord("PRIMARY", "Maria");
		addWord("PROCEDURE", "Maria");
		addWord("PURGE", "Maria");
		addWord("RANGE", "Maria");
		addWord("READ", "Maria");
		addWord("READ_WRITE", "Maria");
		addWord("READS", "Maria");
		addWord("REAL", "Maria");
		addWord("REFERENCES", "Maria");
		addWord("REGEXP", "Maria");
		addWord("RELEASE", "Maria");
		addWord("RENAME", "Maria");
		addWord("REPEAT", "Maria");
		addWord("REPLACE", "Maria");
		addWord("REQUIRE", "Maria");
		addWord("RESIGNAL", "Maria");
		addWord("RESTRICT", "Maria");
		addWord("RETURN", "Maria");
		addWord("REVOKE", "Maria");
		addWord("RIGHT", "Maria");
		addWord("RLIKE", "Maria");
		addWord("SCHEMA", "Maria");
		addWord("SCHEMAS", "Maria");
		addWord("SECOND_MICROSECOND", "Maria");
		addWord("SELECT", "Maria");
		addWord("SENSITIVE", "Maria");
		addWord("SEPARATOR", "Maria");
		addWord("SET", "Maria");
		addWord("SHOW", "Maria");
		addWord("SIGNAL", "Maria");
		addWord("SLOW", "Maria");
		addWord("SMALLINT", "Maria");
		addWord("SPATIAL", "Maria");
		addWord("SPECIFIC", "Maria");
		addWord("SQL", "Maria");
		addWord("SQL_BIG_RESULT", "Maria");
		addWord("SQL_CALC_FOUND_ROWS", "Maria");
		addWord("SQL_SMALL_RESULT", "Maria");
		addWord("SQLEXCEPTION", "Maria");
		addWord("SQLSTATE", "Maria");
		addWord("SQLWARNING", "Maria");
		addWord("SSL", "Maria");
		addWord("STARTING", "Maria");
		addWord("STRAIGHT_JOIN", "Maria");
		addWord("TABLE", "Maria");
		addWord("TERMINATED", "Maria");
		addWord("THEN", "Maria");
		addWord("TINYBLOB", "Maria");
		addWord("TINYINT", "Maria");
		addWord("TINYTEXT", "Maria");
		addWord("TO", "Maria");
		addWord("TRAILING", "Maria");
		addWord("TRIGGER", "Maria");
		addWord("UNDO", "Maria");
		addWord("UNION", "Maria");
		addWord("UNIQUE", "Maria");
		addWord("UNLOCK", "Maria");
		addWord("UNSIGNED", "Maria");
		addWord("UPDATE", "Maria");
		addWord("USAGE", "Maria");
		addWord("USE", "Maria");
		addWord("USING", "Maria");
		addWord("UTC_DATE", "Maria");
		addWord("UTC_TIME", "Maria");
		addWord("UTC_TIMESTAMP", "Maria");
		addWord("VALUES", "Maria");
		addWord("VARBINARY", "Maria");
		addWord("VARCHAR", "Maria");
		addWord("VARCHARACTER", "Maria");
		addWord("VARYING", "Maria");
		addWord("WHEN", "Maria");
		addWord("WHERE", "Maria");
		addWord("WHILE", "Maria");
		addWord("WITH", "Maria");
		addWord("WRITE", "Maria");
		addWord("XOR", "Maria");
		addWord("YEAR_MONTH", "Maria");
		addWord("ZEROFILL", "Maria");
		addWord("FALSE", "Maria");
		addWord("TRUE", "Maria");
	}

	static {
		// =============H2=============================
		// From
		// http://www.h2database.com/html/advanced.html
		addWord("CROSS", "H2");
		addWord("CURRENT_DATE", "H2");
		addWord("CURRENT_TIME", "H2");
		addWord("CURRENT_TIMESTAMP", "H2");
		addWord("DISTINCT", "H2");
		addWord("EXCEPT", "H2");
		addWord("EXISTS", "H2");
		addWord("FALSE", "H2");
		addWord("FETCH", "H2");
		addWord("FOR", "H2");
		addWord("FROM", "H2");
		addWord("FULL", "H2");
		addWord("GROUP", "H2");
		addWord("HAVING", "H2");
		addWord("INNER", "H2");
		addWord("INTERSECT", "H2");
		addWord("IS", "H2");
		addWord("JOIN", "H2");
		addWord("LIKE", "H2");
		addWord("LIMIT", "H2");
		addWord("MINUS", "H2");
		addWord("NATURAL", "H2");
		addWord("NOT", "H2");
		addWord("NULL", "H2");
		addWord("OFFSET", "H2");
		addWord("ON", "H2");
		addWord("ORDER", "H2");
		addWord("PRIMARY", "H2");
		addWord("ROWNUM", "H2");
		addWord("SELECT", "H2");
		addWord("SYSDATE", "H2");
		addWord("SYSTIME", "H2");
		addWord("SYSTIMESTAMP", "H2");
		addWord("TODAY", "H2");
		addWord("TRUE", "H2");
		addWord("UNION", "H2");
		addWord("UNIQUE", "H2");
		addWord("WHERE", "H2");
	}

	static {
		// =============Sybase=============================
		// From
		// http://infocenter.sybase.com/help/index.jsp?topic=/com.sybase.dc36271_1251/html/blocks/X10189.htm
		addWord("add", "Sybase");
		addWord("all", "Sybase");
		addWord("alter", "Sybase");
		addWord("and", "Sybase");
		addWord("any", "Sybase");
		addWord("arith_overflow", "Sybase");
		addWord("as", "Sybase");
		addWord("asc", "Sybase");
		addWord("at", "Sybase");
		addWord("authorization", "Sybase");
		addWord("avg", "Sybase");
		addWord("begin", "Sybase");
		addWord("between", "Sybase");
		addWord("break", "Sybase");
		addWord("browse", "Sybase");
		addWord("bulk", "Sybase");
		addWord("by", "Sybase");
		addWord("cascade", "Sybase");
		addWord("case", "Sybase");
		addWord("char_convert", "Sybase");
		addWord("check", "Sybase");
		addWord("checkpoint", "Sybase");
		addWord("close", "Sybase");
		addWord("clustered", "Sybase");
		addWord("coalesce", "Sybase");
		addWord("commit", "Sybase");
		addWord("compute", "Sybase");
		addWord("confirm", "Sybase");
		addWord("connect", "Sybase");
		addWord("constraint", "Sybase");
		addWord("continue", "Sybase");
		addWord("controlrow", "Sybase");
		addWord("convert", "Sybase");
		addWord("count", "Sybase");
		addWord("create", "Sybase");
		addWord("current", "Sybase");
		addWord("cursor", "Sybase");
		addWord("database", "Sybase");
		addWord("dbcc", "Sybase");
		addWord("deallocate", "Sybase");
		addWord("declare", "Sybase");
		addWord("default", "Sybase");
		addWord("delete", "Sybase");
		addWord("desc", "Sybase");
		addWord("deterministic", "Sybase");
		addWord("diskdistinct", "Sybase");
		addWord("double", "Sybase");
		addWord("drop", "Sybase");
		addWord("dummy", "Sybase");
		addWord("dump", "Sybase");
		addWord("else", "Sybase");
		addWord("end", "Sybase");
		addWord("endtran", "Sybase");
		addWord("errlvl", "Sybase");
		addWord("errordata", "Sybase");
		addWord("errorexit", "Sybase");
		addWord("escape", "Sybase");
		addWord("except", "Sybase");
		addWord("exclusive", "Sybase");
		addWord("exec", "Sybase");
		addWord("execute", "Sybase");
		addWord("exists", "Sybase");
		addWord("exit", "Sybase");
		addWord("exp_row_size", "Sybase");
		addWord("external", "Sybase");
		addWord("fetch", "Sybase");
		addWord("fillfactor", "Sybase");
		addWord("for", "Sybase");
		addWord("foreign", "Sybase");
		addWord("from", "Sybase");
		addWord("func", "Sybase");
		addWord("goto", "Sybase");
		addWord("grant", "Sybase");
		addWord("group", "Sybase");
		addWord("having", "Sybase");
		addWord("holdlock", "Sybase");
		addWord("identity", "Sybase");
		addWord("identity_gap", "Sybase");
		addWord("identity_insert", "Sybase");
		addWord("identity_start", "Sybase");
		addWord("if", "Sybase");
		addWord("in", "Sybase");
		addWord("index", "Sybase");
		addWord("inout", "Sybase");
		addWord("insert", "Sybase");
		addWord("install", "Sybase");
		addWord("intersect", "Sybase");
		addWord("into", "Sybase");
		addWord("is", "Sybase");
		addWord("isolation", "Sybase");
		addWord("jar", "Sybase");
		addWord("join", "Sybase");
		addWord("key", "Sybase");
		addWord("kill", "Sybase");
		addWord("level", "Sybase");
		addWord("like", "Sybase");
		addWord("lineno", "Sybase");
		addWord("load", "Sybase");
		addWord("lock", "Sybase");
		addWord("max", "Sybase");
		addWord("max_rows_per_page", "Sybase");
		addWord("min", "Sybase");
		addWord("mirror", "Sybase");
		addWord("mirrorexit", "Sybase");
		addWord("modify", "Sybase");
		addWord("national", "Sybase");
		addWord("new", "Sybase");
		addWord("noholdlock", "Sybase");
		addWord("nonclustered", "Sybase");
		addWord("not", "Sybase");
		addWord("null", "Sybase");
		addWord("nullif", "Sybase");
		addWord("numeric_truncation", "Sybase");
		addWord("of", "Sybase");
		addWord("off", "Sybase");
		addWord("offsets", "Sybase");
		addWord("on", "Sybase");
		addWord("once", "Sybase");
		addWord("online", "Sybase");
		addWord("only", "Sybase");
		addWord("open", "Sybase");
		addWord("option", "Sybase");
		addWord("or", "Sybase");
		addWord("order", "Sybase");
		addWord("out", "Sybase");
		addWord("output", "Sybase");
		addWord("over", "Sybase");
		addWord("partition", "Sybase");
		addWord("perm", "Sybase");
		addWord("permanent", "Sybase");
		addWord("plan", "Sybase");
		addWord("precision", "Sybase");
		addWord("prepare", "Sybase");
		addWord("primary", "Sybase");
		addWord("print", "Sybase");
		addWord("privileges", "Sybase");
		addWord("proc", "Sybase");
		addWord("procedure", "Sybase");
		addWord("processexit", "Sybase");
		addWord("proxy_table", "Sybase");
		addWord("public", "Sybase");
		addWord("quiesce", "Sybase");
		addWord("raiserror", "Sybase");
		addWord("read", "Sybase");
		addWord("readpast", "Sybase");
		addWord("readtext", "Sybase");
		addWord("reconfigure", "Sybase");
		addWord("referencesremove", "Sybase");
		addWord("reorg", "Sybase");
		addWord("replace", "Sybase");
		addWord("replication", "Sybase");
		addWord("reservepagegap", "Sybase");
		addWord("return", "Sybase");
		addWord("returns", "Sybase");
		addWord("revoke", "Sybase");
		addWord("role", "Sybase");
		addWord("rollback", "Sybase");
		addWord("rowcount", "Sybase");
		addWord("rows", "Sybase");
		addWord("rule", "Sybase");
		addWord("save", "Sybase");
		addWord("schema", "Sybase");
		addWord("select", "Sybase");
		addWord("set", "Sybase");
		addWord("setuser", "Sybase");
		addWord("shared", "Sybase");
		addWord("shutdown", "Sybase");
		addWord("some", "Sybase");
		addWord("statistics", "Sybase");
		addWord("stringsize", "Sybase");
		addWord("stripe", "Sybase");
		addWord("sum", "Sybase");
		addWord("syb_identity", "Sybase");
		addWord("syb_restree", "Sybase");
		addWord("syb_terminate", "Sybase");
		addWord("table", "Sybase");
		addWord("temp", "Sybase");
		addWord("temporary", "Sybase");
		addWord("textsize", "Sybase");
		addWord("to", "Sybase");
		addWord("tran", "Sybase");
		addWord("transaction", "Sybase");
		addWord("trigger", "Sybase");
		addWord("truncate", "Sybase");
		addWord("tsequal", "Sybase");
		addWord("union", "Sybase");
		addWord("unique", "Sybase");
		addWord("unpartition", "Sybase");
		addWord("update", "Sybase");
		addWord("use", "Sybase");
		addWord("user", "Sybase");
		addWord("user_option", "Sybase");
		addWord("using", "Sybase");
		addWord("values", "Sybase");
		addWord("varying", "Sybase");
		addWord("view", "Sybase");
		addWord("waitfor", "Sybase");
		addWord("when", "Sybase");
		addWord("where", "Sybase");
		addWord("while", "Sybase");
		addWord("with", "Sybase");
		addWord("work", "Sybase");
		addWord("writetext", "Sybase");
	}

	static {
		// =============FireBird=============================
		// From
		// https://firebirdsql.org/refdocs/langrefupd25-reskeywords-full-reswords.html
		addWord("ADD", "Firebird");
		addWord("ADMIN", "Firebird");
		addWord("ALL", "Firebird");
		addWord("ALTER", "Firebird");
		addWord("AND", "Firebird");
		addWord("ANY", "Firebird");
		addWord("AS", "Firebird");
		addWord("AT", "Firebird");
		addWord("AVG", "Firebird");
		addWord("BEGIN", "Firebird");
		addWord("BETWEEN", "Firebird");
		addWord("BIGINT", "Firebird");
		addWord("BIT_LENGTH", "Firebird");
		addWord("BLOB", "Firebird");
		addWord("BOTH", "Firebird");
		addWord("BY", "Firebird");
		addWord("CASE", "Firebird");
		addWord("CAST", "Firebird");
		addWord("CHAR", "Firebird");
		addWord("CHAR_LENGTH", "Firebird");
		addWord("CHARACTER", "Firebird");
		addWord("CHARACTER_LENGTH", "Firebird");
		addWord("CHECK", "Firebird");
		addWord("CLOSE", "Firebird");
		addWord("COLLATE", "Firebird");
		addWord("COLUMN", "Firebird");
		addWord("COMMIT", "Firebird");
		addWord("CONNECT", "Firebird");
		addWord("CONSTRAINT", "Firebird");
		addWord("COUNT", "Firebird");
		addWord("CREATE", "Firebird");
		addWord("CROSS", "Firebird");
		addWord("CURRENT", "Firebird");
		addWord("CURRENT_CONNECTION", "Firebird");
		addWord("CURRENT_DATE", "Firebird");
		addWord("CURRENT_ROLE", "Firebird");
		addWord("CURRENT_TIME", "Firebird");
		addWord("CURRENT_TIMESTAMP", "Firebird");
		addWord("CURRENT_TRANSACTION", "Firebird");
		addWord("CURRENT_USER", "Firebird");
		addWord("CURSOR", "Firebird");
		addWord("DATE", "Firebird");
		addWord("DAY", "Firebird");
		addWord("DEC", "Firebird");
		addWord("DECIMAL", "Firebird");
		addWord("DECLARE", "Firebird");
		addWord("DEFAULT", "Firebird");
		addWord("DELETE", "Firebird");
		addWord("DISCONNECT", "Firebird");
		addWord("DISTINCT", "Firebird");
		addWord("DOUBLE", "Firebird");
		addWord("DROP", "Firebird");
		addWord("ELSE", "Firebird");
		addWord("END", "Firebird");
		addWord("ESCAPE", "Firebird");
		addWord("EXECUTE", "Firebird");
		addWord("EXISTS", "Firebird");
		addWord("EXTERNAL", "Firebird");
		addWord("EXTRACT", "Firebird");
		addWord("FETCH", "Firebird");
		addWord("FILTER", "Firebird");
		addWord("FLOAT", "Firebird");
		addWord("FOR", "Firebird");
		addWord("FOREIGN", "Firebird");
		addWord("FROM", "Firebird");
		addWord("FULL", "Firebird");
		addWord("FUNCTION", "Firebird");
		addWord("GDSCODE", "Firebird");
		addWord("GLOBAL", "Firebird");
		addWord("GRANT", "Firebird");
		addWord("GROUP", "Firebird");
		addWord("HAVING", "Firebird");
		addWord("HOUR", "Firebird");
		addWord("IN", "Firebird");
		addWord("INDEX", "Firebird");
		addWord("INNER", "Firebird");
		addWord("INSENSITIVE", "Firebird");
		addWord("INSERT", "Firebird");
		addWord("INT", "Firebird");
		addWord("INTEGER", "Firebird");
		addWord("INTO", "Firebird");
		addWord("IS", "Firebird");
		addWord("JOIN", "Firebird");
		addWord("LEADING", "Firebird");
		addWord("LEFT", "Firebird");
		addWord("LIKE", "Firebird");
		addWord("LONG", "Firebird");
		addWord("LOWER", "Firebird");
		addWord("MAX", "Firebird");
		addWord("MAXIMUM_SEGMENT", "Firebird");
		addWord("MERGE", "Firebird");
		addWord("MIN", "Firebird");
		addWord("MINUTE", "Firebird");
		addWord("MONTH", "Firebird");
		addWord("NATIONAL", "Firebird");
		addWord("NATURAL", "Firebird");
		addWord("NCHAR", "Firebird");
		addWord("NO", "Firebird");
		addWord("NOT", "Firebird");
		addWord("NULL", "Firebird");
		addWord("NUMERIC", "Firebird");
		addWord("OCTET_LENGTH", "Firebird");
		addWord("OF", "Firebird");
		addWord("ON", "Firebird");
		addWord("ONLY", "Firebird");
		addWord("OPEN", "Firebird");
		addWord("OR", "Firebird");
		addWord("ORDER", "Firebird");
		addWord("OUTER", "Firebird");
		addWord("PARAMETER", "Firebird");
		addWord("PLAN", "Firebird");
		addWord("POSITION", "Firebird");
		addWord("POST_EVENT", "Firebird");
		addWord("PRECISION", "Firebird");
		addWord("PRIMARY", "Firebird");
		addWord("PROCEDURE", "Firebird");
		addWord("RDB$DB_KEY", "Firebird");
		addWord("REAL", "Firebird");
		addWord("RECORD_VERSION", "Firebird");
		addWord("RECREATE", "Firebird");
		addWord("RECURSIVE", "Firebird");
		addWord("REFERENCES", "Firebird");
		addWord("RELEASE", "Firebird");
		addWord("RETURNING_VALUES", "Firebird");
		addWord("RETURNS", "Firebird");
		addWord("REVOKE", "Firebird");
		addWord("RIGHT", "Firebird");
		addWord("ROLLBACK", "Firebird");
		addWord("ROW_COUNT", "Firebird");
		addWord("ROWS", "Firebird");
		addWord("SAVEPOINT", "Firebird");
		addWord("SECOND", "Firebird");
		addWord("SELECT", "Firebird");
		addWord("SENSITIVE", "Firebird");
		addWord("SET", "Firebird");
		addWord("SIMILAR", "Firebird");
		addWord("SMALLINT", "Firebird");
		addWord("SOME", "Firebird");
		addWord("SQLCODE", "Firebird");
		addWord("SQLSTATE(2.5.1)", "Firebird");
		addWord("START", "Firebird");
		addWord("SUM", "Firebird");
		addWord("TABLE", "Firebird");
		addWord("THEN", "Firebird");
		addWord("TIME", "Firebird");
		addWord("TIMESTAMP", "Firebird");
		addWord("TO", "Firebird");
		addWord("TRAILING", "Firebird");
		addWord("TRIGGER", "Firebird");
		addWord("TRIM", "Firebird");
		addWord("UNION", "Firebird");
		addWord("UNIQUE", "Firebird");
		addWord("UPDATE", "Firebird");
		addWord("UPPER", "Firebird");
		addWord("USER", "Firebird");
		addWord("USING", "Firebird");
		addWord("VALUE", "Firebird");
		addWord("VALUES", "Firebird");
		addWord("VARCHAR", "Firebird");
		addWord("VARIABLE", "Firebird");
		addWord("VARYING", "Firebird");
		addWord("VIEW", "Firebird");
		addWord("WHEN", "Firebird");
		addWord("WHERE", "Firebird");
		addWord("WHILE", "Firebird");
		addWord("WITH", "Firebird");
		addWord("YEAR", "Firebird");
	}

}
