/*
 * Copyright (C) 2013 Alex Andres
 *
 * This file is part of JavaAV.
 *
 * JavaAV is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version (subject to the "Classpath"
 * exception as provided in the LICENSE file that accompanied
 * this code).
 *
 * JavaAV is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JavaAV. If not, see <http://www.gnu.org/licenses/>.
 */

package com.github.hoary.javaav;

import static com.googlecode.javacv.cpp.avcodec.*;
import static com.googlecode.javacv.cpp.avformat.*;
import static com.googlecode.javacv.cpp.avutil.*;

public class Demuxer extends Configurable {

	private MediaPacket mediaPacket;

	private Decoder videoDecoder;
	private Decoder audioDecoder;

	private AVFormatContext formatContext;

	private AVStream videoStream;
	private AVStream audioStream;

	private AVCodecContext videoCodecContext;
	private AVCodecContext audioCodecContext;

	private AVPacket avPacket;

	private String format = null;


	public void open(String inputPath) throws JavaAVException {
		videoDecoder = null;
		audioDecoder = null;

		AVInputFormat inputFormat = null;
		if (format != null && format.length() > 0) {
			inputFormat = av_find_input_format(format);

			if (inputFormat == null)
				throw new JavaAVException("Could not find input format: " + format);
		}

		formatContext = new AVFormatContext(null);

		AVDictionary options = new AVDictionary(null);
		if (frameRate > 0) {
			AVRational r = av_d2q(frameRate, 1001000);
			av_dict_set(options, "framerate", r.num() + "/" + r.den(), 0);
		}
		if (imageWidth > 0 && imageHeight > 0)
			av_dict_set(options, "video_size", imageWidth + "x" + imageHeight, 0);

		if (sampleRate > 0)
			av_dict_set(options, "sample_rate", "" + sampleRate, 0);

		if (audioChannels > 0)
			av_dict_set(options, "channels", "" + audioChannels, 0);

		if (avformat_open_input(formatContext, inputPath, inputFormat, options) < 0)
			throw new JavaAVException("Could not open input: " + inputPath);

		av_dict_free(options);

		// retrieve stream information
		if (avformat_find_stream_info(formatContext, (AVDictionary) null) < 0)
			throw new JavaAVException("Could not find stream information.");

		// find the first video and audio stream
		int streams = formatContext.nb_streams();

		// get a pointer to the codec context for the video or audio stream
		for (int i = 0; i < streams; i++) {
			AVStream stream = formatContext.streams(i);
			AVCodecContext context = stream.codec();

			if (videoStream == null && context.codec_type() == AVMEDIA_TYPE_VIDEO) {
				videoStream = stream;
				videoCodecContext = context;
			}
			else if (audioStream == null && context.codec_type() == AVMEDIA_TYPE_AUDIO) {
				audioStream = stream;
				audioCodecContext = context;
			}
		}

		if (videoStream == null && audioStream == null)
			throw new JavaAVException("Could not find any video or audio stream.");

		// try creating decoder in separate way, since video or audio may be not present
		if (videoCodecContext != null)
			initVideoDecoder(videoCodecContext);

		if (audioCodecContext != null)
			initAudioDecoder(audioCodecContext);

		avPacket = new AVPacket();
	}

	public void close() {
		if (formatContext != null && !formatContext.isNull()) {
			avformat_close_input(formatContext);
			formatContext = null;
		}

		// set explicitly to null since avformat_close_input already released the memory.
		// this way the decoder knows the context is already closed.
		if (videoCodecContext != null)
			videoCodecContext.setNull();

		if (audioCodecContext != null)
			audioCodecContext.setNull();

		if (videoDecoder != null) {
			videoDecoder.close();
			videoDecoder = null;
		}
		if (audioDecoder != null) {
			audioDecoder.close();
			audioDecoder = null;
		}
	}

	public MediaFrame readFrame() throws JavaAVException {
		MediaFrame mediaFrame = new MediaFrame();

		while (mediaFrame != null && !mediaFrame.hasFrame()) {
			if (av_read_frame(formatContext, avPacket) < 0) {
				if (videoStream != null) {
					// video codec may have buffered some frames
					avPacket.stream_index(videoStream.index());
					avPacket.data(null);
					avPacket.size(0);
				}
				else {
					return null;
				}
			}

			mediaPacket = new MediaPacket(avPacket);

			if (videoStream != null && avPacket.stream_index() == videoStream.index()) {
				mediaFrame = videoDecoder.decodeVideo(mediaPacket);
			}
			else if (audioStream != null && avPacket.stream_index() == audioStream.index()) {
				mediaFrame = audioDecoder.decodeAudio(mediaPacket);
			}

			av_free_packet(avPacket);
		}

		return mediaFrame;
	}

	public void setInputFormat(String format) {
		this.format = format;
	}

	public String getInputFormat() {
		if (formatContext == null) {
			return format;
		}
		else {
			return formatContext.iformat().name().getString();
		}
	}

	public int getImageWidth() {
		return videoDecoder == null ? super.getImageWidth() : videoDecoder.getImageWidth();
	}

	public int getImageHeight() {
		return videoDecoder == null ? super.getImageHeight() : videoDecoder.getImageHeight();
	}

	public int getAudioChannels() {
		return audioDecoder == null ? super.getAudioChannels() : audioDecoder.getAudioChannels();
	}

	public PixelFormat getPixelFormat() {
		return audioDecoder == null ? super.getPixelFormat() : audioDecoder.getPixelFormat();
	}

	public SampleFormat getSampleFormat() {
		return audioDecoder == null ? super.getSampleFormat() : audioDecoder.getSampleFormat();
	}

	public double getFrameRate() {
		if (videoStream == null) {
			return super.getFramerate();
		}
		else {
			AVRational rate = videoStream.r_frame_rate();
			return av_q2d(rate);
		}
	}

	public int getSampleRate() {
		return audioDecoder == null ? super.getSampleRate() : audioDecoder.getSampleRate();
	}

	private void initVideoDecoder(AVCodecContext codecContext) {
		if (codecContext == null)
			return;

		try {
			CodecID codecId = CodecID.byId(codecContext.codec_id());

			videoDecoder = new Decoder(codecId, codecContext);
			videoDecoder.setPixelFormat(getPixelFormat());
			videoDecoder.open(null);
		}
		catch (JavaAVException e) {
			e.printStackTrace();
		}
	}

	private void initAudioDecoder(AVCodecContext codecContext) {
		if (codecContext == null)
			return;

		try {
			CodecID codecId = CodecID.byId(codecContext.codec_id());

			audioDecoder = new Decoder(codecId, codecContext);
			audioDecoder.open(null);
		}
		catch (JavaAVException e) {
			e.printStackTrace();
		}
	}
	
}
