package com.github.houbb.monitor.client.java.thread;

import com.alibaba.fastjson.JSON;
import com.github.houbb.checksum.core.util.ChecksumHelper;
import com.github.houbb.common.api.api.dto.req.CommonApiRequest;
import com.github.houbb.heaven.util.util.CollectionUtil;
import com.github.houbb.hibernator.valid.util.ValidHelper;
import com.github.houbb.http.client.util.OkHttpClientHelper;
import com.github.houbb.log.integration.core.Log;
import com.github.houbb.log.integration.core.LogFactory;
import com.github.houbb.monitor.client.api.request.MonitorLogRequest;
import com.github.houbb.monitor.client.api.request.component.MonitorLogItem;
import com.github.houbb.monitor.client.api.response.MonitorLogResponse;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;

/**
 * 推送策略线程
 *
 * @since 1.0.0
 * @author dh
 */
public class MonitorServerPushThread extends Thread {

    /**
     * 日志编号
     * @since 1.0.0
     */
    private static final Log LOG = LogFactory.getLog(MonitorServerPushThread.class);

    private final MonitorServerPushThreadConfig config;

    public MonitorServerPushThread(MonitorServerPushThreadConfig config) {
        this.config = config;
    }

    @Override
    public void run() {
        try {
            final Queue<MonitorLogItem> queue = config.getQueue();

            if(queue.isEmpty()) {
                LOG.warn("队列为空，忽略处理");
                return;
            }
            // 定时执行（单次大小可调整）
            final int maxLimit = config.getMaxBatchSize();
            List<MonitorLogItem> list = new ArrayList<>(maxLimit);

            while (!queue.isEmpty() && list.size() < maxLimit) {
                // 移除队首的元素
                MonitorLogItem item = queue.poll();
                list.add(item);
            }

            if(CollectionUtil.isEmpty(list)) {
                LOG.warn("队列为空，忽略处理。");
                return;
            }

            // 调用服务端
            MonitorLogRequest request = new MonitorLogRequest();
            request.setItemList(list);
            fillAndCallServer(request, "api/monitor/log/add", MonitorLogResponse.class);
        } catch (Exception e) {
            LOG.error("任务执行异常", e);
        }
    }

    /**
     * 填充信息，并且调用服务端
     * @param request 请求
     * @param urlSuffix 请求地址
     * @param responseClass 响应类
     * @param <R> 泛型
     * @return 结果
     */
    private <R> R fillAndCallServer(CommonApiRequest request,
                                    String urlSuffix,
                                    Class<R> responseClass) {
        try {
            final String traceId = config.getTraceIdGenerator().id();

            request.setTraceId(traceId);
            request.setRequestTime(System.currentTimeMillis());
            request.setSystemId(config.getSystemId());
            request.setEnvId(config.getEnvId());
            ChecksumHelper.fill(request, config.getSaltKey());

            ValidHelper.failFast(request);

            // 调用
            String url = config.getServerAddress() + urlSuffix;
            String json = JSON.toJSONString(request);
            LOG.info("开始请求服务端url {}, json {}", url, json);
            String resultJson = OkHttpClientHelper.postJson(url, json);
            LOG.info("完成请求服务端 {}", resultJson);
            return JSON.parseObject(resultJson, responseClass);
        } catch (Exception e) {
            LOG.error("请求服务端异常", e);
            return null;
        }
    }

}
