package org.hcjf.layers.storage.postgres;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.hcjf.errors.Errors;
import org.hcjf.layers.storage.StorageLayer;
import org.hcjf.layers.storage.postgres.errors.PostgressErrors;
import org.hcjf.layers.storage.postgres.properties.PostgresProperties;
import org.hcjf.log.Log;
import org.hcjf.properties.SystemProperties;
import org.postgresql.ds.PGSimpleDataSource;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;

/**
 * Base layer to create a pooling connection with a postgres data base engine.
 * @author Javier Quiroga.
 */
public abstract class PostgresStorageLayer<S extends PostgresStorageSession> extends StorageLayer<S> {

    private DataSource source;

    public PostgresStorageLayer(String implName) {
        super(implName);
    }

    /**
     * The first time this method creates a postgres pooling data source, then
     * only return a session with a postgres connection.
     * @return Postgres storage session.
     */
    @Override
    public S begin() {
        synchronized (this) {
            if(source == null) {

                HikariConfig hikariConfig = new HikariConfig();
                hikariConfig.setDataSourceClassName(PGSimpleDataSource.class.getName());
                hikariConfig.addDataSourceProperty(SystemProperties.get(PostgresProperties.Pool.SERVER_NAME_FIELD), getServerName());
                hikariConfig.addDataSourceProperty(SystemProperties.get(PostgresProperties.Pool.DATABASE_NAME_FIELD), getDatabaseName());
                hikariConfig.addDataSourceProperty(SystemProperties.get(PostgresProperties.Pool.USER_FIELD), getUserName());
                hikariConfig.addDataSourceProperty(SystemProperties.get(PostgresProperties.Pool.PASSWORD_FIELD), getPassword());
                hikariConfig.addDataSourceProperty(SystemProperties.get(PostgresProperties.Pool.PORT_NUMBER_FIELD), getPortNumber());
                hikariConfig.setPoolName(getDataSourceName());
                hikariConfig.setMaximumPoolSize(getMaxConnections());
                hikariConfig.setMinimumIdle(getInitialConnections());
                hikariConfig.setIdleTimeout(getIdleTimeout());
                hikariConfig.setMaxLifetime(getMaxLifeTime());

                source = new HikariDataSource(hikariConfig);

                try {
                    Connection connection = source.getConnection();
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
        try {
            Connection connection = source.getConnection();
            connection.setAutoCommit(false);
            return getSessionInstance(getImplName(), connection);
        } catch (SQLException ex) {
            Log.e(SystemProperties.get(PostgresProperties.POSTGRES_STORAGE_LAYER_LOG_TAG),
                    Errors.getMessage(PostgressErrors.UNABLE_TO_CREATE_CONNECTION), ex);
            throw new RuntimeException(Errors.getMessage(PostgressErrors.UNABLE_TO_CREATE_CONNECTION), ex);
        }
    }

    protected abstract S getSessionInstance(String implName, Connection connection);

    /**
     * Return a name for a data source.
     * @return Data source name.
     */
    protected abstract String getDataSourceName();

    /**
     * Return the host of the data base engine.
     * @return Data base engine host.
     */
    protected abstract String getServerName();

    /**
     * Return the data base name.
     * @return Data base name.
     */
    protected abstract String getDatabaseName();

    /**
     * Return the user name.
     * @return User name.
     */
    protected abstract String getUserName();

    /**
     * Return the password.
     * @return Password.
     */
    protected abstract String getPassword();

    /**
     * Return the initial connection size for the pool.
     * @return Initial connection size.
     */
    protected Integer getInitialConnections() {
        return SystemProperties.getInteger(PostgresProperties.Pool.INIT_CONNECTIONS);
    }

    /**
     * Return the max connection size for the pool.
     * @return Max connection size.
     */
    protected Integer getMaxConnections() {
        return SystemProperties.getInteger(PostgresProperties.Pool.MAX_CONNECTIONS);
    }

    /**
     * Return the port number of the server.
     * @return Port number.
     */
    protected abstract Integer getPortNumber();

    protected Long getIdleTimeout() {
        return SystemProperties.getLong(PostgresProperties.Pool.IDLE_TIMEOUT);
    }

    protected Long getMaxLifeTime() {
        return SystemProperties.getLong(PostgresProperties.Pool.MAX_LIFE_TIME);
    }

}
