/*
 * Decompiled with CFR 0.152.
 */
package org.hcjf.log;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.hcjf.log.LogPrinter;
import org.hcjf.properties.SystemProperties;
import org.hcjf.service.Service;
import org.hcjf.service.ServiceSession;
import org.hcjf.service.ServiceThread;
import org.hcjf.utils.Strings;

public final class Log
extends Service<LogPrinter> {
    private static final Log instance = new Log();
    private List<LogPrinter> printers;
    private Queue<LogRecord> queue;
    private Object logMonitor;
    private Boolean shuttingDown;

    private Log() {
        super(SystemProperties.get("hcjf.log.service.name"), SystemProperties.getInteger("hcjf.log.service.priority"));
    }

    @Override
    protected void init() {
        this.printers = new ArrayList<LogPrinter>();
        this.queue = new PriorityBlockingQueue<LogRecord>(SystemProperties.getInteger("hcjf.log.queue.initial.size"), (o1, o2) -> (int)(o1.getDate().getTime() - o2.getDate().getTime()));
        this.logMonitor = new Object();
        this.shuttingDown = false;
        for (int i = 0; i < SystemProperties.getInteger("hcjf.log.consumers.size"); ++i) {
            this.fork(new LogRunnable());
        }
        List<String> logConsumers = SystemProperties.getList("hcjf.log.consumers");
        logConsumers.forEach(S -> {
            try {
                LogPrinter printer = (LogPrinter)Class.forName(S).getConstructor(new Class[0]).newInstance(new Object[0]);
                this.registerConsumer(printer);
            }
            catch (Exception ex) {
                ex.printStackTrace();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void shutdown(Service.ShutdownStage stage) {
        switch (stage) {
            case START: {
                this.shuttingDown = true;
                Object object = this.logMonitor;
                synchronized (object) {
                    this.logMonitor.notifyAll();
                    break;
                }
            }
        }
    }

    @Override
    public void registerConsumer(LogPrinter consumer) {
        if (consumer == null) {
            throw new NullPointerException("Log printer null");
        }
        this.printers.add(consumer);
    }

    @Override
    public void unregisterConsumer(LogPrinter consumer) {
        this.printers.remove(consumer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private UUID addRecord(LogRecord record) {
        if (Log.instance.queue.add(record)) {
            Object object = this.logMonitor;
            synchronized (object) {
                this.logMonitor.notifyAll();
            }
        }
        return record.getId();
    }

    private static String[] getCallStackInformation() {
        StackTraceElement element = Thread.currentThread().getStackTrace()[3];
        return new String[]{element.getClassName(), element.getMethodName(), Integer.toString(element.getLineNumber())};
    }

    public static void addPrinter(LogPrinter printer) {
        instance.registerConsumer(printer);
    }

    private boolean verifyPrinters() {
        return SystemProperties.getBoolean("hcjf.log.system.out.enabled") != false || Log.instance.printers.size() > 0;
    }

    public static UUID d(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.DEBUG, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID d(String tag, String message, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.DEBUG, tag, message, Log.getCallStackInformation(), throwable, params));
        }
        return result;
    }

    public static UUID d(String tag, String message, Boolean printThrowable, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.DEBUG, tag, message, Log.getCallStackInformation(), printThrowable, throwable, params));
        }
        return result;
    }

    public static UUID i(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.INFO, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID in(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.INPUT, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID out(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.OUTPUT, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID w(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.WARNING, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID w(String tag, String message, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.WARNING, tag, message, Log.getCallStackInformation(), throwable, params));
        }
        return result;
    }

    public static UUID w(String tag, String message, Boolean printThrowable, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.WARNING, tag, message, Log.getCallStackInformation(), printThrowable, throwable, params));
        }
        return result;
    }

    public static UUID e(String tag, String message, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.ERROR, tag, message, Log.getCallStackInformation(), params));
        }
        return result;
    }

    public static UUID e(String tag, String message, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.ERROR, tag, message, Log.getCallStackInformation(), throwable, params));
        }
        return result;
    }

    public static UUID e(String tag, String message, Boolean printThrowable, Throwable throwable, Object ... params) {
        UUID result = null;
        if (instance.verifyPrinters()) {
            result = instance.addRecord(new LogRecord(LogGroup.ERROR, tag, message, Log.getCallStackInformation(), printThrowable, throwable, params));
        }
        return result;
    }

    public static Integer getLogQueueSize() {
        return Log.instance.queue.size();
    }

    private class LogRunnable
    implements Runnable {
        private LogRunnable() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                while (!Log.this.shuttingDown.booleanValue()) {
                    if (Log.instance.queue.isEmpty()) {
                        Object object = Log.this.logMonitor;
                        synchronized (object) {
                            Log.this.logMonitor.wait();
                        }
                    }
                    try {
                        this.writeRecord(Log.instance.queue.remove());
                    }
                    catch (Exception exception) {}
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        private void writeRecord(LogRecord record) {
            ServiceSession.runAs(() -> {
                if (record.getGroup().getOrder() >= SystemProperties.getInteger("hcjf.log.level")) {
                    Log.this.printers.forEach(printer -> printer.print(record));
                    if (SystemProperties.getBoolean("hcjf.log.system.out.enabled").booleanValue()) {
                        if (SystemProperties.getBoolean("hcjf.log.java.standard.logger.enabled").booleanValue()) {
                            Supplier<String> message = () -> String.format(record.getOriginalMessage(), record.getParams());
                            if (record.getThrowable() != null) {
                                Logger.getGlobal().logp(record.getGroup().getStandardLevel(), record.getClassName(), record.getMethodName(), record.getThrowable(), message);
                            } else {
                                Logger.getGlobal().logp(record.getGroup().getStandardLevel(), record.getClassName(), record.getMethodName(), message);
                            }
                        } else if (record.getThrowable() != null) {
                            System.err.println(record.toString());
                            System.err.flush();
                        } else {
                            System.out.println(record.toString());
                            System.out.flush();
                        }
                    }
                }
            }, record.getCurrentSession());
        }
    }

    public static final class LogRecord {
        private final UUID id = UUID.randomUUID();
        private final Date date = new Date();
        private final LogGroup group;
        private final String tag;
        private final String originalMessage;
        private String message;
        private final String className;
        private final String methodName;
        private final String lineNumber;
        private final SimpleDateFormat dateFormat;
        private final Object[] params;
        private final Throwable throwable;
        private final Boolean printThrowable;
        private final ServiceSession currentSession;

        private LogRecord(LogGroup group, String tag, String message, String[] callStackInformation, Boolean printThrowable, Throwable throwable, Object ... params) {
            this.group = group;
            this.tag = tag;
            this.dateFormat = new SimpleDateFormat(SystemProperties.get("hcjf.log.date.format"));
            this.originalMessage = message;
            this.className = callStackInformation[0];
            this.methodName = callStackInformation[1];
            this.lineNumber = callStackInformation[2];
            this.params = params;
            this.printThrowable = printThrowable;
            this.throwable = throwable;
            this.currentSession = Thread.currentThread() instanceof ServiceThread ? ServiceSession.getCurrentIdentity() : ServiceSession.getGuestSession();
        }

        private LogRecord(LogGroup group, String tag, String message, String[] callStackInformation, Throwable throwable, Object ... params) {
            this(group, tag, message, callStackInformation, true, throwable, params);
        }

        private LogRecord(LogGroup group, String tag, String message, String[] callStackInformation, Object ... params) {
            this(group, tag, message, callStackInformation, false, null, params);
        }

        public UUID getId() {
            return this.id;
        }

        private String createMessage() {
            StringWriter stringWriter = new StringWriter();
            PrintWriter printWriter = new PrintWriter(stringWriter);
            String group = this.getGroup().getGroup();
            String tag = this.getTag();
            if (SystemProperties.getBoolean("hcjf.log.truncate.tag").booleanValue()) {
                int truncateSize = SystemProperties.getInteger("hcjf.log.truncate.tag.size");
                if (truncateSize > tag.length()) {
                    tag = Strings.rightPad(tag, truncateSize);
                } else if (truncateSize < tag.length()) {
                    tag = tag.substring(0, truncateSize);
                }
            }
            if (this.getGroup().equals((Object)LogGroup.ERROR)) {
                printWriter.print("\u001b[1;31m");
            } else if (this.getGroup().equals((Object)LogGroup.WARNING)) {
                printWriter.print("\u001b[1;33m");
            } else if (this.getGroup().equals((Object)LogGroup.DEBUG)) {
                printWriter.print("\u001b[1;35m");
            } else {
                printWriter.print("\u001b[1;32m");
            }
            printWriter.print(this.dateFormat.format(this.getDate()));
            printWriter.print("\u001b[0m");
            printWriter.print(" [");
            if (this.getGroup().equals((Object)LogGroup.ERROR)) {
                printWriter.print("\u001b[1;31m");
            } else if (this.getGroup().equals((Object)LogGroup.WARNING)) {
                printWriter.print("\u001b[1;33m");
            } else if (this.getGroup().equals((Object)LogGroup.DEBUG)) {
                printWriter.print("\u001b[1;35m");
            } else {
                printWriter.print("\u001b[1;32m");
            }
            printWriter.print(group);
            printWriter.print("\u001b[0m");
            printWriter.print("][");
            printWriter.print(tag);
            printWriter.print("][");
            printWriter.print(this.getClassName());
            printWriter.print("][");
            printWriter.print(this.getMethodName());
            printWriter.print("][");
            printWriter.print(this.getLineNumber());
            printWriter.print("] ");
            printWriter.printf(this.originalMessage, this.params);
            if (this.throwable != null && this.printThrowable.booleanValue()) {
                printWriter.print("\r\n");
                this.throwable.printStackTrace(printWriter);
            }
            printWriter.flush();
            printWriter.close();
            return stringWriter.toString();
        }

        public ServiceSession getCurrentSession() {
            return this.currentSession;
        }

        public Date getDate() {
            return this.date;
        }

        public LogGroup getGroup() {
            return this.group;
        }

        public String getTag() {
            return this.tag;
        }

        public synchronized String getMessage() {
            if (this.message == null) {
                this.message = this.createMessage();
            }
            return this.message;
        }

        public String getOriginalMessage() {
            return this.originalMessage;
        }

        public String toString() {
            return this.getMessage();
        }

        public Object[] getParams() {
            return this.params;
        }

        public Throwable getThrowable() {
            return this.throwable;
        }

        public String getClassName() {
            return this.className;
        }

        public String getMethodName() {
            return this.methodName;
        }

        public String getLineNumber() {
            return this.lineNumber;
        }
    }

    public static enum LogGroup {
        DEBUG("D", 0),
        INPUT("A", 1),
        OUTPUT("O", 1),
        INFO("I", 1),
        WARNING("W", 2),
        ERROR("E", 3);

        private Integer order;
        private String group;

        private LogGroup(String tag, Integer order) {
            this.order = order;
            this.group = tag;
        }

        public Integer getOrder() {
            return this.order;
        }

        public String getGroup() {
            return this.group;
        }

        public LogGroup valueOfByTag(String tagString) {
            LogGroup result = null;
            try {
                result = Stream.of(LogGroup.values()).filter(r -> r.getGroup().equals(tagString)).findFirst().get();
            }
            catch (NoSuchElementException noSuchElementException) {
                // empty catch block
            }
            return result;
        }

        public Level getStandardLevel() {
            Level result;
            switch (this) {
                case ERROR: {
                    result = Level.SEVERE;
                    break;
                }
                case WARNING: {
                    result = Level.WARNING;
                    break;
                }
                default: {
                    result = Level.INFO;
                }
            }
            return result;
        }
    }
}

