/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.agcod;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.agcod.model.AgcodException;
import software.amazon.awssdk.services.agcod.model.CancelGiftCardRequest;
import software.amazon.awssdk.services.agcod.model.CancelGiftCardResponse;
import software.amazon.awssdk.services.agcod.model.CreateGiftCardRequest;
import software.amazon.awssdk.services.agcod.model.CreateGiftCardResponse;
import software.amazon.awssdk.services.agcod.model.GetAvailableFundsRequest;
import software.amazon.awssdk.services.agcod.model.GetAvailableFundsResponse;
import software.amazon.awssdk.services.agcod.model.InternalServerErrorException;
import software.amazon.awssdk.services.agcod.model.PartnerAccessErrorException;
import software.amazon.awssdk.services.agcod.model.PartnerInputErrorException;
import software.amazon.awssdk.services.agcod.model.ResendErrorException;
import software.amazon.awssdk.services.agcod.model.SystemErrorException;
import software.amazon.awssdk.services.agcod.model.UnknownErrorException;
import software.amazon.awssdk.services.agcod.transform.CancelGiftCardRequestMarshaller;
import software.amazon.awssdk.services.agcod.transform.CreateGiftCardRequestMarshaller;
import software.amazon.awssdk.services.agcod.transform.GetAvailableFundsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AgcodAsyncClient}.
 *
 * @see AgcodAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAgcodAsyncClient implements AgcodAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAgcodAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAgcodAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The <code>CancelGiftCardRequest</code> request cancels the Gift Card. This operation can only be executed within
     * 15 minutes of the creation request time stamp
     * </p>
     *
     * @param cancelGiftCardRequest
     *        Represents the input of a <code>CancelGiftCardRequest</code> operation.
     * @return A Java Future containing the result of the CancelGiftCard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SystemErrorException F100: System Errors</li>
     *         <li>PartnerInputErrorException F200: Partner Input Errors</li>
     *         <li>PartnerAccessErrorException F300: Partner Account/Access/Onboarding Errors</li>
     *         <li>ResendErrorException F400: RESEND Errors</li>
     *         <li>UnknownErrorException F500: Unknown Errors</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AgcodException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AgcodAsyncClient.CancelGiftCard
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/agcod-2019-11-29/CancelGiftCard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelGiftCardResponse> cancelGiftCard(CancelGiftCardRequest cancelGiftCardRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelGiftCardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelGiftCardResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelGiftCardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelGiftCardRequest, CancelGiftCardResponse>()
                            .withOperationName("CancelGiftCard")
                            .withMarshaller(new CancelGiftCardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(cancelGiftCardRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateGiftCardRequest</code> requests creates a Amazon Gift Card
     * </p>
     *
     * @param createGiftCardRequest
     *        Represents the input of a <code>CreateGiftCardRequest</code> operation.
     * @return A Java Future containing the result of the CreateGiftCard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SystemErrorException F100: System Errors</li>
     *         <li>PartnerInputErrorException F200: Partner Input Errors</li>
     *         <li>PartnerAccessErrorException F300: Partner Account/Access/Onboarding Errors</li>
     *         <li>ResendErrorException F400: RESEND Errors</li>
     *         <li>UnknownErrorException F500: Unknown Errors</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AgcodException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AgcodAsyncClient.CreateGiftCard
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/agcod-2019-11-29/CreateGiftCard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGiftCardResponse> createGiftCard(CreateGiftCardRequest createGiftCardRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGiftCardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGiftCardResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGiftCardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGiftCardRequest, CreateGiftCardResponse>()
                            .withOperationName("CreateGiftCard")
                            .withMarshaller(new CreateGiftCardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createGiftCardRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetAvailableFundsRequest</code> returns the amount of funds currently available in your Amazon
     * Incentives account. It provides an alternative to logging into our portal for viewing available funds. A
     * potential use-case of this operation is to develop a customized alerting system or checking the balance of an
     * account periodically.
     * </p>
     *
     * @param getAvailableFundsRequest
     *        Represents the input of a <code>GetAvailableFundsRequest</code> operation.
     * @return A Java Future containing the result of the GetAvailableFunds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SystemErrorException F100: System Errors</li>
     *         <li>PartnerInputErrorException F200: Partner Input Errors</li>
     *         <li>PartnerAccessErrorException F300: Partner Account/Access/Onboarding Errors</li>
     *         <li>ResendErrorException F400: RESEND Errors</li>
     *         <li>UnknownErrorException F500: Unknown Errors</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AgcodException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AgcodAsyncClient.GetAvailableFunds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/agcod-2019-11-29/GetAvailableFunds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAvailableFundsResponse> getAvailableFunds(GetAvailableFundsRequest getAvailableFundsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAvailableFundsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAvailableFundsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAvailableFundsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAvailableFundsRequest, GetAvailableFundsResponse>()
                            .withOperationName("GetAvailableFunds")
                            .withMarshaller(new GetAvailableFundsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAvailableFundsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AgcodException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .customErrorCodeFieldName("errorCode")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("F100").exceptionBuilderSupplier(SystemErrorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("F200")
                                .exceptionBuilderSupplier(PartnerInputErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("F300")
                                .exceptionBuilderSupplier(PartnerAccessErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("F500").exceptionBuilderSupplier(UnknownErrorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("F400").exceptionBuilderSupplier(ResendErrorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
