package internal.sql.odbc;

import java.lang.Iterable;
import java.util.Comparator;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Predicate;
import java.util.stream.StreamSupport;
import nbbrd.sql.odbc.OdbcConnectionSupplierSpi;

/**
 * Custom service loader for {@link nbbrd.sql.odbc.OdbcConnectionSupplierSpi}.
 * <br>This class is thread-safe.
 * <p>Properties:
 * <ul>
 * <li>Quantifier: OPTIONAL</li>
 * <li>Fallback: null</li>
 * <li>Preprocessing: wrapper: none filters:[isAvailable] sorters:[getCost]</li>
 * <li>Mutability: NONE</li>
 * <li>Singleton: false</li>
 * <li>Name: internal.sql.odbc.OdbcConnectionSupplierSpiLoader</li>
 * <li>Backend: null</li>
 * <li>Cleaner: null</li>
 * <li>Batch: false</li>
 * <li>Batch name: null</li>
 * </ul>
 */
public final class OdbcConnectionSupplierSpiLoader {
  private final Iterable<OdbcConnectionSupplierSpi> source = ServiceLoader.load(OdbcConnectionSupplierSpi.class);

  private final Predicate<OdbcConnectionSupplierSpi> filter = OdbcConnectionSupplierSpi::isAvailable;

  private final Comparator<OdbcConnectionSupplierSpi> sorter = Comparator.comparingInt(OdbcConnectionSupplierSpi::getCost);

  private final Optional<OdbcConnectionSupplierSpi> resource = doLoad();

  private Optional<OdbcConnectionSupplierSpi> doLoad() {
    return StreamSupport.stream(source.spliterator(), false)
        .filter(filter)
        .sorted(sorter)
        .findFirst();
  }

  /**
   * Gets an optional {@link nbbrd.sql.odbc.OdbcConnectionSupplierSpi} instance.
   * <br>This method is thread-safe.
   * @return the current non-null value
   */
  public Optional<OdbcConnectionSupplierSpi> get() {
    return resource;
  }

  /**
   * Gets an optional {@link nbbrd.sql.odbc.OdbcConnectionSupplierSpi} instance.
   * <br>This is equivalent to the following code: <code>new OdbcConnectionSupplierSpiLoader().get()</code>
   * <br>Therefore, the returned value might be different at each call.
   * <br>This method is thread-safe.
   * @return a non-null value
   */
  public static Optional<OdbcConnectionSupplierSpi> load() {
    return new OdbcConnectionSupplierSpiLoader().get();
  }
}
