package org.api.manapool.services;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.api.manapool.model.InventoryItem;
import org.api.manapool.model.OrderSummary;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.api.manapool.model.Account;
import org.api.manapool.model.Inventory;
import org.api.manapool.model.PriceVariation;
import org.api.manapool.model.ProductQueryEntry;
import org.api.manapool.tools.ManaPoolConstants;
import org.api.manapool.tools.RestClient;

import com.google.gson.JsonObject;

public class ManaPoolAPIService {

	private RestClient client;
	protected static Logger logger = LogManager.getLogger(ManaPoolAPIService.class);

	public ManaPoolAPIService(String email, String token) {
		client = new RestClient(email, token);
	}
	
	public ManaPoolAPIService(Properties p) {
		client = new RestClient(p.getProperty("EMAIL"), p.getProperty("TOKEN"));
	}
	
	public RestClient getClient() {
		return client;
	}
	
	public List<InventoryItem> getSellerInventory() throws IOException
	{
		return client.get("/seller/inventory", null, Inventory.class).getItems();
	}
	
	public List<InventoryItem> getSellerInventory(int limit, int offset) throws IOException
	{
		return client.get("/seller/inventory?limit="+limit+"&offset="+offset, null, Inventory.class).getItems();
	}
	
	public List<InventoryItem> addInventoryItems(List<ProductQueryEntry> inventoryList) throws IOException
	{
		return client.post("/seller/inventory/scryfall_id",inventoryList,null,Inventory.class).getItems();
	}
	
	public InventoryItem updateInventoryItems(ProductQueryEntry item) throws IOException
	{
		var obj = client.put("/seller/inventory/scryfall_id/"+item.getScryfallId()+"?language_id="+item.getLanguage()+"&finish_id="+item.getFinishId()+"&condition_id="+item.getCondition(),item,null,JsonObject.class);
		return client.fromJson(obj.get("inventory").getAsJsonObject().toString(), InventoryItem.class);
	}
	
	public InventoryItem deleteInventoryItems(ProductQueryEntry item) throws IOException
	{
		var obj = client.delete("/seller/inventory/scryfall_id/"+item.getScryfallId()+"?language_id="+item.getLanguage()+"&finish_id="+item.getFinishId()+"&condition_id="+item.getCondition(),item,null,JsonObject.class);
		return client.fromJson(obj.get("inventory").getAsJsonObject().toString(), InventoryItem.class);
	}
	
	public List<PriceVariation> listSinglesPrices() throws IOException
	{
		var arr = client.get("/prices/singles", null, JsonObject.class).get("data").getAsJsonArray();
		return arr.asList().stream().map(e->client.fromJson(e.toString(),PriceVariation.class)).toList();
	}
	
	public List<PriceVariation> listSealedPrices() throws IOException
	{
		var arr = client.get("/prices/sealed", null, JsonObject.class).get("data").getAsJsonArray();
		return arr.asList().stream().map(e->client.fromJson(e.toString(),PriceVariation.class)).toList();
	}
	
	public List<PriceVariation> listVariantsPrices() throws IOException
	{
		var arr = client.get("/prices/variants", null, JsonObject.class).get("data").getAsJsonArray();
		return arr.asList().stream().map(e->client.fromJson(e.toString(),PriceVariation.class)).toList();
	}
	
	public List<OrderSummary> listSellsOrders(int limit, int offset) throws IOException
	{
		var arr = client.get("/seller/orders?limit="+limit+"&offset="+offset, null, JsonObject.class).get("orders").getAsJsonArray();
		return arr.asList().stream().map(e->client.fromJson(e.toString(),OrderSummary.class)).toList();
	}
	
	public double userCredits() throws IOException
	{
		var arr = client.get("/buyer/credit", null, JsonObject.class).get("user_credit_cents").getAsInt();
		return arr/100.0;
	}
	
	public Account userAccount() throws IOException
	{
		return client.get("/account", null, Account.class);
	}
	
	
	public static Date parseDate(String date)
	{
		try {
			return new SimpleDateFormat(ManaPoolConstants.DATE_FORMAT).parse(date);
		} catch (ParseException e) {
			logger.error(e);
			return null;
		}
	}
	
	public static String toDate(Date date)
	{
			return new SimpleDateFormat(ManaPoolConstants.DATE_FORMAT).format(date);
	}
	
}
