package seleniumConsulting.ch.selenium.framework.driver;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.util.Map;

import com.google.common.base.Strings;
import com.neotys.rest.design.client.DesignAPIClient;
import com.neotys.rest.error.NeotysAPIException;
import com.neotys.selenium.proxies.DesignManager;
import com.neotys.selenium.proxies.NLWebDriverFactory;
import com.neotys.selenium.proxies.helpers.SeleniumProxyConfig;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Point;
import org.openqa.selenium.Proxy;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.edge.EdgeDriver;
import org.openqa.selenium.edge.EdgeOptions;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.ie.InternetExplorerDriver;
import org.openqa.selenium.ie.InternetExplorerOptions;
import org.openqa.selenium.remote.CapabilityType;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import seleniumConsulting.ch.selenium.framework.dataLoader.TestDataProvider;
import vendors.grid.TestInfo;
import vendors.grid.VendorProvider;

import javax.ws.rs.Uri;


/**
 * Factory to create {@link WebDriver} and {@link RemoteWebDriver}
 */
public class WebDriverFactory {

    /**
     * Definition of remote-grid-url Property-Key
     */
    public static final String REMOTE_GRID_URL = "remote.grid.url";

    /**
     * Definition of capabilities Property-Key
     */
    public static final String CAPABILITIES = "capabilities";

    /**
     * Definition of environment Property-Key
     */
    public static final String ENV = "env";
    public static final String WD_HUB = "/wd/hub";
    public static final String EDGE = "edge";
    public static final String CHROME_BINARY = "chromeBinary";
    public static final String NEOLOAD = "neoload";

    /**
     * Definition of Property-Value 'local' for environment
     */
    public static String ENV_LOCAL = "local";

    /**
     * Definition of firefox Property-Value
     */
    public static final String FIREFOX = "firefox";

    /**
     * Definition of chrome Property-Value
     */
    public static final String CHROME = "chrome";

    /**
     * Definition of 'internet explorer' Property-Value
     */
    public static final String INTERNET_EXPLORER = "internet explorer";


    /**
     * Create a {@link WebDriver} from Browser = browserName, Version = browserVersion with capibilities = capibilitiesMetadata
     * Additional this method reads the testconfig/data.properties File and uses the properties to create the Browser.
     * @param browserName of Browser
     * @param browserVersion of Browser
     * @param capibilitiesMetadata of Browser
     * @return WebDriver of created Driver
     */
    public static WebDriver createDriver(String browserName, String browserVersion, Map<String, String> capibilitiesMetadata, TestInfo testInfo) {
        String env = System.getProperty(ENV);
        WebDriver driver;

        if (ENV_LOCAL.equalsIgnoreCase(env)) {
            //TODO: OPTIONS SETZEN!!!
            switch (browserName) {
                case FIREFOX:
                    FirefoxOptions firefoxOptions = new FirefoxOptions();
                    firefoxOptions.setCapability("acceptInsecureCerts", true);
                    firefoxOptions.setCapability("marionette", true);
                    firefoxOptions = (FirefoxOptions)addNeoloadCapIfNeeded(firefoxOptions);
                    driver = new FirefoxDriver(firefoxOptions);
                    break;

                case CHROME:
                    ChromeOptions chromeOptions = new ChromeOptions();
                    chromeOptions.setExperimentalOption("useAutomationExtension", false);
                    if(!Strings.isNullOrEmpty(TestDataProvider.getTestData(CHROME_BINARY))) {
                        chromeOptions.setBinary(TestDataProvider.getTestData(CHROME_BINARY));
                    }
                    chromeOptions = (ChromeOptions)addNeoloadCapIfNeeded(chromeOptions);
                    driver = new ChromeDriver(chromeOptions);
                    break;

                case INTERNET_EXPLORER:
                    DesiredCapabilities desiredCapabilities = DesiredCapabilities.internetExplorer();
                    desiredCapabilities.setCapability(InternetExplorerDriver.INTRODUCE_FLAKINESS_BY_IGNORING_SECURITY_DOMAINS, true);
                    desiredCapabilities.setCapability(InternetExplorerDriver.IGNORE_ZOOM_SETTING, true);
                    desiredCapabilities.setCapability(InternetExplorerDriver.ELEMENT_SCROLL_BEHAVIOR, 1);
                    desiredCapabilities.setCapability(InternetExplorerDriver.NATIVE_EVENTS, false);
                    desiredCapabilities = (DesiredCapabilities) addNeoloadCapIfNeeded(desiredCapabilities);
                    InternetExplorerOptions internetExplorerOptions = new InternetExplorerOptions(desiredCapabilities);
                    driver = new InternetExplorerDriver(internetExplorerOptions);
                    break;

                case EDGE:
                    DesiredCapabilities desiredCapabilitiesEdge = DesiredCapabilities.edge();
                    desiredCapabilitiesEdge.setCapability(InternetExplorerDriver.INTRODUCE_FLAKINESS_BY_IGNORING_SECURITY_DOMAINS, true);
                    desiredCapabilitiesEdge = (DesiredCapabilities) addNeoloadCapIfNeeded(desiredCapabilitiesEdge);
                    EdgeOptions edgeOptions = new EdgeOptions();
                    edgeOptions.merge(desiredCapabilitiesEdge);
                    driver = new EdgeDriver(edgeOptions);
                    break;


                default:
                    throw new RuntimeException("Browser " + browserName + " is not available on local platform ");
            }
            if(!Strings.isNullOrEmpty(System.getProperty(NEOLOAD))){
                driver = NLWebDriverFactory.newNLWebDriver(driver, testInfo.getTestname(), null);
            }
        } else {
            // ENV = grid
            MutableCapabilities options;
            switch (browserName) {
                case CHROME:
                    options = new ChromeOptions();
                    break;

                case FIREFOX:
                    options = new FirefoxOptions();
                    break;

                case EDGE:
                    options = new EdgeOptions();
                    break;

                case INTERNET_EXPLORER:
                    options = new InternetExplorerOptions();
                    break;

                default:
                    throw new RuntimeException("Browser " + browserName + " is not supported in the Selenium toolkit");
            }
            if(!Strings.isNullOrEmpty(browserVersion)){
                options.setCapability(CapabilityType.BROWSER_VERSION, browserVersion);
            }
            options.setCapability("acceptInsecureCerts", true);
            VendorProvider.getVendor().addCapibilities(testInfo, options);
            setConfigCapibilities(options);
            setWebTestAnnotationCapibilities(options, capibilitiesMetadata);

            driver = createRemoteDriver(System.getProperty(REMOTE_GRID_URL)+ WD_HUB, options);
        }
        System.out.println(((RemoteWebDriver) driver).getCapabilities().toString());
        setWindowSizeFullscreen(driver);
        return driver;
    }

    public static MutableCapabilities addNeoloadCapIfNeeded(MutableCapabilities mutableCapabilities) {
        if(!Strings.isNullOrEmpty(System.getProperty(NEOLOAD))) {
            System.setProperty("nl.selenium.proxy.mode", "Design");
            DesignAPIClient designAPIClient = DesignManager.getDesignApiClient();
            String host = getDomainName(SeleniumProxyConfig.getDesignAPIURL());

            int port;
            try {
                port = designAPIClient.getRecorderSettings().getProxySettings().getPort();
            } catch (IOException | URISyntaxException | GeneralSecurityException | NeotysAPIException e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            }
            String proxyString = host + ":" +port;
            Proxy proxy = new Proxy();
            proxy.setProxyType(Proxy.ProxyType.MANUAL);
            proxy.setHttpProxy(proxyString);
            proxy.setSslProxy(proxyString);
            mutableCapabilities.setCapability("proxy", proxy);
            mutableCapabilities.setCapability("acceptSslCert", true);
        }
        return mutableCapabilities;
    }

    private static String getDomainName(String url){
        try {
            URI uri = new URI(url);
            String domain = uri.getHost();
            return domain.startsWith("www.") ? domain.substring(4) : domain;
        } catch (URISyntaxException e){
            e.printStackTrace();
            return "localhost";
        }
    }

    /**
     * This method adds all capibilitiesMetadata to the capibilities and parse boolean-values to Boolean.
     * @param capabilities which will be filled up with the other map
     * @param capibilitiesMetadata to add to the capibilities
     */
    private static void setWebTestAnnotationCapibilities(MutableCapabilities capabilities, Map<String, String> capibilitiesMetadata) {
        if(capibilitiesMetadata != null && !capibilitiesMetadata.isEmpty()){
            capibilitiesMetadata.forEach((key, value) -> {
                if (value.equals("true") || value.equals("false")) {
                    capabilities.setCapability(key, Boolean.parseBoolean(value));
                } else {
                    capabilities.setCapability(key, value);
                }
            });
        }
    }

    /**
     * This Metod reads the testconfig/data.properties via TestDataProvider and adds the Values to the Capabilities
     * @param capabilities which will be filled up with the properties
     */
    private static void setConfigCapibilities(MutableCapabilities capabilities) {
        if(!Strings.isNullOrEmpty(TestDataProvider.getTestData(CAPABILITIES))) {
            Map<String, String> map = ImmutableMap.copyOf(Splitter.on(",")
                    .withKeyValueSeparator("=")
                    .split(TestDataProvider.getTestData(CAPABILITIES)));
            map.forEach((key, value) -> {
                if (value.equals("true") || value.equals("false")) {
                    capabilities.setCapability(key, Boolean.parseBoolean(value));
                } else {
                    capabilities.setCapability(key, value);
                }
            });
        }
    }


    /**
     * Create a {@link RemoteWebDriver} on the Grid found by url, with the desiredCapabilities.
     * @param url to the GRID
     * @param desiredCapabilities capibilities for the Grid to create Session
     * @return RemoteWebDriver of the Browser
     */
    private static RemoteWebDriver createRemoteDriver(String url, MutableCapabilities desiredCapabilities) {
        try {
            return new RemoteWebDriver(new URL(url), desiredCapabilities);
        } catch (MalformedURLException e) {
            throw new RuntimeException("Unable to create Remotewebdriver for grid url:" + url+ " message:"+ e.getMessage());
        }
    }

    /**
     * Set the Windowsize to maximize on the {@link WebDriver}
     * @param driver to change size
     */
    private static void setWindowSizeFullscreen(WebDriver driver) {
        driver.manage().window().setPosition(new Point(0, 0));
        driver.manage().window().maximize();
    }
}
