package seleniumConsulting.ch.selenium.framework.driver;


import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Strings;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.support.events.EventFiringWebDriver;
import org.openqa.selenium.support.events.WebDriverEventListener;

import seleniumConsulting.ch.selenium.framework.listeners.webdriver.NLWebDriverEventListener;
import seleniumConsulting.ch.selenium.framework.metadata.MetadataManager;
import vendors.grid.TestInfo;

import static seleniumConsulting.ch.selenium.framework.driver.WebDriverFactory.CAPABILITIES;
import static seleniumConsulting.ch.selenium.framework.driver.WebDriverFactory.NEOLOAD;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERNAME;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERVERSION;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataManager.getMetadata;

/**
 * The WebDriverManager manages all {@link WebDriver}/{@link RemoteWebDriver} for the Tests
 * With WebDriverManager.getWebdriver() you will get everytime the {@link WebDriver}/{@link RemoteWebDriver} of the current Thread.
 */
public class WebDriverManager {

    /**
    * Map<ThreadId, {@link WebDriver}> to store the webDrivers to the threadIds.
    */
    private static Map<Long, WebDriver> webDriverMap = new HashMap();

    /**
     * Map<ThreadId, {@link SessionId}> to store the {@link SessionId} of {@link RemoteWebDriver} to the threadIds.
     */
    private static Map<Long, SessionId> webDriverSessionIDMap = new HashMap();

    /**
     * Get the {@link WebDriver} to your current {@link Thread}
     * @return WebDriver of current Thread
     */
    public static WebDriver getWebdriver() {
        return webDriverMap.get(Thread.currentThread().getId());
    }

    /**
     * Get the {@link SessionId} to your current {@link Thread}
     * @return SessionId or {@code null} if this Thread has no {@link RemoteWebDriver}
     */
    public static SessionId getWebdriverSessionId() {
        return webDriverSessionIDMap.get(Thread.currentThread().getId());
    }

    /**
     * Create a new {@link WebDriver} with the Configs from testconfig/data.prolerties and @Webtest-Properties.
     * It will also store the WebDriver/SessionId to the maps
     * @param testinfo to get the Metadata like BrowserName/Version from the {@link seleniumConsulting.ch.selenium.framework.metadata.MetadataManager}
     * @return WebDriver
     */
    public static WebDriver createWebDriver(TestInfo testinfo) {
        // Create Driver
        WebDriver webDriver = WebDriverFactory.createDriver(
            (String)MetadataManager.getMetadata(testinfo, BROWSERNAME),
            (String)MetadataManager.getMetadata(testinfo, BROWSERVERSION),
            (Map<String, String>)getMetadata(testinfo, CAPABILITIES),
            testinfo
        );

        //Create and register WebDriverEventListener
        EventFiringWebDriver driver = new EventFiringWebDriver(webDriver);

        if(!Strings.isNullOrEmpty(System.getProperty(NEOLOAD))){
            driver.register(new NLWebDriverEventListener());
        } else {
            WebDriverEventListener webDriverEventListener = new seleniumConsulting.ch.selenium.framework.listeners.webdriver.WebDriverEventListener();
            driver.register(webDriverEventListener);
        }

        //Store RemoteWebDriver SessionId if webDriver is instanceOf RemoteWebDriver
        if (webDriver instanceof RemoteWebDriver) {
            webDriverSessionIDMap.put(Thread.currentThread().getId(), ((RemoteWebDriver) webDriver).getSessionId());
        }

        //Store WebDriver
        webDriverMapPut(driver);

        return driver;
    }

    /**
     * Quite Driver, remove Driver and SessionId from store and delete Metadata from {@link MetadataManager}
     */
    public static void quitWebDriver(){
        getWebdriver().quit();
        webDriverMap.remove(Thread.currentThread().getId());
        webDriverSessionIDMap.remove(Thread.currentThread().getId());
    }

    /**
     * Add the {@link WebDriver} to the ThreadId synchronized to the map
     * @param driver to add to the map
     */
    private static void webDriverMapPut(WebDriver driver){
        webDriverMap.put(Thread.currentThread().getId(), driver);
    }

}
