package seleniumConsulting.ch.selenium.framework.dataLoader;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import seleniumConsulting.ch.selenium.framework.allure.AllureTextEnum;


@Slf4j
public class TestDataProvider {

    private static final String FOLDER_TESTDATA_PROPERTIES = "testdata/";
    private static final String FOLDER_TESTCONFIG_PROPERTIES = "testconfig/";

    private static final String FILE_TESTDATA_PROPERTIES = "data.properties";
    private static final String FILE_ALLURETEXTE_PROPERTIES = "allureTexte.properties";
    public static final String USER_NAME = "user.name";

    private Properties testDataProperties;

    private static TestDataProvider instance = null;

    /** Singelton-Pattern*/
    public static TestDataProvider getInstance() {
        if (instance == null) {
            instance = new TestDataProvider();
        }
        return instance;
    }

    private TestDataProvider() {
        testDataProperties = loadConfigAndSetAsSystemProperties(FOLDER_TESTCONFIG_PROPERTIES, FILE_TESTDATA_PROPERTIES, false);
        testDataProperties.putAll(loadConfigAndSetAsSystemProperties(FOLDER_TESTDATA_PROPERTIES, FILE_TESTDATA_PROPERTIES, false));
        testDataProperties.putAll(loadConfigAndSetAsSystemProperties("", FILE_ALLURETEXTE_PROPERTIES, true));
        testDataProperties.putAll(loadConfigAndSetAsSystemProperties(FOLDER_TESTCONFIG_PROPERTIES, FILE_ALLURETEXTE_PROPERTIES, false));
    }

    public static String getTestData(String key) {
        try {
            return getInstance().getTestDataProperties().get(key).toString();
        } catch (NullPointerException e) {
            log.info("TestDataProvider.getTestData: NullPointer, key:("+key+") and returns \"\"");
            return "";
        }
    }
    public static String getTestData(AllureTextEnum key) {
        try {
            return getInstance().getTestDataProperties().get(key.getValue()).toString();
        } catch (NullPointerException e) {
            return "";
        }
    }

    private static Properties loadConfigAndSetAsSystemProperties(String folder, String propertieFileName, boolean fromToolkit) {
        Properties props = loadConfig(folder, propertieFileName, fromToolkit);

        for (Object o : props.entrySet()) {
            Map.Entry entry = (Map.Entry) o;
            System.setProperty((String) entry.getKey(), (String) entry.getValue());
        }

        return props;
    }

    private static Properties loadConfig(String folder, String propertieFileName, boolean fromToolkit) {

        String defaultProps = folder + propertieFileName;
        Properties properties = loadProperties(defaultProps, fromToolkit);
        String filename = folder + System.getProperties().getProperty(USER_NAME) + "." + propertieFileName;
        Properties userProperties = loadProperties(filename, fromToolkit);

        if (userProperties != null) {
            properties.putAll(userProperties);
        }

        return properties;
    }

    private static Properties loadProperties(String name, boolean fromToolkit) {
        ClassLoader loader;
        if(fromToolkit) {
            loader = TestDataProvider.class.getClassLoader();
        } else {
            loader = Thread.currentThread().getContextClassLoader();
        }
        URL url = loader.getResource(name);
        Properties props = new Properties();
        if (url != null) {
            try {
                InputStream e = url.openStream();
                props.load(e);
            } catch (IOException e) {
                log.error("TestDataProvider.loadProperties faild. Exception-Message:" + e.getMessage());
            }
        }
        return props;
    }

    private Properties getTestDataProperties() {
        return testDataProperties;
    }
}
