package seleniumConsulting.ch.selenium.framework.driver;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;

import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Point;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.ie.InternetExplorerDriver;
import org.openqa.selenium.ie.InternetExplorerOptions;
import org.openqa.selenium.remote.CapabilityType;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.testng.SkipException;
import org.testng.util.Strings;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import seleniumConsulting.ch.selenium.framework.dataLoader.TestDataProvider;



public class WebDriverFactory {

    public static final String REMOTE_GRID_URL = "remote.grid.url";
    public static final String CAPABILITIES = "capabilities";
    public static final String BROWSER_NAME = "browser.name";
    public static final String BROWSER_VERSION = "browser.version";
    public static final String ENV = "env";
    public static final String FIREFOX = "firefox";
    public static final String CHROME = "chrome";
    public static final String INTERNET_EXPLORER = "internet explorer";
    public static final String COUNT_URL = "https://www.selenium-consulting.ch/usage.php";
    public static String ENV_LOCAL = "local";
    public static String TOOLKITCOUNT = "toolkitcount";


    public static WebDriver createDriver(String browserName, String browserVersion, Map<String, String> capibilitiesMetadata) {
        String env = System.getProperty(ENV);

        WebDriver driver;
        if (ENV_LOCAL.equalsIgnoreCase(env)) {
            switch (browserName) {
                case FIREFOX:
                    FirefoxOptions firefoxOptions = new FirefoxOptions();
                    firefoxOptions.setCapability("marionette", true);
                    driver = new FirefoxDriver(firefoxOptions);
                    break;

                case CHROME:
                    ChromeOptions chromeOptions = new ChromeOptions();
                    chromeOptions.setExperimentalOption("useAutomationExtension", false);
                    driver = new ChromeDriver(chromeOptions);
                    break;

                case INTERNET_EXPLORER:
                    DesiredCapabilities desiredCapabilities = DesiredCapabilities.internetExplorer();
                    desiredCapabilities.setCapability(InternetExplorerDriver.INTRODUCE_FLAKINESS_BY_IGNORING_SECURITY_DOMAINS, true);
                    desiredCapabilities.setCapability(InternetExplorerDriver.IGNORE_ZOOM_SETTING, true);
                    driver = new InternetExplorerDriver(desiredCapabilities);
                    break;

                default:
                    throw new RuntimeException("Browser " + browserName + " is not available on local platform ");
            }
        } else {
            // ENV = grid
            MutableCapabilities options;
            switch (browserName) {
                case "chrome":
                    options = new ChromeOptions();
                    break;

                case "firefox":
                    options = new FirefoxOptions();
                    break;

                case "internet explorer":
                    options = new InternetExplorerOptions();
                    break;

                    default:
                        throw new RuntimeException("Browser " + browserName + " is not supported in the Selenium toolkit");

            }

            options.setCapability(CapabilityType.BROWSER_VERSION, browserVersion);
            options.setCapability("acceptInsecureCerts", true);

            setConfigCapibilities(options);
            setWebTestAnnotationCapibilities(options, capibilitiesMetadata);

            driver = createRemoteDriver(System.getProperty(REMOTE_GRID_URL+"/wd/hub"), options);
            System.out.println(((RemoteWebDriver) driver).getCapabilities().toString());


        }

        setWindowSizeFullscreen(driver);

        return driver;

    }

    private static void setWebTestAnnotationCapibilities(MutableCapabilities capabilities, Map<String, String> capibilitiesMetadata) {

        if(capibilitiesMetadata != null && !capibilitiesMetadata.isEmpty()){
            capibilitiesMetadata.forEach((key, value) -> {
                if (value.equals("true") || value.equals("false")) {
                    capabilities.setCapability(key, Boolean.parseBoolean(value));
                } else {
                    capabilities.setCapability(key, value);
                }
            });
        }
    }

    private static void setConfigCapibilities(MutableCapabilities capabilities) {

        if(!Strings.isNullOrEmpty(TestDataProvider.getTestData(CAPABILITIES))) {
            Map<String, String> map = ImmutableMap.copyOf(Splitter.on(",")
                    .withKeyValueSeparator("=")
                    .split(TestDataProvider.getTestData(CAPABILITIES)));

            map.forEach((key, value) -> {
                if (value.equals("true") || value.equals("false")) {
                    capabilities.setCapability(key, Boolean.parseBoolean(value));
                } else {
                    capabilities.setCapability(key, value);
                }
            });
        }
    }


    private static RemoteWebDriver createRemoteDriver(String url, MutableCapabilities desiredCapabilities)
            throws SkipException {
        final RemoteWebDriver remoteWebDriver;
        try {
            remoteWebDriver = new RemoteWebDriver(new URL(url), desiredCapabilities);
            return remoteWebDriver;
        } catch (MalformedURLException e) {
            throw new RuntimeException("Unable to create Remotewebdriver for grid url:" + url+ " message:"+ e.getMessage());
        }
    }

    private static void setWindowSizeFullscreen(WebDriver driver) {
        driver.manage().window().setPosition(new Point(0, 0));
        driver.manage().window().maximize();
    }
}
