package seleniumConsulting.ch.selenium.framework.listeners.testng;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.openqa.selenium.remote.Augmentable;
import org.openqa.selenium.remote.Augmenter;
import org.openqa.selenium.remote.SessionId;
import org.testng.*;
import org.testng.internal.MethodInstance;
import org.testng.util.Strings;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Allure;
import io.qameta.allure.Attachment;
import io.qameta.allure.model.Link;
import io.qameta.allure.model.Parameter;
import seleniumConsulting.ch.selenium.framework.annotation.WebTest;
import seleniumConsulting.ch.selenium.framework.dataLoader.TestDataProvider;
import seleniumConsulting.ch.selenium.framework.driver.WebDriverFactory;
import seleniumConsulting.ch.selenium.framework.driver.WebDriverManager;
import seleniumConsulting.ch.selenium.framework.metadata.MetadataKey;

import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERNAME;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERVERSION;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataManager.getMetadata;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataManager.setMetadata;


public class WebTestAnnotationListener implements IMethodInterceptor, ITestListener {

    private static String KEY_BROWSER = "browser";
    private static String KEY_VERSION = "version";

    @Override
    public List<IMethodInstance> intercept(List<IMethodInstance> methods, ITestContext iTestContext) {
        List<IMethodInstance> result = new ArrayList<IMethodInstance>();
        for (IMethodInstance m : methods) {
            WebTest webTest = getWebTest(m);
            if(webTest != null) {
                result.addAll(generateMultipleBrowserTests(m));
            } else {
                String desc = m.getMethod().getRealClass().toString() + "." + m.getMethod().getMethodName();
                m.getMethod().setDescription(desc);
                result.add(m);
            }
        }
        return result;
    }

    private List<IMethodInstance> generateMultipleBrowserTests(IMethodInstance method){
        List<IMethodInstance> result = new ArrayList<>();
        WebTest webTest = getWebTest(method);
        String[] browserList = webTest.browser();

        if(browserList.length == 0){
            browserList = TestDataProvider.getTestData(KEY_BROWSER).split(",");
        }
        for (String browser : browserList) {
            IMethodInstance methodInstance = new MethodInstance(method.getMethod().clone());
            setMetadataOfTest(methodInstance, browser);
            result.add(methodInstance);
        }
        return result;
    }

    private void setMetadataOfTest(IMethodInstance methodInstance, String browser){
        WebTest webTest = getWebTest(methodInstance);
        Map<String, String> browserConfig = ImmutableMap.copyOf(Splitter.on("&")
                .withKeyValueSeparator("=")
                .split(browser));
        String desc = methodInstance.getMethod().getRealClass().toString() + "." + methodInstance.getMethod().getMethodName();
        desc += " B=" + browserConfig.get(KEY_BROWSER);
        desc += browserConfig.get(KEY_VERSION) != null ? " V=" + browserConfig.get(KEY_VERSION) : "";
        methodInstance.getMethod().setDescription(desc);

        setMetadata(methodInstance.getMethod(), BROWSERNAME, browserConfig.get(KEY_BROWSER));
        if (browserConfig.get(KEY_VERSION) != null) {
            setMetadata(methodInstance.getMethod(), BROWSERVERSION, browserConfig.get(KEY_VERSION));
        } else {
            setMetadata(methodInstance.getMethod(), BROWSERVERSION, "");
        }
        setMetadataOfConfig(methodInstance, webTest.capibilities(), MetadataKey.CAPABILITIES);

    }

    private void setMetadataOfConfig(IMethodInstance methodInstance, String[] config, String metadataKey){
        if(config.length > 0){
            Map<String, String> configMap =
                    Arrays.stream(config).collect(Collectors.toMap(c -> c.split("=")[0], c -> c.split("=")[1]));
            setMetadata(methodInstance, metadataKey, configMap);
        }
    }

    @Override
    public void onTestStart(ITestResult iTestResult) {
        if(isWebTest(iTestResult)) {
            WebDriverManager.createWebDriver(iTestResult.getMethod());
            if(WebDriverManager.getWebdriver() != null) {
                createVideoLink();
            }
        }
    }

    @Override
    public void onTestSuccess(ITestResult iTestResult) {
        if(isWebTest(iTestResult) && WebDriverManager.getWebdriver() != null) {
            WebDriverManager.getWebdriver().quit();
        }
    }

    @Override
    public void onTestFailure(ITestResult iTestResult) {
        if(isWebTest(iTestResult) && WebDriverManager.getWebdriver() != null) {
            takeScreenshot("Fail Screenshot");
            WebDriverManager.getWebdriver().quit();
        }
    }

    @Attachment(value = "{message}", type = "image/png")
    public byte[] takeScreenshot(String message) {
        TakesScreenshot takesScreenshot = (TakesScreenshot) (WebDriverManager.getWebdriver().getClass().isAnnotationPresent(Augmentable.class) ? (new Augmenter()).augment(WebDriverManager.getWebdriver()) : WebDriverManager.getWebdriver());
        return takesScreenshot.getScreenshotAs(OutputType.BYTES);
    }

    @Override
    public void onTestSkipped(ITestResult iTestResult) {

    }

    @Override
    public void onTestFailedButWithinSuccessPercentage(ITestResult iTestResult) {

    }

    @Override
    public void onStart(ITestContext iTestContext) {

    }

    @Override
    public void onFinish(ITestContext iTestContext) {

    }

    private void createVideoLink(){
        //TODO: ADD CAP from Metadata
        boolean isVideoActive = false;
        SessionId id = WebDriverManager.getWebdriverSessionId();
        if(!Strings.isNullOrEmpty(System.getProperty(MetadataKey.CAPABILITIES))) {
            Map<String, String> map = ImmutableMap.copyOf(Splitter.on(',')
                    .withKeyValueSeparator("=")
                    .split(System.getProperty(MetadataKey.CAPABILITIES)));
            if( map.get("e34:video") != null && map.get("e34:video").equals("true")){
                isVideoActive = true;
            }
        }
        if(id != null && isVideoActive) {
            //TODO: Config for VideoLink
            Allure.addLinks(new Link()
                    .setName("Selenium Box Video")
                    .setUrl(System.getProperty(WebDriverFactory.REMOTE_GRID_URL)+"/videos/"+ WebDriverManager.getWebdriverSessionId() + ".mp4"));
        }
    }

    public static List<Parameter> addParameterIfWebTest(List<Parameter> parameters, ITestNGMethod method){
        if(isWebTest(method)){
            String browser = (String)getMetadata(method, BROWSERNAME);
            if(Strings.isNotNullAndNotEmpty(browser)) {
                parameters.add(new Parameter().withName("Browser").withValue(browser));
            }
            String version = (String)getMetadata(method, BROWSERVERSION);
            if(Strings.isNotNullAndNotEmpty(version)) {
                parameters.add(new Parameter().withName("Version").withValue(version));
            }
        }
        return parameters;
    }

    private static WebTest getWebTest(IMethodInstance methodInstance){
        return getWebTest(methodInstance.getMethod());
    }

    private static WebTest getWebTest(ITestResult testResult){
        return getWebTest(testResult.getMethod());
    }

    private static WebTest getWebTest(ITestNGMethod method){
        return method.getConstructorOrMethod().getMethod().getAnnotation(WebTest.class);
    }

    public static boolean isWebTest(ITestResult testResult){
        return getWebTest(testResult) != null;
    }

    public static boolean isWebTest(ITestNGMethod method){
        return getWebTest(method) != null;
    }

    public static boolean isWebTest(IMethodInstance methodInstance){
        return getWebTest(methodInstance) != null;
    }
}
