package seleniumConsulting.ch.selenium.framework.listeners.testng;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.openqa.selenium.remote.Augmentable;
import org.openqa.selenium.remote.Augmenter;
import org.openqa.selenium.remote.SessionId;
import org.testng.*;
import org.testng.annotations.Test;
import org.testng.annotations.TestUtil;
import org.testng.internal.MethodInstance;
import org.testng.util.Strings;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Attachment;
import io.qameta.allure.model.Parameter;
import lombok.extern.slf4j.Slf4j;
import seleniumConsulting.ch.selenium.framework.dataLoader.TestDataProvider;
import seleniumConsulting.ch.selenium.framework.driver.WebDriverManager;
import seleniumConsulting.ch.selenium.framework.metadata.MetadataKey;
import seleniumConsulting.ch.selenium.framework.metadata.MetadataManager;
import vendors.grid.VendorProvider;

import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERNAME;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataKey.BROWSERVERSION;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataManager.getMetadata;
import static seleniumConsulting.ch.selenium.framework.metadata.MetadataManager.setMetadata;

@Slf4j
/**
 * This Class is an {@link IMethodInstance} and clones Tests by multiple browser tests and assigne every Test one Browser.
 * It's also called from {@link AllureTestNg} onTestStart/onTestSuccess/onTestFailed to handle the {@link WebTest}-Annotation and create or quite Browsers
 */
public class WebTestAnnotationListener implements IMethodInterceptor {

    /**
     * Definition of testconfig/data.properties browser-Key
     */
    private static String KEY_BROWSER = "browser";

    /**
     * Definition of testconfig/data.properties browserversion-Key
     */
    private static String KEY_VERSION = "version";

    /**
     * This Method will intercept into TestNG an clone Tests if there is a {@link Test} Annotation and multiple Browsers configured.
     * It stores for every Method the right Browser in the {@link seleniumConsulting.ch.selenium.framework.metadata.MetadataManager}
     * @param methods List of all Tests
     * @param iTestContext
     * @return List of all methods with clones
     */
    @Override
    public List<IMethodInstance> intercept(List<IMethodInstance> methods, ITestContext iTestContext) {
        List<IMethodInstance> result = new ArrayList<IMethodInstance>();
        for (IMethodInstance m : methods) {
            if (TestUtil.isWebTest(m)) {
                result.addAll(generateMultipleBrowserTests(m));
            } else {
                String desc = m.getMethod().getRealClass().toString() + "." + m.getMethod().getMethodName();
                m.getMethod().setDescription(desc);
                result.add(m);
            }
        }
        return result;
    }

    /**
     * Generate the multiple Browser Clones. Read Config for Browser from {@link TestDataProvider}
     * @param method to check and clone
     * @return List of all new Tests
     */
    private List<IMethodInstance> generateMultipleBrowserTests(IMethodInstance method) {
        List<IMethodInstance> result = new ArrayList<>();
        Test test = TestUtil.getTestAnnotation(method);
        String[] browserList = test.browser();

        if (browserList.length == 0) {
            browserList = TestDataProvider.getTestData(KEY_BROWSER).split(",");
        }
        for (String browser : browserList) {
            IMethodInstance methodInstance = new MethodInstance(method.getMethod().clone());
            setMetadataOfTest(methodInstance, browser);
            result.add(methodInstance);
        }
        return result;
    }

    /**
     * Set the Metadatas (Browser and Version) in the {@link seleniumConsulting.ch.selenium.framework.metadata.MetadataManager}
     * @param methodInstance of Test
     * @param browser config to split into browser and version
     */
    private void setMetadataOfTest(IMethodInstance methodInstance, String browser) {
        Test test = TestUtil.getTestAnnotation(methodInstance);
        Map<String, String> browserConfig = ImmutableMap.copyOf(Splitter.on("&")
                .withKeyValueSeparator("=")
                .split(browser));
        String desc = methodInstance.getMethod().getRealClass().toString() + "." + methodInstance.getMethod().getMethodName();
        desc += " B=" + browserConfig.get(KEY_BROWSER);
        desc += browserConfig.get(KEY_VERSION) != null ? " V=" + browserConfig.get(KEY_VERSION) : "";
        methodInstance.getMethod().setDescription(desc);

        setMetadata(methodInstance.getMethod(), BROWSERNAME, browserConfig.get(KEY_BROWSER));
        if (browserConfig.get(KEY_VERSION) != null) {
            setMetadata(methodInstance.getMethod(), BROWSERVERSION, browserConfig.get(KEY_VERSION));
        } else {
            setMetadata(methodInstance.getMethod(), BROWSERVERSION, "");
        }
        setMetadataOfConfig(methodInstance, test.capibilities(), MetadataKey.CAPABILITIES);

    }

    /**
     * Store Metadata info {@link seleniumConsulting.ch.selenium.framework.metadata.MetadataManager}
     * @param methodInstance of Test
     * @param config to store, will bi splitted by '='
     * @param metadataKey key of Metadata in {@link seleniumConsulting.ch.selenium.framework.metadata.MetadataManager}
     */
    private void setMetadataOfConfig(IMethodInstance methodInstance, String[] config, String metadataKey) {
        if (config.length > 0) {
            Map<String, String> configMap =
                    Arrays.stream(config).collect(Collectors.toMap(c -> c.split("=")[0], c -> c.split("=")[1]));
            MetadataManager.setMetadata(methodInstance, metadataKey, configMap);
        }
    }

    /**
     * onTestStart called by {@link AllureTestNg} in onTestStart to create new WebDriver if the Test has {@link Test}-Annotation
     * @param iTestResult of test
     */
    public static void onTestStart(ITestResult iTestResult) {
        if (TestUtil.isWebTest(iTestResult) && iTestResult.getThrowable() == null) {
            WebDriverManager.createWebDriver(iTestResult.getMethod());
            VendorProvider.getVendor().onTestStart(iTestResult);
        }
    }

    /**
     * onTestSuccess called by {@link AllureTestNg} in onTestSuccess to create VideoLink/Download and quite Browser if the Test has {@link Test}-Annotation
     * @param iTestResult of test
     */
    public static void onTestSuccess(ITestResult iTestResult) {
        if (TestUtil.isWebTest(iTestResult) && WebDriverManager.getWebdriver() != null) {
            SessionId id = WebDriverManager.getWebdriverSessionId();
            VendorProvider.getVendor().onTestSuccess(iTestResult);
            WebDriverManager.quitWebDriver(iTestResult.getMethod());
            VendorProvider.getVendor().createVideo(iTestResult, id);

        }
    }

    /**
     * onTestFailure called by {@link AllureTestNg} in onTestFailure to create VideoLink/Download and quite Browser if the Test has {@link Test}-Annotation
     * @param iTestResult of test
     */
    public static void onTestFailure(ITestResult iTestResult) {
        if (TestUtil.isWebTest(iTestResult) && WebDriverManager.getWebdriver() != null) {
            takeScreenshot("Fail Screenshot");
            SessionId id = WebDriverManager.getWebdriverSessionId();
            VendorProvider.getVendor().onTestFailure(iTestResult);
            WebDriverManager.quitWebDriver(iTestResult.getMethod());
            VendorProvider.getVendor().createVideo(iTestResult, id);

        }
    }

    /**
     * onTestSkipped called by {@link AllureTestNg} in onTestSkipped to quite Browser if the Test has {@link Test}-Annotation
     * @param iTestResult of test
     */
    public static void onTestSkipped(ITestResult iTestResult) {
        if (TestUtil.isWebTest(iTestResult) && WebDriverManager.getWebdriver() != null) {
            VendorProvider.getVendor().onTestSkipped(iTestResult);
            WebDriverManager.quitWebDriver(iTestResult.getMethod());

        }
    }

    /**
     *  Takes a Screenshor and add it to Allure Report
     * @param message is shown in Report as Screenshot-Name
     * @return byte[] to store image
     */
    @Attachment(value = "{message}", type = "image/png")
    public static byte[] takeScreenshot(String message) {
        TakesScreenshot takesScreenshot = (TakesScreenshot) (WebDriverManager.getWebdriver().getClass().isAnnotationPresent(Augmentable.class) ? (new Augmenter()).augment(WebDriverManager.getWebdriver()) : WebDriverManager.getWebdriver());
        return takesScreenshot.getScreenshotAs(OutputType.BYTES);
    }


    /**
     * Read Metadata out of {@link MetadataManager} to the method and adds the Metadata to the parameters
     * @param parameters will be filled up with the additional parameters
     * @param method to get Metadata
     * @return List&lt;{@link Parameter}&gt; new parammeter-list
     */
    public static List<Parameter> addParameterIfWebTest(List<Parameter> parameters, ITestNGMethod method) {
        if (TestUtil.isWebTest(method)) {
            String browser = (String) getMetadata(method, BROWSERNAME);
            if (Strings.isNotNullAndNotEmpty(browser)) {
                parameters.add(new Parameter().withName("Browser").withValue(browser));
            }
            String version = (String) getMetadata(method, BROWSERVERSION);
            if (Strings.isNotNullAndNotEmpty(version)) {
                parameters.add(new Parameter().withName("Version").withValue(version));
            }
        }
        return parameters;
    }

}