package seleniumConsulting.ch.selenium.framework.listeners.webdriver;

import java.text.MessageFormat;
import java.util.Arrays;

import org.openqa.selenium.*;
import org.openqa.selenium.remote.Augmentable;
import org.openqa.selenium.remote.Augmenter;
import org.openqa.selenium.support.events.AbstractWebDriverEventListener;

import io.qameta.allure.Attachment;
import io.qameta.allure.model.Parameter;
import seleniumConsulting.ch.selenium.framework.allure.AllureTextEnum;
import seleniumConsulting.ch.selenium.framework.allure.AllureUtils;
import seleniumConsulting.ch.selenium.framework.dataLoader.TestDataProvider;
import seleniumConsulting.ch.selenium.framework.driver.WebDriverManager;

/**
 * Is a WebDriverEventListener that listens to the Selenium commands and create automatically TestSteps to Allure Report
 */
public class WebDriverEventListener extends AbstractWebDriverEventListener {

    /**
     * Definition of Value-Attribute
     */
    public static final String VALUE = "value";

    /**
     * Definition of Type-Attribute
     */
    public static final String TYPE = "type";

    /**
     * Definition of Password-Type-Attribute value
     */
    public static final String TYPE_PASSWORD = "password";

    /**
     * To store the By Selector
     */
    By by;

    /**
     * To store Value-Change message
     */
    String message = "";

    /**
     * BeforeClickOn-Listener to start Teststep, highlight Element to click, create screenshot and remove highlight
     * @param element which should be clicked
     * @param driver of Browser
     */
    @Override
    public void beforeClickOn(WebElement element, WebDriver driver) {
        AllureUtils.startStep(MessageFormat.format(TestDataProvider.getTestData(AllureTextEnum.Click), by.toString()), new Parameter().setName("element").setValue(element.toString()));
        addHighlightElement(element, "Click");
        takeScreenshot(TestDataProvider.getTestData(AllureTextEnum.ScreenshotClick));
        removeHighlightElement();
    }

    /**
     * AfterClickOn-Listener to stop Teststep
     * @param element which was clicked
     * @param driver of Browser
     */
    @Override
    public void afterClickOn(WebElement element, WebDriver driver) {
        AllureUtils.stopStepPassed();
    }

    /**
     * BeforeFindBy-Listener to save the By Value
     * @param by to find Element
     * @param element which should be found
     * @param driver of Browser
     */
    @Override
    public void beforeFindBy(By by, WebElement element, WebDriver driver) {
        this.by = by;
    }

    /**
     * AfterChangeValueOf-Listener to save Log Message and start TestStep
     * @param element which value was changed
     * @param driver of Browser
     * @param keysToSend which was sended
     */
    @Override
    public void beforeChangeValueOf(WebElement element, WebDriver driver, CharSequence[] keysToSend) {
        if(element.getTagName().equalsIgnoreCase("input")) {
            String oldValue = element.getAttribute(VALUE);
            String newValue = Arrays.toString(keysToSend);
            if (element.getAttribute(TYPE).equals(TYPE_PASSWORD)) {
                oldValue = "**********";
                newValue = "**********";
            }
            message = MessageFormat.format(TestDataProvider.getTestData(AllureTextEnum.ValueChange), oldValue, newValue);
            AllureUtils.startStep(message, new Parameter().setName("element").setValue(element.toString()));
        } else {

            message = MessageFormat.format(TestDataProvider.getTestData(AllureTextEnum.SendKeys), keysToSend);
            AllureUtils.startStep(message, new Parameter().setName("element").setValue(element.toString()));
        }
    }

    /**
     * AfterChangeValueOf-Listener to highlight changed Element, create a Screenshot, remove highlight and end TestStep
     * @param element to send the keys
     * @param driver of Browser
     * @param keysToSend (CharSequence[])
     */
    @Override
    public void afterChangeValueOf(WebElement element, WebDriver driver, CharSequence[] keysToSend) {
        addHighlightElement(element, message);
        takeScreenshot(TestDataProvider.getTestData(AllureTextEnum.ScreenshotValueChange));
        removeHighlightElement();
        AllureUtils.stopStepPassed();
        message = "";
    }

    /**
     * Create Highlight to Element in driver and adds message to the Highlight
     * @param element to highlight
     * @param message shown by the highlight
     */
    private void addHighlightElement(WebElement element, String message) {
        //TODO: TimeTracking with Breakpoint
        ((JavascriptExecutor) WebDriverManager.getWebdriver()).executeScript(
                "var node = document.createElement(\"div\");\n" +
                        "var childnode = document.createElement(\"div\");\n" +
                        "var textnode = document.createElement(\"span\");\n" +
                        "node.setAttribute('id','seleniumHighlight');\n" +
                        "node.style.cssText = 'position: absolute;z-index: 100000;width: '+arguments[2]+'px;top:'+arguments[1]+'px;left: '+arguments[0]+'px';\n" +
                        "textnode.style.cssText = 'white-space: nowrap;background: rgba(81, 203, 238, 1) !important';\n" +
                        "childnode.style.cssText = 'box-shadow: 0 0 20px rgba(81, 203, 238, 0.5) !important;height: '+arguments[3]+'px;border: 3px solid rgba(81, 203, 238, 0.5) !important;background: rgba(81, 203, 238, 0.5) !important;'\n" +
                        "textnode.textContent = arguments[5]+arguments[4];\n" +
                        "node.appendChild(textnode);\n" +
                        "node.appendChild(childnode);\n" +
                        "document.body.appendChild(node);",
                element.getLocation().x - 10, element.getLocation().y - 12, element.getSize().width+20, element.getSize().height+10, by.toString(), message.isEmpty()?"": message+ ": ");
        // set timeout and remove it
    }

    /**
     * Remove Highlight from highlighted Element in {@link WebDriver}
     */
    private void removeHighlightElement() {
        ((JavascriptExecutor) WebDriverManager.getWebdriver()).executeScript(
                "document.getElementById('seleniumHighlight') != undefined ? document.getElementById('seleniumHighlight').parentNode.removeChild(document.getElementById('seleniumHighlight')) : ''");
    }

    /**
     * Take Screenshot and add it to Allure as Attachment
     * @param message name of Image
     * @return byte[] to store Attachment
     */
    @Attachment(value = "{message}", type = "image/png")
    private byte[] takeScreenshot(String message) {
        TakesScreenshot takesScreenshot = (TakesScreenshot) (WebDriverManager.getWebdriver().getClass().isAnnotationPresent(Augmentable.class) ? (new Augmenter()).augment(WebDriverManager.getWebdriver()) : WebDriverManager.getWebdriver());
        return takesScreenshot.getScreenshotAs(OutputType.BYTES);
    }

}
