/*
 * Decompiled with CFR 0.152.
 */
package ace.data.file.handling.reader;

import ace.core.events.DatabaseInitiatedEvent;
import ace.core.events.FailedToLoadFileEvent;
import ace.data.file.handling.reader.IFileReader;
import ace.data.file.handling.reader.VariablesGlossary;
import ace.database.IDataContentRepository;
import ace.database.entities.Variable;
import ace.database.entities.VariableType;
import ace.database.entities.content.BlockContent;
import ace.database.entities.content.DataContent;
import ace.database.entities.content.FloatContent;
import ace.database.entities.content.IdContent;
import ace.database.entities.content.IntContent;
import ace.database.entities.content.StreamContent;
import ace.database.entities.content.UTF16Content;
import ace.database.entities.content.UTF8Content;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Component;

@Component
public class DatabaseFileReader
implements IFileReader<ByteBuffer> {
    private static final Logger log = LoggerFactory.getLogger(DatabaseFileReader.class);
    private static final byte[] BEGIN_BLOCK = new byte[]{11, 0, 0, 0, 98, 101, 103, 105, 110, 95, 98, 108, 111, 99, 107, -50, -6, 29, -80};
    private static final byte[] END_BLOCK = new byte[]{9, 0, 0, 0, 101, 110, 100, 95, 98, 108, 111, 99, 107, -34, -64, -83, -34};
    private ByteBuffer byteBuffer;
    @Autowired
    private IDataContentRepository contentRepository;
    @Autowired
    private ApplicationEventPublisher eventPublisher;

    @Override
    public ByteBuffer loadFile(String filePath) {
        log.info("Started parsing file...");
        this.contentRepository.deleteAll();
        this.readFile(filePath);
        this.loadFileIntoDatabase(this.getRawData());
        log.info("Database initiated");
        this.eventPublisher.publishEvent((ApplicationEvent)new DatabaseInitiatedEvent(this));
        log.info("Finished parsing file");
        return this.getRawData();
    }

    @Override
    public ByteBuffer getRawData() {
        return this.byteBuffer;
    }

    private void readFile(String filePath) {
        try {
            File f = new File(filePath);
            FileInputStream fis = new FileInputStream(f);
            this.prepareByteBuffer((int)f.length());
            this.readData(fis);
            fis.close();
        }
        catch (IOException e) {
            log.error("Error while reading file {}", (Object)e.getMessage());
            throw new RuntimeException(e.getCause());
        }
    }

    private void loadFileIntoDatabase(ByteBuffer rawData) {
        while (rawData.remaining() > 0) {
            String variableName = this.readUTF8(rawData);
            this.contentRepository.saveAndFlush(this.parseVariable(rawData, variableName));
        }
    }

    private void prepareByteBuffer(int size) {
        this.byteBuffer = ByteBuffer.allocate(size);
        this.byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    }

    private <E extends InputStream> void readData(E dataStream) {
        try {
            this.byteBuffer.put(dataStream.readAllBytes()).rewind();
        }
        catch (IOException e) {
            throw new RuntimeException(e.getCause());
        }
    }

    private DataContent<?> parseVariable(ByteBuffer data, String variableName) {
        VariableType variableType = VariablesGlossary.lookupVariable(variableName);
        if (variableType == null) {
            log.error("Could not find variableName '{}' in the glossary", (Object)variableName);
            this.eventPublisher.publishEvent((ApplicationEvent)new FailedToLoadFileEvent(this));
            throw new RuntimeException("variableName not in glossary");
        }
        Variable variable = new Variable(variableName, variableType);
        return this.parseDataContent(data, variable);
    }

    private DataContent<?> parseDataContent(ByteBuffer data, Variable variable) {
        switch (variable.getVariableType()) {
            case INT: {
                return new IntContent(variable, data.getInt());
            }
            case FLOAT: {
                return new FloatContent(variable, Float.valueOf(data.getFloat()));
            }
            case UTF8: {
                return new UTF8Content(variable, this.readUTF8(data));
            }
            case UTF16: {
                return new UTF16Content(variable, this.readUTF16(data));
            }
            case ID: {
                byte[] id = new byte[16];
                data.get(id, 0, 16);
                return new IdContent(variable, id);
            }
            case STREAM: {
                return new StreamContent(variable, this.readByteData(data));
            }
            case BLOCK: {
                data.getInt();
                return new BlockContent(variable, variable.getName().equals("begin_block") ? BEGIN_BLOCK : END_BLOCK);
            }
        }
        log.error("Invalid variable type '{}'", (Object)variable.getVariableType());
        throw new RuntimeException("Invalid variable type");
    }

    private String readUTF8(ByteBuffer data) {
        return new String(this.readBytes(data, false));
    }

    private String readUTF16(ByteBuffer data) {
        return new String(this.readBytes(data, true));
    }

    private byte[] readBytes(ByteBuffer data, boolean doubleLen) {
        int dataSize = data.getInt();
        if (dataSize > data.remaining()) {
            log.error("dataSize '{}' is more than what's left in the buffer '{}'", (Object)dataSize, (Object)data.remaining());
            this.eventPublisher.publishEvent((ApplicationEvent)new FailedToLoadFileEvent(this));
            throw new RuntimeException("Insufficient data left in file we're trying to load");
        }
        if (doubleLen) {
            dataSize *= 2;
        }
        byte[] dataContent = new byte[dataSize];
        data.get(dataContent, 0, dataSize);
        return dataContent;
    }

    private byte[] readByteData(ByteBuffer data) {
        int dataSize = data.getInt();
        byte[] dataContent = new byte[dataSize];
        data.get(dataContent, 0, dataSize);
        return dataContent;
    }
}

