package com.webapp.utils.spring;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@Component
public class CtxBeanUtils implements ApplicationContextAware {

	private static final Logger logger = LoggerFactory.getLogger(CtxBeanUtils.class);
	private static ApplicationContext ctx = null;

	public static <T> void addBean(Class<T> clz){
		ConfigurableApplicationContext cactx = (ConfigurableApplicationContext) ctx;
		DefaultListableBeanFactory bf = (DefaultListableBeanFactory)cactx.getBeanFactory();
		BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(clz);

		bf.registerBeanDefinition(StringUtils.uncapitalize(clz.getSimpleName()), builder.getRawBeanDefinition());

	}

    public void setApplicationContext(ApplicationContext context) throws BeansException {
		ctx = context;

	    try {
	    	MutablePropertySources mps = CtxPropsResolver.initPropertySources(true);
	    	CtxPropsResolver.resolverPropertySourcesPlaceholder(ctx, mps);
	    	CtxPropsResolver.resolverPropertyPlaceholder(ctx, mps);
	    	CtxPropsResolver.resolverPropertyFactory(ctx, mps);
	    	CtxPropsResolver.inject(ctx, mps);
        } catch (Exception e) {
	        logger.error("Error collecting property", e);
        }
    }

	public static <T> T getBean(Class<T> clz) {
		checkAppCtx();
		return ctx.getBean(clz);
    }

	public static Object getBean(String name) {
		checkAppCtx();
		return ctx.getBean(name);
    }

	public static Object getBeanTarget(String name) {
		checkAppCtx();
		return AopTargetUtils.getTarget(ctx.getBean(name));
    }

	public static void showBean() {
		checkAppCtx();
		String[] beans = ctx.getBeanDefinitionNames();
		Arrays.parallelSort(beans);

		List<String> full = new ArrayList<>();
		int i=0;
		for(String bean : beans){
			if(bean.contains(".")) {
				full.add(bean);
				continue;
			}
			System.out.println(++i + " " + bean);
			System.out.println("\t|--" + ctx.getBean(bean).getClass());
		}
		System.out.println("-------------------------------------------");
		for(String bean : full){
			System.out.println(++i + " " + bean);
			System.out.println("\t|--" + ctx.getBean(bean).getClass());
		}
	}

	public static ApplicationContext getAppCtx() {
		checkAppCtx();
		return ctx;
	}

	public static void cleanAppCtx() {
		ctx = null;
	}

	private static void checkAppCtx() {
		if (ctx == null) {
			throw new IllegalStateException("applicaitonContext is null");
		}
	}


}
