package cdc.util.data.paths;

import java.util.regex.Pattern;

// ITEM ::= YYY | YYY[@YYY='ZZZ'] | @XXX | . | ..
// PATH ::= ITEM (/ITEM)*
/**
 * Elementary parts that constitute a Path.
 * <p>
 * A Part has a type.
 * <ul>
 * <li>element
 * <li>attribute
 * <li>text
 * <li>comment
 * </ul>
 * This is mainly used to compute statistics.
 *
 * comment|text|element+|element+attribute|element+comment|element+text
 *
 * element*(text|comment)|element+attribute
 *
 * @author Damien Carbonne
 *
 */
public class Part {
    private static final Pattern DOT_PATTERN = Pattern.compile("^.$");
    private static final Pattern DOT_DOT_PATTERN = Pattern.compile("^..$");
    private static final Pattern ELEMENT_PATTERN = Pattern.compile("^[\\p{Alnum}_ -]+$");
    private static final Pattern ATTRIBUTE_PATTERN = Pattern.compile("^@[\\p{Alnum}_ -]+$");
    private static final Pattern SELECTOR_PATTERN = Pattern.compile("^[\\p{Alnum}_-]+\\[@[\\p{Alnum}_-]+='[\\p{Alnum}_ -]+'\\]$");

    public static final Part DOT = new Part(".");
    public static final Part DOT_DOT = new Part("..");

    /**
     * Enumeration of possible part types.
     *
     * @author Damien Carbonne
     *
     */
    public enum Type {
        /**
         * The part designates an attribute.
         */
        ATTRIBUTE,

        /**
         * The part designates a named element.
         */
        ELEMENT,

        /**
         * The part designates an element selector, using an attribute.
         */
        SELECTOR, // ATTRIBUTE_SELECTOR

        // ELEMENT_INDEX_SELECTOR
        // TEXT
        // COMMENT
        DOT,
        DOT_DOT
    }

    private final Part.Type type;
    private final String name;

    public Part(String name) {
        this.type = getType(name);
        this.name = name;
        if (type == null) {
            throw new IllegalArgumentException("Invalid part name:'" + name + "'");
        }
    }

    public static Part.Type getType(String name) {
        if (ELEMENT_PATTERN.matcher(name).find()) {
            return Type.ELEMENT;
        } else if (ATTRIBUTE_PATTERN.matcher(name).find()) {
            return Type.ATTRIBUTE;
        } else if (SELECTOR_PATTERN.matcher(name).find()) {
            return Type.SELECTOR;
        } else if (DOT_PATTERN.matcher(name).find()) {
            return Type.DOT;
        } else if (DOT_DOT_PATTERN.matcher(name).find()) {
            return Type.DOT_DOT;
        } else {
            return null;
        }
    }

    public Part.Type getType() {
        return type;
    }

    public String getName() {
        return name;
    }

    public String getElementName() {
        if (type == Type.ELEMENT) {
            return name;
        } else if (type == Type.SELECTOR) {
            return name.substring(0, name.indexOf('['));
        } else {
            return null;
        }
    }

    public String getAttributeName() {
        if (type == Type.ATTRIBUTE) {
            return name.substring(1);
        } else {
            return null;
        }
    }

    public String getSelectorName() {
        if (type == Type.SELECTOR) {
            final int start = name.indexOf('@') + 1;
            final int end = name.indexOf('=');
            return name.substring(start, end);
        } else {
            return null;
        }
    }

    public String getSelectorValue() {
        if (type == Type.SELECTOR) {
            final int start = name.indexOf('=') + 2;
            final int end = name.indexOf(']') - 1;
            return name.substring(start, end);
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return name;
    }
}