package cdc.util.data.paths;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

import cdc.util.data.Element;
import cdc.util.data.Parent;
import cdc.util.lang.Checks;
import cdc.util.lang.CollectionsUtil;

/**
 * Simple path.
 * <p>
 * It must have the form: {@code (part/)*part} or {@code (part(/part)*)@part}
 *
 * @author Damien Carbonne
 *
 */
public final class SPath {
    private final List<String> parts;
    private final boolean isAttribute;

    public SPath(String text) {
        final int index = text.lastIndexOf('@');
        this.isAttribute = index >= 0;

        final String[] tmp;
        if (index == 0) {
            tmp = new String[0];
        } else if (index > 0) {
            tmp = text.substring(0, index).split("/");
        } else {
            tmp = text.split("/");
        }
        final List<String> l = CollectionsUtil.toList(tmp);
        if (isAttribute) {
            final String last = text.substring(index + 1);
            Checks.isFalse(last.contains("/"), "An attribute name can not contain '/'");
            l.add(last);
        }
        this.parts = Collections.unmodifiableList(l);
    }

    public List<String> getParts() {
        return parts;
    }

    public boolean isAttribute() {
        return isAttribute;
    }

    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder();
        final int size = parts.size();
        for (int index = 0; index < size; index++) {
            if (index > 0) {
                if (index == size - 1 && isAttribute) {
                    builder.append('/');
                } else {
                    builder.append('@');
                }
            }
            builder.append(parts.get(index));
        }
        return builder.toString();
    }

    private boolean matches(Parent parent,
                            String name,
                            boolean attribute) {
        if (attribute != isAttribute || parts.isEmpty()) {
            return false;
        } else {
            if (Objects.equals(name, parts.get(parts.size() - 1))) {
                Parent p = parent;
                for (int index = parts.size() - 2; index > 0; index--) {
                    if (p instanceof Element) {
                        final Element e = (Element) p;
                        if (e.getName().equals(parts.get(index))) {
                            p = e.getParent();
                        } else {
                            return false;
                        }
                    } else {
                        return false;
                    }
                }
                return true;
            } else {
                return false;
            }
        }
    }

    public boolean matchesElement(Parent parent) {
        if (parent instanceof Element) {
            final Element element = (Element) parent;
            return matchesElement(element.getParent(), element.getName());

        } else {
            return false;
        }
    }

    public boolean matchesElement(Parent parent,
                                  String name) {
        return matches(parent, name, false);
    }

    public boolean matchesAttribute(Element parent,
                                    String name) {
        return matches(parent, name, true);
    }
}