package cdc.util.data.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import cdc.util.compress.Compressor;
import cdc.util.data.DataException;
import cdc.util.data.Element;
import cdc.util.data.xml.XmlDataReader;
import cdc.util.lang.FailureReaction;

/**
 * Helper class for data loading.
 *
 * @author Damien Carbonne
 *
 * @param <R> The return type.
 */
public abstract class AbstractResourceLoader<R> {
    private final Logger logger;
    private final FailureReaction reaction;

    public AbstractResourceLoader(FailureReaction reaction) {
        this.logger = LogManager.getLogger(getClass());
        this.reaction = reaction;
    }

    public final Logger getLogger() {
        return logger;
    }

    public final FailureReaction getReaction() {
        return reaction;
    }

    public final R loadXml(File file,
                           Compressor compressor,
                           XmlDataReader.Feature... features) throws IOException {
        logger.info("loadXml(" + file + ", " + compressor + ")");
        final Element root = XmlDataReader.loadRoot(file, compressor, features);
        return loadRoot(root);
    }

    public final R loadXml(File file,
                           XmlDataReader.Feature... features) throws IOException {
        return loadXml(file, Compressor.NONE, features);
    }

    public final R loadXml(String filename,
                           Compressor compressor,
                           XmlDataReader.Feature... features) throws IOException {
        logger.info("loadXml(" + filename + ", " + compressor + ")");
        final Element root = XmlDataReader.loadRoot(filename, compressor, features);
        return loadRoot(root);
    }

    public final R loadXml(String filename,
                           XmlDataReader.Feature... features) throws IOException {
        return loadXml(filename, Compressor.NONE, features);

    }

    public final R loadXml(URL url,
                           Compressor compressor,
                           XmlDataReader.Feature... features) throws IOException {
        logger.info("loadXml(" + url + ", " + compressor + ")");
        final Element root = XmlDataReader.loadRoot(url, compressor, features);
        return loadRoot(root);
    }

    public final R loadXml(URL url,
                           XmlDataReader.Feature... features) throws IOException {
        return loadXml(url, Compressor.NONE, features);
    }

    public final void loadXml(final List<URL> urls,
                              Compressor compressor,
                              XmlDataReader.Feature... features) {
        for (final URL url : urls) {
            try {
                loadXml(url, compressor, features);
            } catch (final IOException e) {
                logger.catching(e);
            }
        }
    }

    public final void loadXml(final List<URL> urls,
                              XmlDataReader.Feature... features) {
        loadXml(urls, Compressor.NONE, features);
    }

    public final R loadXml(InputStream is,
                           String systemId,
                           XmlDataReader.Feature... features) throws IOException {
        logger.info("loadXml(" + systemId + ")");
        final Element root = XmlDataReader.loadRoot(is, systemId, features);
        return loadRoot(root);
    }

    protected abstract R loadRoot(Element root);

    protected final void onError(String message) {
        FailureReaction.onError(message, logger, reaction, DataException::new);
    }

    protected final <T> T onError(String message,
                                  T def) {
        return FailureReaction.onError(message, logger, reaction, def, DataException::new);
    }

    protected final <T> T onResult(T result,
                                   String message,
                                   T def) {
        return FailureReaction.onResult(result, message, logger, reaction, def, DataException::new);
    }

    protected final void unexpectedElement(Element element,
                                           String... expected) {
        unexpectedElement(element, null, expected);
    }

    protected final <V> V unexpectedElement(Element element,
                                            V def,
                                            String... expected) {
        if (expected == null || expected.length == 0) {
            return onError("Unexpected child " + element + " under " + element.getParent(), def);
        } else {
            return onError("Unexpected child " + element + " under " + element.getParent()
                    + " expected one of " + Arrays.toString(expected), def);
        }
    }
}