package cdc.util.data.util;

import java.util.Map;
import java.util.function.Function;

import cdc.util.data.Parent;
import cdc.util.data.paths.SPath;
import cdc.util.lang.Checks;

/**
 * Interface used to convert an element name.
 *
 * @author Damien Carbonne
 *
 */
@FunctionalInterface
public interface ElementNameConverter {
    /**
     * An element name converter that returns the name unchanged.
     */
    public static final ElementNameConverter IDENTITY = (p,
                                                         n) -> n;

    /**
     * Returns the element name to use instead of the original name.
     *
     * @param parent The element parent.
     * @param name The element name.
     * @return The name to use for the element, instead of name.
     */
    public String convertElementName(Parent parent,
                                     String name);

    /**
     * Returns a converter that first applies {@code other} then this converter.
     *
     * @param other The other converter.
     * @return A converter that first applies {@code other} then this converter.
     * @throws IllegalArgumentException When {@code other} is {@code null}.
     */
    public default ElementNameConverter compose(ElementNameConverter other) {
        Checks.isNotNull(other, "other");
        return (Parent parent,
                String name) -> convertElementName(parent, other.convertElementName(parent, name));
    }

    /**
     * Returns a converter that first applies this converter and then {@code other}.
     *
     * @param other The other converter.
     * @return A converter that first applies this converter and then {@code other}.
     * @throws IllegalArgumentException When {@code other} is {@code null}.
     */
    public default ElementNameConverter andThen(ElementNameConverter other) {
        Checks.isNotNull(other, "other");
        return (Parent parent,
                String name) -> other.convertElementName(parent, convertElementName(parent, name));
    }

    /**
     * Creates a new ElementNameConverter from a name converter function.
     * <p>
     * Conversion is independent of element.
     *
     * @param function The function used to convert elements names.
     * @return A new ElementNameConverter from {@code function}.
     * @throws IllegalArgumentException When {@code function} is {@code null}.
     */
    public static ElementNameConverter fromNameFunction(Function<String, String> function) {
        Checks.isNotNull(function, "function");
        return (Parent parent,
                String name) -> function.apply(name);
    }

    /**
     * Creates a new AttributeNameConverter from a map.
     * <p>
     * Conversion is independent of element.<br>
     * If a name is not mapped, then result is that name.
     *
     * @param map The map.
     * @return A new AttributeValueConverter from {@code map}.
     * @throws IllegalArgumentException When {@code map} is {@code null}.
     */
    public static ElementNameConverter fromNameMap(Map<String, String> map) {
        Checks.isNotNull(map, "map");
        return (Parent parent,
                String name) -> map.getOrDefault(name, name);
    }

    public static ElementNameConverter fromPathMap(Map<SPath, String> map) {
        Checks.isNotNull(map, "map");
        return (Parent parent,
                String name) -> {
            for (final Map.Entry<SPath, String> entry : map.entrySet()) {
                if (entry.getKey().matchesElement(parent, name)) {
                    return entry.getValue();
                }
            }
            return name;
        };
    }
}