package cdc.util.data.xml;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.Attributes;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.ext.DefaultHandler2;

import cdc.util.compress.CompressionUtil;
import cdc.util.compress.Compressor;
import cdc.util.data.Child;
import cdc.util.data.Comment;
import cdc.util.data.Document;
import cdc.util.data.Element;
import cdc.util.data.Parent;
import cdc.util.data.Text;
import cdc.util.data.util.AttributeNameConverter;
import cdc.util.data.util.AttributePredicate;
import cdc.util.data.util.AttributeValueConverter;
import cdc.util.data.util.ElementNameConverter;
import cdc.util.data.util.ElementPredicate;
import cdc.util.data.util.TextContentConverter;
import cdc.util.lang.Cache;
import cdc.util.lang.Checks;
import cdc.util.lang.IntMasks;
import cdc.util.lang.UnexpectedValueException;

/**
 * Class used to read an XML source and produce a Document.
 * <p>
 * It is possible, during loading, to:
 * <ul>
 * <li>filter (keep or remove) attributes
 * <li>convert attributes names
 * <li>convert attributes values
 * <li>filter (keep or remove) elements at creation time (pre) and when all its children are known (post).
 * <li>convert elements names
 * <li>ignore (remove) comments
 * <li>ignore (remove) spaces
 * </ul>
 * This can be used to create only necessary nodes and attributes in memory and adapt their names or content.
 *
 * @author Damien Carbonne
 *
 */
public class XmlDataReader {

    /**
     * The cache of strings used to share attributes and elements names.
     */
    public static final Cache<String> CACHE = new Cache<>(String.class, (s1,
                                                                         s2) -> s1.compareTo(s2));
    protected static final Logger LOGGER = LogManager.getLogger(XmlDataReader.class);
    private int features = 0;

    /**
     * The entity resolver.
     */
    protected EntityResolver entityResolver = null;

    /**
     * The filter attributes.
     * <p>
     * Only accepted attributes are kept and transformed.
     */
    protected AttributePredicate attributeFilter = AttributePredicate.ANY_ATTRIBUTE;

    /**
     * Attribute name converter.
     * <p>
     * Applied on accepted attributes.
     */
    protected AttributeNameConverter attributeNameConverter = AttributeNameConverter.IDENTITY;

    /**
     * Attribute value converter.
     * <p>
     * Applied on accepted attributes.
     */
    protected AttributeValueConverter attributeValueConverter = AttributeValueConverter.INDENTITY;

    /**
     * The filter to use when the element is created.
     */
    protected ElementPredicate elementPreFilter = ElementPredicate.ANY_ELEMENT;

    /**
     * The filter to use when all the children of an element are created.
     */
    protected ElementPredicate elementPostFilter = ElementPredicate.ANY_ELEMENT;

    /**
     * Element name converter.
     * <p>
     * Applied on elements that are pre accepted.
     */
    protected ElementNameConverter elementNameConverter = ElementNameConverter.IDENTITY;

    /**
     * Text content converter.
     */
    protected TextContentConverter textContentConverter = TextContentConverter.IDENTITY;

    private static final String FILTER = "filter";
    private static final String CONVERTER = "converter";

    public enum Feature {
        /**
         * If enabled, comments are loaded (as comment nodes).
         * <p>
         * They are ignored by default.
         */
        LOAD_COMMENTS,

        /**
         * If enabled, spaces are loaded (as text nodes).
         * <p>
         * They are ignored by default.
         */
        LOAD_SPACES,

        /**
         * If enabled, mixed content is allowed.
         */
        ALLOW_MIXED_CONTENT,

        /**
         * If enabled, attributes names are shared using an internal cache.
         * <p>
         * No sharing by default (SAX parsers may already do this).
         */
        SHARE_ATTRIBUTE_NAMES,

        /**
         * If enabled, element names are shared using an internal cache.
         * <p>
         * No sharing by default (SAX parsers may already do this).
         */
        SHARE_ELEMENT_NAMES,

        /**
         * If enabled, a dummy entity resolver is used.
         * <p>
         * This may be used to ignore DTD.<br>
         * However, if entities are used, result will be wrong.
         */
        DUMMY_ENTITY_RESOLVER
    }

    public XmlDataReader() {
        super();
    }

    /**
     * Returns {@code true} when a feature is enabled.
     *
     * @param feature The feature.
     * @return {@code true} if {@code feature} is enabled.
     */
    public boolean isEnabled(Feature feature) {
        return IntMasks.isEnabled(features, feature);
    }

    /**
     * Enables or disables a feature.
     *
     * @param feature The feature.
     * @param enabled If {@code true}, the feature is enabled. It is disabled orthewise.
     */
    public void setEnabled(Feature feature,
                           boolean enabled) {
        features = IntMasks.setEnabled(features, feature, enabled);
    }

    /**
     * @return The entity resolver.
     */
    public EntityResolver getEntityResolver() {
        return entityResolver;
    }

    /**
     * Sets the entity resolver.
     * <p>
     * <b>WARNING:</b> This has interactions with {@link Feature#DUMMY_ENTITY_RESOLVER}.
     *
     * @param resolver The entity resolver.
     */
    public void setEntityResolver(EntityResolver resolver) {
        this.entityResolver = resolver;
    }

    /**
     * @return The attribute filter.
     */
    public AttributePredicate getAttributeFilter() {
        return attributeFilter;
    }

    /**
     * Sets the attribute filter.
     * <p>
     * The name and value that are passed to the filter are the original name and value, before any name or value conversion happens.
     *
     * @param filter The filter.
     * @throws IllegalArgumentException When {@code filter} is null.
     */
    public void setAttributeFilter(AttributePredicate filter) {
        Checks.isNotNull(filter, FILTER);

        this.attributeFilter = filter;
    }

    /**
     * @return The attribute name converter.
     */
    public AttributeNameConverter getAttributeNameConverter() {
        return attributeNameConverter;
    }

    /**
     * Sets the attribute name converter.
     * <p>
     * The conversion is applied after attribute filtering.
     *
     * @param converter The converter.
     * @throws IllegalArgumentException When {@code converter} is null.
     */
    public void setAttributeNameConverter(AttributeNameConverter converter) {
        Checks.isNotNull(converter, CONVERTER);

        this.attributeNameConverter = converter;
    }

    /**
     * @return The attribute value converter.
     */
    public AttributeValueConverter getAttributeValueConverter() {
        return attributeValueConverter;
    }

    /**
     * Sets the attribute value converter.
     * <p>
     * The conversion is applied after attribute filtering.
     *
     * @param converter The converter.
     * @throws IllegalArgumentException When {@code converter} is null.
     */
    public void setAttributeValueConverter(AttributeValueConverter converter) {
        Checks.isNotNull(converter, CONVERTER);

        this.attributeValueConverter = converter;
    }

    /**
     * @return the element pre filter.
     */
    public ElementPredicate getElementPreFilter() {
        return elementPreFilter;
    }

    /**
     * Sets the element pre filter.
     * <p>
     * <b>WARNING:</b>
     * <ul>
     * <li>The element name is the original one (before any conversion happens).
     * <li>The attributes names and values are the converted ones (after all conversions have happened).
     * <li>Only remaining attributes are available.
     * </ul>
     *
     * @param filter The filter.
     * @throws IllegalArgumentException When {@code filter} is null.
     */
    public void setElementPreFilter(ElementPredicate filter) {
        Checks.isNotNull(filter, FILTER);

        this.elementPreFilter = filter;
    }

    /**
     * @return the element post filter.
     */
    public ElementPredicate getElementPostFilter() {
        return elementPostFilter;
    }

    /**
     * Sets the element post filter.
     * <p>
     * <b>WARNING:</b>
     * <ul>
     * <li>The element name is the converted one (after conversion has happened).
     * <li>The attributes names and values are the converted ones (after all conversions have happened).
     * <li>Only remaining attributes are available.
     * </ul>
     *
     * @param filter The filter.
     * @throws IllegalArgumentException When {@code filter} is null.
     */
    public void setElementPostFilter(ElementPredicate filter) {
        Checks.isNotNull(filter, FILTER);

        this.elementPostFilter = filter;
    }

    /**
     * @return The element name converter.
     */
    public ElementNameConverter getElementNameConverter() {
        return elementNameConverter;
    }

    /**
     * Sets the element name converter.
     *
     * @param converter The converter.
     * @throws IllegalArgumentException When {@code converter} is null.
     */
    public void setElementNameConverter(ElementNameConverter converter) {
        Checks.isNotNull(converter, CONVERTER);

        this.elementNameConverter = converter;
    }

    public TextContentConverter getTextContentConverter() {
        return textContentConverter;
    }

    public void setTextContentConverter(TextContentConverter converter) {
        Checks.isNotNull(converter, CONVERTER);

        this.textContentConverter = converter;
    }

    private static class DummyEntityResolver implements EntityResolver {
        public DummyEntityResolver() {
            super();
        }

        @Override
        public InputSource resolveEntity(String publicId,
                                         String systemId) throws SAXException, IOException {
            LOGGER.debug("resolveEntity('" + publicId + "', '" + systemId + "')");
            return new InputSource(new StringReader(""));
        }
    }

    private XMLReader configureReader(SAXParser parser,
                                      Handler handler) throws SAXException {
        final XMLReader reader = parser.getXMLReader();
        reader.setContentHandler(handler);
        reader.setErrorHandler(handler);
        reader.setDTDHandler(handler);
        if (isEnabled(Feature.LOAD_COMMENTS)) {
            reader.setProperty("http://xml.org/sax/properties/lexical-handler", handler);
        }
        // Set dummy entity resolver before user defined one
        if (isEnabled(Feature.DUMMY_ENTITY_RESOLVER)) {
            reader.setEntityResolver(new DummyEntityResolver());
        }
        if (getEntityResolver() != null) {
            if (isEnabled(Feature.DUMMY_ENTITY_RESOLVER)) {
                LOGGER.warn("Dummy entity resolver overwritten by user defined one");
            }
            reader.setEntityResolver(getEntityResolver());
        }
        return reader;
    }

    public Document read(InputStream is) throws IOException {
        LOGGER.debug("read(is=...)");
        final SAXParserFactory factory = SAXParserFactory.newInstance();
        final Handler handler = new Handler();
        try {
            final SAXParser parser = factory.newSAXParser();
            final XMLReader reader = configureReader(parser, handler);
            final InputSource source = new InputSource(is);
            reader.parse(source);
            return handler.getDocument();
        } catch (final ParserConfigurationException e) {
            LOGGER.trace(e);
        } catch (final SAXException e) {
            throw new IOException(e);
        }
        return null;
    }

    public Element readRoot(InputStream is) throws IOException {
        return Document.getRootElement(read(is));
    }

    /**
     * Reads an InputStream.
     *
     * @param is The InputStream.
     * @param systemId The systemId which is needed for resolving relative URIs.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(InputStream is,
                         String systemId) throws IOException {
        LOGGER.debug("read(is=..., " + systemId + ")");
        final SAXParserFactory factory = SAXParserFactory.newInstance();
        final Handler handler = new Handler();
        try {
            final SAXParser parser = factory.newSAXParser();
            final XMLReader reader = configureReader(parser, handler);
            final InputSource source = new InputSource(is);
            source.setSystemId(systemId);
            reader.parse(source);
            return handler.getDocument();
        } catch (final ParserConfigurationException e) {
            LOGGER.trace(e);
        } catch (final SAXException e) {
            throw new IOException(e);
        }
        return null;
    }

    /**
     * Reads an InputStream.
     *
     * @param is The InputStream.
     * @param systemId The systemId which is needed for resolving relative URIs.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(InputStream is,
                            String systemId) throws IOException {
        return Document.getRootElement(read(is, systemId));
    }

    /**
     * Reads a string.
     *
     * @param s The string.
     * @param charset The charset. Must be compliant with string content.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(String s,
                         Charset charset) throws IOException {
        return read(new ByteArrayInputStream(s.getBytes(charset)));
    }

    /**
     * Reads a string.
     *
     * @param s The string.
     * @param charset The charset. Must be compliant with string content.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(String s,
                            Charset charset) throws IOException {
        return Document.getRootElement(read(s, charset));
    }

    /**
     * Reads an URL.
     *
     * @param url The URL.
     * @param compressor The compressor used to compress file.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(URL url,
                         Compressor compressor) throws IOException {
        LOGGER.debug("read(url=" + url + ", " + compressor + ")");
        try (final InputStream is = CompressionUtil.adapt(url.openStream(), compressor)) {
            return read(is);
        }
    }

    /**
     * Reads an URL.
     *
     * @param url The URL.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(URL url) throws IOException {
        return read(url, Compressor.NONE);
    }

    /**
     * Reads an URL.
     *
     * @param url The URL.
     * @param compressor The compressor used to compress file.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(URL url,
                            Compressor compressor) throws IOException {
        return Document.getRootElement(read(url, compressor));
    }

    /**
     * Reads an URL.
     *
     * @param url The URL.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(URL url) throws IOException {
        return readRoot(url, Compressor.NONE);
    }

    /**
     * Reads a file.
     *
     * @param filename The file name.
     * @param compressor The compressor used to compress file.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(String filename,
                         Compressor compressor) throws IOException {
        LOGGER.debug("read(filename=" + filename + ", " + compressor + ")");
        try (InputStream is = new BufferedInputStream(CompressionUtil.adapt(new FileInputStream(filename), compressor))) {
            return read(is, filename);
        }
    }

    /**
     * Reads a file.
     *
     * @param filename The file name.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(String filename) throws IOException {
        return read(filename, Compressor.NONE);
    }

    /**
     * Reads a file.
     *
     * @param filename The file name.
     * @param compressor The compressor used to compress file.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(String filename,
                            Compressor compressor) throws IOException {
        return Document.getRootElement(read(filename, compressor));
    }

    /**
     * Reads a file.
     *
     * @param filename The file name.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(String filename) throws IOException {
        return readRoot(filename, Compressor.NONE);
    }

    /**
     * Reads a file.
     *
     * @param file The file.
     * @param compressor The compressor used to compress file.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(File file,
                         Compressor compressor) throws IOException {
        LOGGER.debug("read(file=" + file + ", " + compressor + ")");
        return read(file.getPath(), compressor);
    }

    /**
     * Reads a file.
     *
     * @param file The file.
     * @return The corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Document read(File file) throws IOException {
        return read(file.getPath(), Compressor.NONE);
    }

    /**
     * Reads a file.
     *
     * @param file The file.
     * @param compressor The compressor used to compress file.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(File file,
                            Compressor compressor) throws IOException {
        return Document.getRootElement(read(file, compressor));
    }

    /**
     * Reads a file.
     *
     * @param file The file.
     * @return The root element of the corresponding Document.
     * @throws IOException When an IO error occurs.
     */
    public Element readRoot(File file) throws IOException {
        return readRoot(file, Compressor.NONE);
    }

    public static XmlDataReader create(Feature... features) {
        final XmlDataReader reader = new XmlDataReader();
        for (final Feature feature : features) {
            reader.setEnabled(feature, true);
        }
        return reader;
    }

    public static Document load(InputStream is,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(is);
    }

    public static Element loadRoot(InputStream is,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(is);
    }

    public static Document load(InputStream is,
                                String systemId,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(is, systemId);
    }

    public static Element loadRoot(InputStream is,
                                   String systemId,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(is, systemId);
    }

    public static Document load(String s,
                                Charset charset,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(s, charset);
    }

    public static Element loadRoot(String s,
                                   Charset charset,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(s, charset);
    }

    public static Document load(URL url,
                                Compressor compressor,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(url, compressor);
    }

    public static Document load(URL url,
                                Feature... features) throws IOException {
        return load(url, Compressor.NONE, features);
    }

    public static Element loadRoot(URL url,
                                   Compressor compressor,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(url, compressor);
    }

    public static Element loadRoot(URL url,
                                   Feature... features) throws IOException {
        return loadRoot(url, Compressor.NONE, features);
    }

    public static Document load(String filename,
                                Compressor compressor,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(filename, compressor);
    }

    public static Document load(String filename,
                                Feature... features) throws IOException {
        return load(filename, Compressor.NONE, features);
    }

    public static Element loadRoot(String filename,
                                   Compressor compressor,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(filename, compressor);
    }

    public static Element loadRoot(String filename,
                                   Feature... features) throws IOException {
        return loadRoot(filename, Compressor.NONE, features);
    }

    public static Document load(File file,
                                Compressor compressor,
                                Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.read(file, compressor);
    }

    public static Document load(File file,
                                Feature... features) throws IOException {
        return load(file, Compressor.NONE, features);
    }

    public static Element loadRoot(File file,
                                   Compressor compressor,
                                   Feature... features) throws IOException {
        final XmlDataReader reader = create(features);
        return reader.readRoot(file, compressor);
    }

    public static Element loadRoot(File file,
                                   Feature... features) throws IOException {
        return loadRoot(file, Compressor.NONE, features);
    }

    private enum Action {
        KEEP,
        IGNORE
    }

    /**
     * Internal class used to load XML source.
     *
     * @author Damien Carbonne
     *
     */
    private class Handler extends DefaultHandler2 {
        private Locator locator = null;
        private Document document = null;
        private Parent currentParent = null;
        private Text activeText = null;

        /**
         * LIFO stack of actions.
         * <p>
         * It must match the parent hierarchy.<br>
         * This stack must contain: {@code KEEP*, IGNORE*}.<br>
         * The {@code currentParent} node corresponds to the top-most {@code KEEP}.
         */
        private final List<Action> actionStack = new ArrayList<>();

        public Handler() {
            super();
        }

        /**
         * @return The top action (the last added) or KEEP (if action stack is empty).
         */
        private Action top() {
            if (actionStack.isEmpty()) {
                return Action.KEEP;
            } else {
                return actionStack.get(actionStack.size() - 1);
            }
        }

        /**
         * Adds a,n action to the action stack.
         * <p>
         * If top action is IGNORE, then the added action will also be IGNORE.
         *
         * @param action The action.
         */
        private void pushAction(Action action) {
            switch (top()) {
            case KEEP:
                actionStack.add(action);
                break;
            case IGNORE:
                actionStack.add(Action.IGNORE);
                break;
            default:
                throw new UnexpectedValueException(top());
            }
        }

        /**
         * Removes the top (last added) action from the action stack.
         */
        private void popAction() {
            actionStack.remove(actionStack.size() - 1);
        }

        private void resetActiveText() {
            if (activeText != null) {
                activeText.setContent(textContentConverter.convertTextContent(activeText.getParent(), activeText.getContent()));
                activeText = null;
            }
        }

        /**
         * Removes active text if possible and necessary.
         *
         * @param preserve If {@code true} should be preserved if necessary.
         *
         * @throws SAXException When mixed content is found and is not allowed.
         */
        private void checkActiveText(boolean preserve) throws SAXException {
            if (activeText != null) {
                if (preserve && currentParent.getChildrenCount() == 1) {
                    // Preserve the text that is the only child of its parent (element)
                    resetActiveText();
                } else if (activeText.getContent().isEmpty()
                        || activeText.isIgnorable() && !isEnabled(Feature.LOAD_SPACES)) {
                    currentParent.removeChildAt(currentParent.getChildrenCount() - 1);
                    resetActiveText();
                } else if (currentParent.getChildrenCount() > 1 && !isEnabled(Feature.ALLOW_MIXED_CONTENT)) {
                    throw new SAXException("Mixed content not allowed");
                }
            }
        }

        public Document getDocument() {
            return document;
        }

        @Override
        public void setDocumentLocator(Locator locator) {
            this.locator = locator;
        }

        @Override
        public void startDocument() throws SAXException {
            LOGGER.trace("startDocument()");
            document = new Document();
            currentParent = document;
            resetActiveText();
        }

        @Override
        public void endDocument() throws SAXException {
            LOGGER.trace("endDocument()");
            // Ignore
        }

        @Override
        public void startPrefixMapping(String prefix,
                                       String uri) throws SAXException {
            // Ignore
        }

        @Override
        public void endPrefixMapping(String prefix) throws SAXException {
            // Ignore
        }

        private final boolean acceptsAttribute(Element element,
                                               String name,
                                               String value) {
            return attributeFilter.accepts(element, name, value);
        }

        private String shareAttributeName(String name) {
            if (isEnabled(Feature.SHARE_ATTRIBUTE_NAMES)) {
                return CACHE.get(name);
            } else {
                return name;
            }
        }

        private final String convertAttributeName(Element element,
                                                  String name) {
            return shareAttributeName(attributeNameConverter.convertAttributeName(element, name));
        }

        private final String convertAttributeValue(Element element,
                                                   String name,
                                                   String value) {
            return attributeValueConverter.convertAttributeValue(element, name, value);
        }

        private final boolean acceptsElementPre(Parent parent,
                                                Element element) {
            return elementPreFilter.accepts(parent, element);
        }

        private final boolean acceptsElementPost(Parent parent,
                                                 Element element) {
            return elementPostFilter.accepts(parent, element);
        }

        private String shareElementName(String name) {
            if (isEnabled(Feature.SHARE_ELEMENT_NAMES)) {
                return CACHE.get(name);
            } else {
                return name;
            }
        }

        private final String convertElementName(Parent parent,
                                                String name) {
            return shareElementName(elementNameConverter.convertElementName(parent, name));
        }

        @Override
        public void startElement(String uri,
                                 String localName,
                                 String qName,
                                 Attributes atts) throws SAXException {
            LOGGER.trace("startElement()");
            if (top() == Action.KEEP) {
                checkActiveText(false);
                if (!isEnabled(Feature.ALLOW_MIXED_CONTENT) && currentParent.hasChildren(Text.class)) {
                    throw new SAXException("Mixed content not allowed");
                }
                final Element element = new Element(convertElementName(currentParent, qName));
                for (int index = 0; index < atts.getLength(); index++) {
                    final String name = atts.getQName(index);
                    final String value = atts.getValue(index);
                    if (acceptsAttribute(element, name, value)) {
                        element.addAttribute(convertAttributeName(element, name),
                                             convertAttributeValue(element, name, value));
                    }
                }

                if (acceptsElementPre(currentParent, element)) {
                    pushAction(Action.KEEP);
                    currentParent.addChild(element);
                    currentParent = element;
                    resetActiveText();
                } else {
                    pushAction(Action.IGNORE);
                }
            } else {
                pushAction(Action.IGNORE);
            }
        }

        @Override
        public void endElement(String uri,
                               String localName,
                               String qName) throws SAXException {
            LOGGER.trace("endElement()");
            if (top() == Action.KEEP) {
                final Element current = (Element) currentParent;

                checkActiveText(true);
                currentParent = ((Child) currentParent).getParent();
                resetActiveText();

                if (!acceptsElementPost(current.getParent(), current)) {
                    current.detach();
                }
            }
            popAction();
        }

        @Override
        public void characters(char[] ch,
                               int start,
                               int length) throws SAXException {
            LOGGER.trace("characters()");
            if (top() == Action.KEEP) {
                if (activeText == null) {
                    activeText = new Text();
                    currentParent.addChild(activeText);
                }
                activeText.appendContent(ch, start, length);
            }
        }

        @Override
        public void ignorableWhitespace(char[] ch,
                                        int start,
                                        int length) throws SAXException {
            LOGGER.trace("ignorableWhitespace()");
            // Ignore
        }

        @Override
        public void processingInstruction(String target,
                                          String data) throws SAXException {
            // Ignore
        }

        @Override
        public void skippedEntity(String name) throws SAXException {
            // Ignore
        }

        @Override
        public void startDTD(String name,
                             String publicId,
                             String systemId) throws SAXException {
            // Ignore
        }

        @Override
        public void endDTD() throws SAXException {
            // Ignore
        }

        @Override
        public void startEntity(String name) throws SAXException {
            // Ignore
        }

        @Override
        public void endEntity(String name) throws SAXException {
            // Ignore
        }

        @Override
        public void startCDATA() throws SAXException {
            // Ignore
        }

        @Override
        public void endCDATA() throws SAXException {
            // Ignore
        }

        @Override
        public void comment(char[] ch,
                            int start,
                            int length) throws SAXException {
            LOGGER.trace("comment()");
            if (top() == Action.KEEP) {
                // IGNORE_COMMENTS already handled in parser configuration
                checkActiveText(false);
                final Comment comment = new Comment(new String(ch, start, length));
                currentParent.addChild(comment);
                resetActiveText();
            }
        }

        @Override
        public void warning(SAXParseException exception) throws SAXException {
            LOGGER.warn(locator.getLineNumber() + ":" + locator.getColumnNumber(), exception);
        }

        @Override
        public void error(SAXParseException exception) throws SAXException {
            LOGGER.error(locator.getLineNumber() + ":" + locator.getColumnNumber(), exception);
        }

        @Override
        public void fatalError(SAXParseException exception) throws SAXException {
            LOGGER.fatal(locator.getLineNumber() + ":" + locator.getColumnNumber(), exception);
            super.fatalError(exception);
        }
    }
}