package cdc.test.util.data;

import static org.junit.jupiter.api.Assertions.assertEquals;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.junit.jupiter.api.Test;

import cdc.util.data.Attribute;
import cdc.util.data.Document;
import cdc.util.data.Node;
import cdc.util.data.util.AttributeNameConverter;
import cdc.util.data.util.AttributePredicate;
import cdc.util.data.util.AttributeValueConverter;
import cdc.util.data.util.DataUtil;
import cdc.util.data.util.ElementNameConverter;
import cdc.util.data.util.TextContentConverter;
import cdc.util.data.util.TextPredicate;
import cdc.util.data.xml.XmlDataReader;
import cdc.util.data.xml.XmlDataWriter;
import cdc.util.function.Predicates;
import cdc.util.xml.XmlWriter;

public class DataUtilTest {
    private static final String XML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
    private static final String INDENT = null;
    private static final String EOL = "";

    @Test
    public void testSetNameOfMatchingAttributes() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child y='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root X='a'>" + EOL
                + "<child Y='a'/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.setNameOfMatchingAttributes(doc,
                                             AttributePredicate.ANY_ATTRIBUTE,
                                             AttributeNameConverter.fromNameFunction(String::toUpperCase),
                                             DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSetValueOfMatchingAttributes() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child y='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='A'>" + EOL
                + "<child y='A'/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.setValueOfMatchingAttributes(doc,
                                              AttributePredicate.ANY_ATTRIBUTE,
                                              AttributeValueConverter.fromValueFunction(String::toUpperCase),
                                              DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSortAttributes() throws IOException {
        final String source = XML + EOL
                + "<root y='b' x='a'>" + EOL
                + "<child y='b' x='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='a' y='b'>" + EOL
                + "<child x='a' y='b'/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.sortAttributes(doc,
                                Attribute.NAME_COMPARATOR,
                                DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testRemoveEmptyAttributes() throws IOException {
        final String source = XML + EOL
                + "<root x='' y='b'>" + EOL
                + "<child x='' y=''/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root y='b'>" + EOL
                + "<child/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.removeEmptyAttributes(doc,
                                       DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testRemovePureElements() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child>" + EOL
                + "<child/>" + EOL
                + "<child/>" + EOL
                + "</child>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='a'/>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.removePureElements(doc, DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSetNameOfMatchingElements() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child x='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<ROOT x='a'>" + EOL
                + "<CHILD x='a'/>" + EOL
                + "</ROOT>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.setNameOfMatchingElements(doc,
                                           Predicates.alwaysTrue(),
                                           ElementNameConverter.fromNameFunction(String::toUpperCase),
                                           DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSortChildrenElementsByName() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child2 x='a'/>" + EOL
                + "<child1 x='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='a'>" + EOL
                + "<child1 x='a'/>" + EOL
                + "<child2 x='a'/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.sortChildren(doc,
                              Predicates.alwaysTrue(),
                              Node.ELEMENT_NAME_COMPARATOR,
                              DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSortChildrenElementsByNameAndAttributes() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child x='b'/>" + EOL
                + "<child x='a'/>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='a'>" + EOL
                + "<child x='a'/>" + EOL
                + "<child x='b'/>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.sortChildren(doc,
                              Predicates.alwaysTrue(),
                              Node.ELEMENT_NAME_AND_ATTRIBUTES_COMPARATOR,
                              DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }

    @Test
    public void testSetContentOfMatchingTexts() throws IOException {
        final String source = XML + EOL
                + "<root x='a'>" + EOL
                + "<child y='a'>aaa</child>" + EOL
                + "</root>";
        final String expected = XML + EOL
                + "<root x='a'>" + EOL
                + "<child y='a'>AAA</child>" + EOL
                + "</root>";
        final Document doc = XmlDataReader.load(source, StandardCharsets.UTF_8);
        DataUtil.setContentOfMatchingTexts(doc,
                                           TextPredicate.ANY_TEXT,
                                           TextContentConverter.fromContentFunction(String::toUpperCase),
                                           DataUtil.RECURSE);
        final String target = XmlDataWriter.toString(doc, INDENT, XmlWriter.Feature.USE_SINGLE_QUOTE);
        assertEquals(expected, target);
    }
}