package cdc.test.util.data;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.net.URL;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.io.IoBuilder;
import org.junit.jupiter.api.Test;

import cdc.util.data.Attribute;
import cdc.util.data.Comment;
import cdc.util.data.Document;
import cdc.util.data.Element;
import cdc.util.data.Node;
import cdc.util.data.Nodes;
import cdc.util.data.Text;
import cdc.util.data.util.DataStats;
import cdc.util.data.util.DataUtils;
import cdc.util.data.xml.XmlDataReader;
import cdc.util.data.xml.XmlDataWriter;
import cdc.util.debug.Memory;
import cdc.util.files.Files;
import cdc.util.time.Chronometer;
import cdc.util.xml.XmlWriter;

public class XmlDataReaderTest {
    private static final Logger LOGGER = LogManager.getLogger(XmlDataReaderTest.class);
    private static final PrintStream OUT = IoBuilder.forLogger(LOGGER).setLevel(Level.DEBUG).buildPrintStream();

    private static void generateTestData(String filename) {
        try {
            final int folders = 20;
            final int objects = 500;
            final int attributes = 100;
            final XmlWriter writer = new XmlWriter(filename, "UTF-8");
            writer.setEnabled(XmlWriter.Feature.PRETTY_PRINT, true);
            writer.beginDocument();
            writer.beginElement("root");
            for (int findex = 0; findex < folders; findex++) {
                writer.beginElement("folder");
                writer.addAttribute("type", "type" + findex);
                for (int oindex = 0; oindex < objects; oindex++) {
                    writer.beginElement("object");
                    writer.addAttribute("id", findex + "." + oindex);
                    writer.addAttribute("type", "type" + findex);
                    for (int aindex = 0; aindex < attributes; aindex++) {
                        writer.beginElement("attribute");
                        writer.addAttribute("name", "att" + aindex);
                        writer.addAttribute("value", "value" + aindex);
                        writer.endElement();
                    }
                    writer.endElement();
                }
                writer.endElement();
            }
            writer.endElement();
            writer.endDocument();
            writer.close();
        } catch (final IOException e) {
            e.printStackTrace();
        }
    }

    @Test
    public void testAttributeNameConversion() throws IOException {
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        reader.setAttributeNameConverter((element,
                                          name) -> name.toUpperCase());

        final URL url = getClass().getClassLoader().getResource("test-reader.xml");
        final Document doc = reader.read(Files.toFile(url).getPath());
        assertTrue(DataUtils.hasAllDescendantsMatching(doc,
                                                       node -> {
                                                           if (node instanceof Element) {
                                                               final Element e = (Element) node;
                                                               for (final Attribute a : e.getAttributes()) {
                                                                   if (!a.getName().toUpperCase().equals(a.getName())) {
                                                                       return false;
                                                                   }
                                                               }
                                                               return true;
                                                           } else {
                                                               return true;
                                                           }
                                                       },
                                                       true));
        LOGGER.debug("Upper case attribute names");
        XmlDataWriter.print(doc, OUT, "  ", false);
    }

    @Test
    public void testAttributeValueConversion() throws IOException {
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        reader.setAttributeValueConverter((element,
                                           name,
                                           value) -> value.toUpperCase());

        final URL url = getClass().getClassLoader().getResource("test-reader.xml");
        final Document doc = reader.read(Files.toFile(url).getPath());

        LOGGER.debug("Upper case attribute values");
        XmlDataWriter.print(doc, OUT, "  ", false);
    }

    @Test
    public void testAttributeFiltering() throws IOException {
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        reader.setAttributeFilter((element,
                                   name,
                                   value) -> "id".equals(name));

        final URL url = getClass().getClassLoader().getResource("test-reader.xml");
        final Document doc = reader.read(Files.toFile(url).getPath());

        LOGGER.debug("Only id attributes");
        XmlDataWriter.print(doc, OUT, "  ", false);
    }

    @Test
    public void testElementNameConversion() throws IOException {
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        reader.setElementNameConverter((parent,
                                        name) -> name.toUpperCase());

        final URL url = getClass().getClassLoader().getResource("test-reader.xml");
        final Document doc = reader.read(Files.toFile(url).getPath());

        LOGGER.debug("Upper case element names");
        XmlDataWriter.print(doc, OUT, "  ", false);
    }

    @Test
    public void testElementFiltering() throws IOException {
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        reader.setElementPreFilter((parent,
                                    element) -> "root".equals(element.getName())
                                            || "E1".equals(element.getAttributeValue("id", ""))
                                            || "E1.1".equals(element.getAttributeValue("id", ""))
                                            || "E2".equals(element.getAttributeValue("id", "")));

        final URL url = getClass().getClassLoader().getResource("test-reader.xml");
        final Document doc = reader.read(Files.toFile(url).getPath());

        LOGGER.debug("Only E1, E1.1 and E2");
        XmlDataWriter.print(doc, OUT, "  ", false);
    }

    @Test
    public void testLargeElementFiltering() throws IOException {
        final File file = File.createTempFile("cdc-test", "xml");
        final Chronometer chrono = new Chronometer();
        chrono.start();
        generateTestData(file.getPath());
        chrono.suspend();
        LOGGER.debug("Generation time: " + chrono);
        Memory.warmUp();
        Memory.runGC();
        final long init = Memory.usedMemory();
        final XmlDataReader reader = new XmlDataReader();
        reader.setEnabled(XmlDataReader.Feature.LOAD_SPACES, false);
        // reader.setEnabled(XmlDataReader.Feature.SHARE_ATTRIBUTE_NAMES, true);
        // reader.setEnabled(XmlDataReader.Feature.SHARE_ELEMENT_NAMES, true);
        final Predicate<Node> p = node -> {
            if (node instanceof Element) {
                final Element e = (Element) node;
                return "folder".equals(e.getName()) && "type0".equals(e.getAttributeValue("type", ""));
            } else {
                return false;
            }
        };
        reader.setElementPreFilter((parent,
                                    element) -> {
            final boolean ok = "root".equals(element.getName())
                    || p.test(element)
                    || DataUtils.hasAncestorMatching(parent, p);
            return ok;
        });
        chrono.start();
        final Document doc = reader.read(file);
        Memory.runGC();
        final long end = Memory.usedMemory();
        chrono.suspend();
        LOGGER.debug("Reading time: " + chrono + " " + (end - init));
        DataStats.print(doc, OUT, 0);
    }

    private static InputStream createInputStream(String s) {
        return new ByteArrayInputStream(s.getBytes());
    }

    private static void test(String expected,
                             String source,
                             Consumer<Document> docChecker,
                             XmlDataReader.Feature... features) throws IOException {
        LOGGER.debug("test(" + source + ")");
        final InputStream is = createInputStream(source);
        final Document doc = XmlDataReader.load(is, features);
        if (docChecker != null) {
            docChecker.accept(doc);
        }
        final String result = XmlDataWriter.toString(doc, null, XmlWriter.Feature.USE_XML_EOL);
        assertEquals(expected, result);
    }

    @Test
    public void testFeaturesDefault0() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root/>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!-- comment --><root/><!-- comment -->",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(0, doc.getRootElement().getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault1() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root/>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!-- comment --><root></root><!-- comment -->",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(0, doc.getRootElement().getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault2() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root>   </root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!-- comment --><root> <!-- comment -->  </root><!-- comment -->",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault3() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root>aaa</root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!-- comment --><root><!-- comment -->aaa</root><!-- comment -->",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
             });
    }

    public void testFeaturesDefaultUnexpectedMixed1() throws IOException {
        assertThrows(IOException.class, () -> {
            final InputStream is = createInputStream("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root>text<child/></root>");
            XmlDataReader.load(is);
        });
    }

    public void testFeaturesDefaultUnexpectedMixed2() throws IOException {
        assertThrows(IOException.class, () -> {
            final InputStream is = createInputStream("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child/>text</root>");
            XmlDataReader.load(is);
        });
    }

    @Test
    public void testFeaturesDefault4() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child> </child></root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \n  <root> \n <child> </child> \n </root>",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
                 assertEquals(1, doc.getRootElement().getChildAt(Element.class, 0).getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault5() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child/></root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \n  <root> \n <child></child> \n </root>",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
                 assertEquals(0, doc.getRootElement().getChildAt(Element.class, 0).getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault6() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child/></root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child/></root>",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
             });
    }

    @Test
    public void testFeaturesDefault7() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><root><child/></root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \n  <root> \n <child/> \n </root>",
             (doc) -> {
                 assertEquals(1, doc.getChildrenCount());
                 assertEquals("root", doc.getRootElement().getName());
                 assertEquals(1, doc.getRootElement().getChildrenCount());
             });
    }

    @Test
    public void testFeaturesComments() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><!--comment--><root><child/></root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \n <!--comment--> <root> \n <child/> \n </root>",
             (doc) -> {
                 assertEquals(2, doc.getChildrenCount());
                 assertEquals(1, doc.getChildrenCount(Comment.class));
             },
             XmlDataReader.Feature.LOAD_COMMENTS);
    }

    @Test
    public void testFeaturesCommentsMixed1() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><!--comment--><root>  aaa\n  aaa<child/>  aaa\n  aaa</root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!--comment--><root>  aaa\n  aaa<child/>  aaa\n  aaa</root>",
             (doc) -> {
                 assertEquals(2, doc.getChildrenCount());
                 assertEquals(1, doc.getChildrenCount(Comment.class));
                 assertEquals(3, doc.getRootElement().getChildrenCount());
                 assertEquals(2, doc.getRootElement().getChildrenCount(Text.class));
             },
             XmlDataReader.Feature.LOAD_COMMENTS,
             XmlDataReader.Feature.LOAD_SPACES,
             XmlDataReader.Feature.ALLOW_MIXED_CONTENT);
    }

    @Test
    public void testFeaturesCommentsMixed2() throws IOException {
        test("<?xml version=\"1.0\" encoding=\"UTF-8\"?><!--comment--><root>aaa\naaa<child/>aaa\naaa</root>",
             "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<!--comment--><root>aaa\naaa<child/>aaa\naaa</root>",
             (doc) -> {
                 Nodes.print(doc, LOGGER, Level.DEBUG);
                 assertEquals(2, doc.getChildrenCount());
                 assertEquals(1, doc.getChildrenCount(Comment.class));
                 assertEquals(3, doc.getRootElement().getChildrenCount());
                 assertEquals(2, doc.getRootElement().getChildrenCount(Text.class));
             },
             XmlDataReader.Feature.LOAD_COMMENTS,
             XmlDataReader.Feature.LOAD_SPACES,
             XmlDataReader.Feature.ALLOW_MIXED_CONTENT);
    }
}