package cdc.util.data.util;

import java.util.Map;
import java.util.function.UnaryOperator;

import cdc.util.data.Element;
import cdc.util.data.Parent;
import cdc.util.data.paths.SPath;
import cdc.util.lang.Checks;
import cdc.util.strings.StringUtils;

@FunctionalInterface
public interface TextContentConverter {
    public static final TextContentConverter IDENTITY = (p,
                                                         c) -> c;

    public String convertTextContent(Parent parent,
                                     String content);

    /**
     * Returns a converter that first applies {@code other} then this converter.
     *
     * @param other The other converter.
     * @return A converter that first applies {@code other} then this converter.
     * @throws IllegalArgumentException When {@code other} is {@code null}.
     */
    public default TextContentConverter compose(TextContentConverter other) {
        Checks.isNotNull(other, "other");
        return (Parent parent,
                String content) -> convertTextContent(parent, other.convertTextContent(parent, content));
    }

    /**
     * Returns a converter that first applies this converter and then {@code other}.
     *
     * @param other The other converter.
     * @return A converter that first applies this converter and then {@code other}.
     * @throws IllegalArgumentException When {@code other} is {@code null}.
     */
    public default TextContentConverter andThen(TextContentConverter other) {
        Checks.isNotNull(other, "other");
        return (Parent parent,
                String content) -> other.convertTextContent(parent, convertTextContent(parent, content));
    }

    public static TextContentConverter fromString(String s) {
        return (Parent parent,
                String content) -> s;
    }

    public static TextContentConverter scramble(boolean preserveSpaces) {
        return (Parent parent,
                String content) -> StringUtils.scrambleWithLettersOrDigits(content, preserveSpaces);
    }

    /**
     * Creates a new TextContentConverter from a content converter function.
     * <p>
     * Conversion is independent of element.
     *
     * @param function The function used to convert content.
     * @return A new TextContentConverter from {@code function}.
     * @throws IllegalArgumentException When {@code function} is {@code null}.
     */
    public static TextContentConverter fromContentFunction(UnaryOperator<String> function) {
        Checks.isNotNull(function, "function");
        return (Parent parent,
                String content) -> function.apply(content);
    }

    /**
     * Creates a new TextContentConverter from a map.
     * <p>
     * Conversion is independent of element and attribute name.<br>
     * If a value is not mapped, then result is that value.
     *
     * @param map The map.
     * @return A new TextContentConverter from {@code map}.
     * @throws IllegalArgumentException When {@code map} is {@code null}.
     */
    public static TextContentConverter fromContentMap(Map<String, String> map) {
        Checks.isNotNull(map, "map");
        return (Parent parent,
                String content) -> map.getOrDefault(content, content);
    }

    public static TextContentConverter fromNameMap(Map<String, String> map) {
        Checks.isNotNull(map, "map");
        return (Parent parent,
                String content) -> parent instanceof Element
                        ? map.getOrDefault(((Element) parent).getName(), content)
                        : content;
    }

    public static TextContentConverter fromPathMap(Map<SPath, String> map) {
        Checks.isNotNull(map, "map");
        return (Parent parent,
                String content) -> {
            for (final Map.Entry<SPath, String> entry : map.entrySet()) {
                if (entry.getKey().matchesElement(parent)) {
                    return entry.getValue();
                }
            }
            return content;
        };
    }
}