package cdc.util.graphs.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import cdc.util.function.IterableUtils;
import cdc.util.graphs.EdgeDirection;
import cdc.util.graphs.EdgeTip;
import cdc.util.graphs.GraphAdapter;
import cdc.util.graphs.GraphEdge;
import cdc.util.lang.Checks;
import cdc.util.lang.InvalidStateException;

/**
 * Basic and naive graph implementation using light nodes.
 * <p>
 * Node can be any object.
 *
 * @author Damien Carbonne
 *
 * @param <N> Node type.
 * @param <E> Edge type.
 */
public class BasicLightGraph<N, E extends GraphEdge<N>> implements GraphAdapter<N, E> {
    /** Set of edges. */
    private final Set<E> edges;
    /** Map from nodes to associated edges. */
    private final Map<N, Set<E>> nodeToEdges;
    private boolean locked = false;

    public BasicLightGraph(boolean sorted) {
        if (sorted) {
            this.edges = new LinkedHashSet<>();
            this.nodeToEdges = new LinkedHashMap<>();
        } else {
            this.edges = new HashSet<>();
            this.nodeToEdges = new HashMap<>();
        }
    }

    public BasicLightGraph() {
        this(false);
    }

    /**
     * Returns the set of edges associated to a node.
     *
     * @param node The node.
     * @return The set of edges associated to {@code node}.
     * @throws IllegalArgumentException When {@code node} is {@code null}
     *             or is not contained in this graph.
     */
    private Set<E> getNodeEdges(N node) {
        Checks.isNotNull(node, "node");
        Checks.isTrue(containsNode(node), "Unknown node");

        final Set<E> tmp = nodeToEdges.get(node);
        return tmp == null ? Collections.emptySet() : tmp;
    }

    /**
     * Checks that this graph is not locked.
     *
     * @throws InvalidStateException When this graph is locked.
     */
    private void checkIsUnlocked() {
        if (locked) {
            throw new InvalidStateException("Locked");
        }
    }

    public void setLocked(boolean locked) {
        this.locked = locked;
    }

    public boolean isLocked() {
        return locked;
    }

    /**
     *
     * Clears this graph.
     *
     * @throws InvalidStateException When this graph is locked.
     */
    public void clear() {
        checkIsUnlocked();

        edges.clear();
        nodeToEdges.clear();
    }

    /**
     * Adds a node to this graph.
     *
     * @param <X> The node type.
     * @param node The node.
     * @return The passed {@code node}.
     * @throws IllegalArgumentException When {@code node} is {@code null}
     *             or is already contained in this graph.
     * @throws InvalidStateException When this graph is locked.
     */
    public <X extends N> X addNode(X node) {
        checkIsUnlocked();
        Checks.isNotNull(node, "node");

        if (containsNode(node)) {
            throw new IllegalArgumentException("Node already declared: " + node);
        }

        nodeToEdges.put(node, null);
        return node;
    }

    /**
     * Remove a node from this graph.
     *
     * @param node The node.
     * @throws IllegalArgumentException When {@code node} is {@code null}
     *             or is not contained in this graph.
     * @throws InvalidStateException When this graph is locked.
     */
    public void removeNode(N node) {
        checkIsUnlocked();
        Checks.isNotNull(node, "node");

        if (!containsNode(node)) {
            throw new IllegalArgumentException("Node does not belong to graph: " + node);
        }

        // Remove edges attached to node
        final Set<E> tmp = new HashSet<>();
        for (final E edge : getEdges(node, null)) {
            tmp.add(edge);
        }
        for (final E edge : tmp) {
            removeEdge(edge);
        }

        // Remove node
        // Associated edges must have been removed by above code
        Checks.assertTrue(getNodeEdges(node).isEmpty(), "Invalid state");
        nodeToEdges.remove(node);
    }

    /**
     * Adds an edge to this graph.
     *
     * @param <X> The edge type.
     * @param edge The edge.
     * @return The passed {@code edge}.
     * @throws IllegalArgumentException When {@code edge} is {@code null}
     *             or is already contained in this graph,
     *             or {@code edge} source or target don't not belong to this graph.
     * @throws InvalidStateException When this graph is locked.
     */
    public <X extends E> X addEdge(X edge) {
        checkIsUnlocked();
        Checks.isNotNull(edge, "edge");

        if (containsEdge(edge)) {
            throw new IllegalArgumentException("Edge already declared: " + edge);
        }
        Checks.isTrue(containsNode(edge.getSource()), "edge source " + edge.getSource() + " does not belong to graph.");
        Checks.isTrue(containsNode(edge.getTarget()), "edge target " + edge.getTarget() + " does not belong to graph.");

        edges.add(edge);
        final Set<E> sourceEdges = nodeToEdges.computeIfAbsent(edge.getSource(), n -> new HashSet<>());
        sourceEdges.add(edge);
        final Set<E> targetEdges = nodeToEdges.computeIfAbsent(edge.getTarget(), n -> new HashSet<>());
        targetEdges.add(edge);

        return edge;
    }

    /**
     * Removes an edge from this graph.
     *
     * @param edge The edge.
     * @throws IllegalArgumentException When {@code edge} is {@code null}
     *             or is not contained in this graph.
     * @throws InvalidStateException When this graph is locked.
     */
    public void removeEdge(E edge) {
        checkIsUnlocked();

        // Remove from edges
        final boolean removed = edges.remove(edge);
        Checks.assertTrue(removed, "Failed to remove " + edge + " from edges");

        // Remove from source node edges
        final Set<E> sourceEdges = nodeToEdges.get(edge.getSource());
        Checks.assertTrue(sourceEdges != null, "Invalid state");
        sourceEdges.remove(edge);

        // Remove from target node edges
        final Set<E> targetEdges = nodeToEdges.get(edge.getTarget());
        Checks.assertTrue(targetEdges != null, "Invalid state");
        targetEdges.remove(edge);
    }

    @Override
    public final Iterable<N> getNodes() {
        return nodeToEdges.keySet();
    }

    @Override
    public final boolean containsNode(N node) {
        return nodeToEdges.containsKey(node);
    }

    @Override
    public final Iterable<E> getEdges() {
        return edges;
    }

    @Override
    public final boolean containsEdge(E edge) {
        return edges.contains(edge);
    }

    /**
     * Return {@code true} if this graph contains at least one edge between a source node and a target node.
     *
     * @param source The source node.
     * @param target The target node.
     * @return {@code true} if this graph contains at least one edge between {@code source} and {@code target}.
     */
    public boolean containsEdge(N source,
                                N target) {
        final Set<E> sourceEdges = nodeToEdges.get(source);
        final Set<E> targetEdges = nodeToEdges.get(target);

        if (sourceEdges == null || targetEdges == null) {
            return false;
        } else if (sourceEdges.size() <= targetEdges.size()) {
            for (final E edge : sourceEdges) {
                if (edge.getSource().equals(source) && edge.getTarget().equals(target)) {
                    return true;
                }
            }
            return false;
        } else {
            for (final E edge : targetEdges) {
                if (edge.getSource().equals(source) && edge.getTarget().equals(target)) {
                    return true;
                }
            }
            return false;
        }
    }

    @Override
    public final Iterable<? extends E> getEdges(N node,
                                                EdgeDirection direction) {
        final Set<E> nodeEdges = getNodeEdges(node);
        if (direction == null) {
            return nodeEdges;
        } else if (direction == EdgeDirection.INGOING) {
            return IterableUtils.filter(nodeEdges, e -> e.getTarget().equals(node));
        } else {
            return IterableUtils.filter(nodeEdges, e -> e.getSource().equals(node));
        }
    }

    @Override
    public final N getTip(E edge,
                          EdgeTip tip) {
        Checks.isNotNull(edge, "edge");
        Checks.isNotNull(tip, "tip");

        if (tip == EdgeTip.SOURCE) {
            return edge.getSource();
        } else {
            return edge.getTarget();
        }
    }
}