package pluginloader.load;

import javax.tools.*;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.Collections;

public class PluginCompiler {
    public static class CompilationFailedException extends Exception{
        private final String code;

        public CompilationFailedException(String message, String code) {
            super(message);
            this.code = code;
        }

        public String getCode() {
            return code;
        }
    }

    private static class JavaStringObject extends SimpleJavaFileObject{
        private final String code;

        public JavaStringObject(String pluginName, String code) {
            super(URI.create(String.format(
                    "string:///%s%s",
                    pluginName.replace('.','/'),
                    Kind.SOURCE.extension
            )), Kind.SOURCE);
            this.code = code;
        }

        @Override
        public CharSequence getCharContent(boolean ignoreEncodingErrors) {
            return code;
        }
    }

    private static class JavaByteObject extends SimpleJavaFileObject{
        private final ByteArrayOutputStream outputStream;

        public JavaByteObject(String name) {
            super(URI.create(String.format("bytes:///%s%s", name, name.replaceAll("\\.", "/"))), Kind.CLASS);
            this.outputStream = new ByteArrayOutputStream();
        }

        @Override
        public OutputStream openOutputStream() throws IOException {
            return outputStream;
        }

        public byte[] getBytes() {
            return outputStream.toByteArray();
        }
    }

    public static byte[] compile(String pluginName, String sourceCode) throws CompilationFailedException {
        JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
        DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<>();
        JavaByteObject javaByteObject = new JavaByteObject(pluginName);
        PluginFileManager pluginFileManager = new PluginFileManager(compiler.getStandardFileManager(diagnostics, null, null), javaByteObject);
        JavaCompiler.CompilationTask compilationTask = compiler.getTask(
                null, pluginFileManager, diagnostics,
                null, null, () -> {
                    JavaFileObject javaFileObject = new JavaStringObject(pluginName, sourceCode);
                    return Collections.singletonList(javaFileObject).iterator();
                });
        boolean compilationSuccessful = compilationTask.call();
        if (!compilationSuccessful){
            throw new CompilationFailedException(String.format("Failed to compile class '%s'", pluginName), sourceCode);
        }
        return javaByteObject.getBytes();
    }
}
