package pluginloader;

import pluginloader.load.PluginClassLoader;
import pluginloader.load.PluginCompiler;
import pluginloader.load.SourceCodeParser;

import java.lang.reflect.InvocationTargetException;

/**
 * This class loads plugins and instantiates them
 */
public class PluginLoader {
    public static class LoadingException extends Exception{
        public LoadingException(String message) {
            super(message);
        }

        public LoadingException(Throwable cause) {
            super(cause);
        }
    }

    /**
     * Compiles a plugin given as a java class in source code form and loads it into the Java Runtime.
     * It only loads classes that extend the given java class.
     * After successfully loading it, an instance of each found class that extends the given class
     * will be returned to the caller.
     * If a class already exists, it will be updated if its code has changed. If the code has
     * not changed, the loading is skipped and a new instance of the class is returned.
     * @param plugin The string containing the source code of the plugin class
     * @param pluginInterface The interface/class the plugin class has to implement/extend
     * @param <OBJECT> The type of the object to be returned
     * @return The new instance of the class
     * @throws LoadingException When something goes wrong while compiling or loading the class
     */
    public <OBJECT> OBJECT load(String plugin, Class<OBJECT> pluginInterface) throws LoadingException {
        try {
            ClassInfo classInfo = new SourceCodeParser().parseClassInfo(plugin);
            byte[] compile = PluginCompiler.compile(classInfo.getFullQualifiedClassName(), plugin);
            PluginClassLoader pluginClassLoader = new PluginClassLoader();
            pluginClassLoader.putClassCode(classInfo.getFullQualifiedClassName(), compile);

            Class<?> pluginClass = pluginClassLoader.findClass(classInfo.getFullQualifiedClassName());
            if (!pluginInterface.isAssignableFrom(pluginClass)){
                throw new LoadingException(String.format("The compiled class is no implementation or subclass of %s", pluginInterface.getCanonicalName()));
            }
            Object pluginInstance = pluginClass.getConstructor().newInstance();
            //noinspection unchecked
            return (OBJECT) pluginInstance;
        } catch (ClassNotFoundException | NoSuchMethodException | InstantiationException
                | IllegalAccessException | InvocationTargetException | PluginCompiler.CompilationFailedException e) {
            throw new LoadingException(e);
        }
    }
}
