/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.examples.dns;

import com.google.cloud.dns.ChangeRequest;
import com.google.cloud.dns.ChangeRequestInfo;
import com.google.cloud.dns.Dns;
import com.google.cloud.dns.DnsOptions;
import com.google.cloud.dns.ProjectInfo;
import com.google.cloud.dns.RecordSet;
import com.google.cloud.dns.Zone;
import com.google.cloud.dns.ZoneInfo;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

public class DnsExample {
    private static final Map<String, DnsAction> ACTIONS = new HashMap<String, DnsAction>();
    private static final DateTimeFormatter FORMATTER = DateTimeFormat.forPattern((String)"YYYY-MM-dd HH:mm:ss");

    private static void printZone(Zone zone) {
        System.out.printf("%nName: %s%n", zone.name());
        System.out.printf("ID: %s%n", zone.generatedId());
        System.out.printf("Description: %s%n", zone.description());
        System.out.printf("Created: %s%n", FORMATTER.print(zone.creationTimeMillis().longValue()));
        System.out.printf("Name servers: %s%n", Joiner.on((String)", ").join((Iterable)zone.nameServers()));
    }

    private static ChangeRequestInfo waitForChangeToFinish(Dns dns, String zoneName, ChangeRequestInfo request) {
        ChangeRequestInfo current = request;
        while (current.status().equals((Object)ChangeRequestInfo.Status.PENDING)) {
            System.out.print(".");
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {
                System.err.println("Thread was interrupted while waiting.");
            }
            current = dns.getChangeRequest(zoneName, current.generatedId(), new Dns.ChangeRequestOption[0]);
        }
        return current;
    }

    private static void printUsage() {
        StringBuilder actionAndParams = new StringBuilder();
        for (Map.Entry<String, DnsAction> entry : ACTIONS.entrySet()) {
            actionAndParams.append(System.lineSeparator()).append('\t').append(entry.getKey());
            String param = entry.getValue().params();
            if (param == null || param.isEmpty()) continue;
            actionAndParams.append(' ').append(param);
        }
        System.out.printf("Usage: %s [<project_id>] operation <args>*%s%n", DnsExample.class.getSimpleName(), actionAndParams);
    }

    public static void main(String ... args) throws Exception {
        String actionName;
        if (args.length < 1) {
            System.out.println("Missing required action");
            DnsExample.printUsage();
            return;
        }
        String projectId = null;
        if (args.length >= 2 && !ACTIONS.containsKey(args[0])) {
            actionName = args[1];
            projectId = args[0];
            args = Arrays.copyOfRange(args, 2, args.length);
        } else {
            actionName = args[0];
            args = Arrays.copyOfRange(args, 1, args.length);
        }
        DnsAction action = ACTIONS.get(actionName);
        if (action == null) {
            System.out.printf("Unrecognized action %s.%n", actionName);
            DnsExample.printUsage();
            return;
        }
        boolean valid = false;
        try {
            valid = action.check(args);
        }
        catch (NumberFormatException ex) {
            System.out.println("Invalid input for action '" + actionName + "'.");
            System.out.println("Ttl must be an integer.");
            System.out.println("Expected: " + action.params());
            return;
        }
        catch (Exception ex) {
            System.out.println("Failed to parse request.");
            System.out.println("Expected: " + action.params());
            ex.printStackTrace();
            return;
        }
        if (valid) {
            DnsOptions.Builder optionsBuilder = DnsOptions.builder();
            if (projectId != null) {
                optionsBuilder.projectId(projectId);
            }
            Dns dns = (Dns)optionsBuilder.build().service();
            action.run(dns, args);
        } else {
            System.out.println("Invalid input for action '" + actionName + "'");
            System.out.println("Expected: " + action.params());
        }
    }

    static {
        ACTIONS.put("create", new CreateZoneAction());
        ACTIONS.put("delete", new DeleteZoneAction());
        ACTIONS.put("get", new GetZoneAction());
        ACTIONS.put("list", new ListAction());
        ACTIONS.put("add-record", new AddDnsRecordAction());
        ACTIONS.put("delete-record", new DeleteDnsRecordAction());
        ACTIONS.put("quota", new GetProjectAction());
    }

    private static class GetProjectAction
    implements DnsAction {
        private GetProjectAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            ProjectInfo project = dns.getProject(new Dns.ProjectOption[0]);
            ProjectInfo.Quota quota = project.quota();
            System.out.printf("Project id: %s%nQuota:%n", ((DnsOptions)dns.options()).projectId());
            System.out.printf("\tZones: %d%n", quota.zones());
            System.out.printf("\tRecord sets per zone: %d%n", quota.rrsetsPerZone());
            System.out.printf("\tRecord sets per DNS record: %d%n", quota.resourceRecordsPerRrset());
            System.out.printf("\tAdditions per change: %d%n", quota.rrsetAdditionsPerChange());
            System.out.printf("\tDeletions per change: %d%n", quota.rrsetDeletionsPerChange());
            System.out.printf("\tTotal data size per change: %d%n", quota.totalRrdataSizePerChange());
        }

        @Override
        public String params() {
            return "";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 0;
        }
    }

    private static class ListAction
    implements DnsAction {
        private ListAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            if (args.length == 0) {
                new ListZonesAction().run(dns, new String[0]);
            } else {
                if (args.length == 1 || "records".equals(args[1])) {
                    new ListDnsRecordsAction().run(dns, args);
                }
                if (args.length == 1 || "changes".equals(args[1])) {
                    new ListChangesAction().run(dns, args);
                }
            }
        }

        @Override
        public boolean check(String ... args) {
            if (args.length == 0 || args.length == 1) {
                return true;
            }
            if ("records".equals(args[1])) {
                return new ListDnsRecordsAction().check(args);
            }
            if ("changes".equals(args[1])) {
                return new ListChangesAction().check(args);
            }
            return false;
        }

        @Override
        public String params() {
            return "[<zone_name> [changes [descending | ascending] | records]]";
        }
    }

    private static class ListChangesAction
    implements DnsAction {
        private ListChangesAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            Iterator iterator;
            String zoneName = args[0];
            if (args.length > 2) {
                Dns.SortingOrder sortOrder = Dns.SortingOrder.valueOf((String)args[2].toUpperCase());
                iterator = dns.listChangeRequests(zoneName, new Dns.ChangeRequestListOption[]{Dns.ChangeRequestListOption.sortOrder((Dns.SortingOrder)sortOrder)}).iterateAll();
            } else {
                iterator = dns.listChangeRequests(zoneName, new Dns.ChangeRequestListOption[0]).iterateAll();
            }
            if (iterator.hasNext()) {
                System.out.printf("Change requests for zone %s:%n", zoneName);
                while (iterator.hasNext()) {
                    ChangeRequest change = (ChangeRequest)iterator.next();
                    System.out.printf("%nID: %s%n", change.generatedId());
                    System.out.printf("Status: %s%n", change.status());
                    System.out.printf("Started: %s%n", FORMATTER.print(change.startTimeMillis().longValue()));
                    System.out.printf("Deletions: %s%n", Joiner.on((String)", ").join((Iterable)change.deletions()));
                    System.out.printf("Additions: %s%n", Joiner.on((String)", ").join((Iterable)change.additions()));
                }
            } else {
                System.out.printf("Zone %s has no change requests.%n", zoneName);
            }
        }

        @Override
        public String params() {
            return "<zone_name> changes [descending | ascending]";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 2 || args.length == 3 && ImmutableList.of((Object)"descending", (Object)"ascending").contains((Object)args[2].toLowerCase());
        }
    }

    private static class ListDnsRecordsAction
    implements DnsAction {
        private ListDnsRecordsAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            Iterator iterator = dns.listRecordSets(zoneName, new Dns.RecordSetListOption[0]).iterateAll();
            if (iterator.hasNext()) {
                System.out.printf("Record sets for zone %s:%n", zoneName);
                while (iterator.hasNext()) {
                    RecordSet recordSet = (RecordSet)iterator.next();
                    System.out.printf("%nRecord name: %s%nTTL: %s%nRecords: %s%n", recordSet.name(), recordSet.ttl(), Joiner.on((String)", ").join((Iterable)recordSet.records()));
                }
            } else {
                System.out.printf("Zone %s has no record sets records.%n", zoneName);
            }
        }

        @Override
        public String params() {
            return "<zone_name> records";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 2;
        }
    }

    private static class AddDnsRecordAction
    implements DnsAction {
        private AddDnsRecordAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            String recordName = args[1];
            String ip = args[2];
            int ttl = 0;
            if (args.length > 3) {
                ttl = Integer.parseInt(args[3]);
            }
            RecordSet recordSet = RecordSet.builder((String)recordName, (RecordSet.Type)RecordSet.Type.A).records((List)ImmutableList.of((Object)ip)).ttl(ttl, TimeUnit.SECONDS).build();
            ChangeRequestInfo changeRequest = ChangeRequest.builder().add(recordSet).build();
            changeRequest = dns.applyChangeRequest(zoneName, changeRequest, new Dns.ChangeRequestOption[0]);
            System.out.printf("The request for adding A record %s for zone %s was successfully submitted and assigned ID %s.%n", recordName, zoneName, changeRequest.generatedId());
            System.out.print("Waiting for addition to happen...");
            DnsExample.waitForChangeToFinish(dns, zoneName, changeRequest);
            System.out.printf("The addition has been completed.%n", new Object[0]);
        }

        @Override
        public String params() {
            return "<zone_name> <record_name> <ip_address> [<ttl>]";
        }

        @Override
        public boolean check(String ... args) {
            if (args.length == 4) {
                Integer.parseInt(args[3]);
                return true;
            }
            return args.length == 3;
        }
    }

    private static class DeleteDnsRecordAction
    implements DnsAction {
        private DeleteDnsRecordAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            String recordName = args[1];
            String ip = args[2];
            int ttl = 0;
            if (args.length > 3) {
                ttl = Integer.parseInt(args[3]);
            }
            RecordSet recordSet = RecordSet.builder((String)recordName, (RecordSet.Type)RecordSet.Type.A).records((List)ImmutableList.of((Object)ip)).ttl(ttl, TimeUnit.SECONDS).build();
            ChangeRequestInfo changeRequest = ChangeRequest.builder().delete(recordSet).build();
            changeRequest = dns.applyChangeRequest(zoneName, changeRequest, new Dns.ChangeRequestOption[0]);
            System.out.printf("The request for deleting A record %s for zone %s was successfully submitted and assigned ID %s.%n", recordName, zoneName, changeRequest.generatedId());
            System.out.print("Waiting for deletion to happen...");
            DnsExample.waitForChangeToFinish(dns, zoneName, changeRequest);
            System.out.printf("%nThe deletion has been completed.%n", new Object[0]);
        }

        @Override
        public String params() {
            return "<zone_name> <record_name> <ip_address> [<ttl>]";
        }

        @Override
        public boolean check(String ... args) {
            if (args.length == 4) {
                Integer.parseInt(args[3]);
                return true;
            }
            return args.length == 3;
        }
    }

    private static class DeleteZoneAction
    implements DnsAction {
        private DeleteZoneAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            boolean deleted = dns.delete(zoneName);
            if (deleted) {
                System.out.printf("Zone %s was deleted.%n", zoneName);
            } else {
                System.out.printf("Zone %s was NOT deleted. It does not exist.%n", zoneName);
            }
        }

        @Override
        public String params() {
            return "<zone_name>";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 1;
        }
    }

    private static class GetZoneAction
    implements DnsAction {
        private GetZoneAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            Zone zone = dns.getZone(zoneName, new Dns.ZoneOption[0]);
            if (zone == null) {
                System.out.printf("No zone with name '%s' exists.%n", zoneName);
            } else {
                DnsExample.printZone(zone);
            }
        }

        @Override
        public String params() {
            return "<zone_name>";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 1;
        }
    }

    private static class ListZonesAction
    implements DnsAction {
        private ListZonesAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            Iterator zoneIterator = dns.listZones(new Dns.ZoneListOption[0]).iterateAll();
            if (zoneIterator.hasNext()) {
                System.out.println("The project contains the following zones:");
                while (zoneIterator.hasNext()) {
                    DnsExample.printZone((Zone)zoneIterator.next());
                }
            } else {
                System.out.println("Project contains no zones.");
            }
        }

        @Override
        public String params() {
            return "";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 0;
        }
    }

    private static class CreateZoneAction
    implements DnsAction {
        private CreateZoneAction() {
        }

        @Override
        public void run(Dns dns, String ... args) {
            String zoneName = args[0];
            String dnsName = args[1];
            String description = args[2];
            ZoneInfo zoneInfo = ZoneInfo.of((String)zoneName, (String)dnsName, (String)description);
            Zone zone = dns.create(zoneInfo, new Dns.ZoneOption[0]);
            System.out.printf("Successfully created zone with name %s which was assigned ID %s.%n", zone.name(), zone.generatedId());
        }

        @Override
        public String params() {
            return "<zone_name> <dns_name> <description>";
        }

        @Override
        public boolean check(String ... args) {
            return args.length == 3;
        }
    }

    private static interface DnsAction {
        public void run(Dns var1, String ... var2);

        public String params();

        public boolean check(String ... var1);
    }
}

