package com.googlecode.jpattern.gwt.client.presenter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.user.client.ui.HasWidgets;
import com.googlecode.jpattern.gwt.client.navigationevent.INavigationEvent;
import com.googlecode.jpattern.gwt.client.navigationevent.INavigationEventCallback;
import com.googlecode.jpattern.gwt.client.navigationevent.INavigationEventData;
import com.googlecode.jpattern.gwt.client.navigationevent.NullNavigationEventData;
import com.googlecode.jpattern.gwt.client.view.IShowViewStrategy;
import com.googlecode.jpattern.gwt.client.view.IView;
import com.googlecode.jpattern.shared.result.IErrorMessage;

/**
 * 
 * @author Francesco Cina'
 *
 * 14 Apr 2011
 */

public abstract class APresenter<T extends IView> implements IPresenter {
	
	private final IShowViewStrategy<T> showViewStrategy;
	private IPresenter parentPresenter;
	private final INavigationEvent navigationEvent;
	private final Map<String, NavigationEventData> registeredNavigationEvents = new HashMap<String, NavigationEventData>();

	public APresenter(IShowViewStrategy<T> showViewStrategy, INavigationEvent navigationEvent) {
		this.showViewStrategy = showViewStrategy;
		this.navigationEvent = navigationEvent;
		showViewStrategy.getView().visit(this);
	}
	
	@Override
	public final void onEventError(List<IErrorMessage> errorMessages) {
		showViewStrategy.getView().getErrorArea().addErrorMessages(errorMessages);
	}
	
	@Override
	public final void onEventStart() {
		showViewStrategy.getView().getErrorArea().clear();
	}
	
	@Override
	public final T getView() {
		return showViewStrategy.getView();
	}
	
	/**
	 * This is the first step performed to a complete visualization of the associated IView.
	 * This is performed just before the IShowViewStrategy.onLoadStart() method.
	 * This method is called before the display of the IView.
	 * DO NOT perform asynchronous actions in this method.
	 */
	public abstract void preDisplay();
	
	/**
	 * This is the second step performed to a complete visualization of the associated IView
	 * This method is called just after the IShowViewStrategy.onLoadStart() method and before the
	 * IShowViewStrategy.onLoadCompleted() method.
	 * Here the asynchronous actions should be performed.
	 * Its VERY important to call the render() method when all the actions are performed otherwise
	 * the IView should not be completely rendered. 
	 */
	public abstract void postDisplay();

	/**
	 * This method must be called when all the necessary operations to load necessary data are ended.
	 * This causes the final rendering of the IView. 
	 */
	@Override
	public final void render() {
		showViewStrategy.onLoadCompleted();
	}
	
	@Override
	public final void render(IPresenter parentPresenter) {
		this.parentPresenter = parentPresenter;
		showViewStrategy.onLoadCompleted(getParentPresenter().getNavigationEventData(getNavigationEvent().getName()).getEventTarget());
	}

	@Override
	public void setParent(IPresenter parentPresenter) {
		this.parentPresenter = parentPresenter;
	}
	
	@Override
	public final void init() {
		preDisplay();
		showViewStrategy.onLoadStart(getParentPresenter().getNavigationEventData(getNavigationEvent().getName()).getEventTarget());
		postDisplay();
	}
	
	@Override
	public final void hierarchy(List<IPresenter> hierarchyResult) {
		getParentPresenter().hierarchy(hierarchyResult);
		hierarchyResult.add(this);
	}
	
	@Override
	public final void registerNavigationEvent(INavigationEvent navigationEvent, HasWidgets eventTarget, INavigationEventCallback navigationEventCallback) {
		registeredNavigationEvents.put(navigationEvent.getName(), new NavigationEventData(navigationEvent, eventTarget, navigationEventCallback));
	}
	
	@Override
	public final IPresenter launchNavigationEvent(String navigationEventName, boolean registerHistory) {
		if (registeredNavigationEvents.containsKey(navigationEventName)) {
			return localLaunchNavigationEvent(navigationEventName, registerHistory);
		}
		return getParentPresenter().launchNavigationEvent(navigationEventName, registerHistory);
	}
	
	@Override
	public final IPresenter launchNavigationEvent(String navigationEventName, boolean registerHistory, String parentName) {
		if (getNavigationEvent().getName().equals(parentName)) {
			return localLaunchNavigationEvent(navigationEventName, registerHistory);
		}
		return getParentPresenter().launchNavigationEvent(navigationEventName, registerHistory, parentName);
	}
	
	private IPresenter localLaunchNavigationEvent(String navigationEventName, boolean registerHistory){
		if (registeredNavigationEvents.containsKey(navigationEventName)) {
			NavigationEventData eventData = registeredNavigationEvents.get(navigationEventName);
			IPresenter newPresenter = eventData.getNavigationEvent().launch(this, registerHistory);
			eventData.getNavigationEventCallback().callback();
			return newPresenter;
		}
		return new NullPresenter();
	}
	
	@Override
	public final INavigationEventData getNavigationEventData(String navigationEventName) {
		if (registeredNavigationEvents.containsKey(navigationEventName)) {
			return registeredNavigationEvents.get(navigationEventName);
		}
		return new NullNavigationEventData();
	}

	
	public IPresenter getParentPresenter() {
		if (parentPresenter==null) {
			parentPresenter = new NullPresenter();
		}
		return parentPresenter;
	}


	@Override
	public final INavigationEvent getNavigationEvent() {
		return navigationEvent;
	}


	class NavigationEventData implements INavigationEventData {
		
		private final INavigationEvent navigationEvent;
		private final INavigationEventCallback navigationEventCallback;
		private final HasWidgets eventTarget;

		NavigationEventData(INavigationEvent navigationEvent, HasWidgets eventTarget, INavigationEventCallback navigationEventCallback) {
			this.navigationEvent = navigationEvent;
			this.eventTarget = eventTarget;
			this.navigationEventCallback = navigationEventCallback;
		}

		@Override
		public INavigationEvent getNavigationEvent() {
			return navigationEvent;
		}

		@Override
		public INavigationEventCallback getNavigationEventCallback() {
			return navigationEventCallback;
		}

		@Override
		public HasWidgets getEventTarget() {
			return eventTarget;
		}
		
	}
	
}
