package com.googlecode.jpattern.gwt.client.presenter;

import java.util.List;

import com.googlecode.jpattern.gwt.client.history.IHistoryManager;
import com.googlecode.jpattern.gwt.client.view.IShowViewStrategy;
import com.googlecode.jpattern.gwt.client.view.IView;
import com.googlecode.jpattern.shared.result.IErrorMessage;

/**
 * 
 * @author Francesco Cina'
 *
 * 14 Apr 2011
 */

public abstract class APresenter<T extends IView> implements IPresenter {
	
	private final IShowViewStrategy<T> showViewStrategy;
	private final IHistoryManager historyManager;

	public APresenter(IShowViewStrategy<T> showViewStrategy, IHistoryManager historyManager) {
		this.showViewStrategy = showViewStrategy;
		this.historyManager = historyManager;
		showViewStrategy.getView().visit(this);
	}
	
	@Override
	public final void onEventError(List<IErrorMessage> errorMessages) {
		showViewStrategy.getView().getErrorArea().addErrorMessages(errorMessages);
	}
	
	@Override
	public final void onEventStart() {
		showViewStrategy.getView().getErrorArea().clear();
	}
	
	protected final T getView() {
		return showViewStrategy.getView();
	}
	
	/**
	 * This is the first step performed to a complete visualization of the associated IView.
	 * This is performed just before the IShowViewStrategy.onLoadStart() method.
	 * This method is called before the display of the IView.
	 * DO NOT perform asynchronous actions in this method.
	 */
	public abstract void preDisplay();
	
	/**
	 * This is the second step performed to a complete visualization of the associated IView
	 * This method is called just after the IShowViewStrategy.onLoadStart() method and before the
	 * IShowViewStrategy.onLoadCompleted() method.
	 * Here the asynchronous actions should be performed.
	 * Its VERY important to call the render() method when all the actions are performed otherwise
	 * the IView should not be completely rendered. 
	 */
	public abstract void postDisplay();

//	/**
//	 * This is the third step performed to a complete visualization of the associated IView
//	 * This method is called by the render() method just after the IShowViewStrategy.onLoadCompleted() execution.
//	 * Due to the fact that the render() method is called by the IHistoryManager, this method is called every time
//	 * the user uses the browser's history buttons.
//	 */
//	public abstract void postRender();
	
	/**
	 * Return the unique name to identify this presenter by the HistoryManager.
	 * To avoid the insertion of the presenter in the navigation History return an empty String.
	 * @return
	 */
	public abstract String historyTokenName();
	
	/**
	 * This method must be called when all the necessary operations to load necessary data are ended.
	 * This causes the final rendering of the IView. 
	 */
	public final void render() {
		showViewStrategy.onLoadCompleted();
	}

	public final void init() {
		historyManager.registerHistory(historyTokenName(), this);
		preDisplay();
		showViewStrategy.onLoadStart();
	    postDisplay();
	}
	
}
