package com.googlecode.jpattern.orm;

import com.googlecode.jpattern.logger.ILogger;
import com.googlecode.jpattern.orm.session.ISession;
import com.googlecode.jpattern.orm.session.ISessionProvider;
import com.googlecode.jpattern.orm.crud.IOrmCRUDQuery;
import com.googlecode.jpattern.orm.crud.OrmCRUDQueryGenerator;
import com.googlecode.jpattern.orm.dialect.DefaultDialect;
import com.googlecode.jpattern.orm.dialect.IDialect;
import com.googlecode.jpattern.orm.exception.OrmConfigurationException;
import com.googlecode.jpattern.orm.logger.OrmLogger;
import com.googlecode.jpattern.orm.mapper.ClassTableMapBuilder;
import com.googlecode.jpattern.orm.mapper.ClassMapperBuilder;
import com.googlecode.jpattern.orm.mapper.IClassMapper;
import com.googlecode.jpattern.orm.mapper.TableMap;
import com.googlecode.jpattern.orm.generator.CojenPersistorGenerator;

/**
 * 
 * @author Francesco Cina
 *
 * 21/mag/2011
 */
public class JPOrm implements IJPOrm {

	private final ISessionProvider sessionProvider;
	private IOrmClassToolMap ormClassToolMap = new OrmClassToolMap();
	private ILogger logger = OrmLogger.getOrmLogger(this.getClass());
	private final IDialect dialect;

	public JPOrm(ISessionProvider sessionProvider) {
		this(sessionProvider, new DefaultDialect());
	}
	
	public JPOrm(ISessionProvider sessionProvider, IDialect dialect) {
		this.sessionProvider = sessionProvider;
		this.dialect = dialect;
	}

	@Override
	public ISession session() {
		return sessionProvider.getSession(ormClassToolMap);
	}

	@Override
	public <T> void register(Class<T> clazz) throws OrmConfigurationException {
		register(clazz, true);
	}
	
	@Override
	public <T> void register(Class<T> clazz, boolean enableDBScan) throws OrmConfigurationException {
		logger.info("register", "register new class: " + clazz.getName());
		try {
			TableMap tableMap = new TableMap();
			new ClassTableMapBuilder(clazz, tableMap).updateTableMap();
			sessionProvider.updateTableMap(tableMap , enableDBScan);
			IClassMapper<T> classMapper = new ClassMapperBuilder<T>(clazz, tableMap).generate();
			IOrmPersistor<T> ormPersistor = new CojenPersistorGenerator<T>(classMapper).generate();
			IOrmCRUDQuery ormCRUDQuery = new OrmCRUDQueryGenerator<T>(dialect, classMapper).generate();
			IOrmClassTool<T> ormClassTool = new OrmClassTool<T>(classMapper, ormPersistor, ormCRUDQuery);
			ormClassToolMap.put(clazz, ormClassTool);
		} catch (Exception e) {
			throw new OrmConfigurationException(e);
		}
	}
	
}
