package com.googlecode.jpattern.orm.crud;

import java.util.ArrayList;
import java.util.List;

import com.googlecode.jpattern.orm.IOrmConstants;
import com.googlecode.jpattern.orm.dialect.IDialect;
import com.googlecode.jpattern.orm.mapper.IClassMapper;

/**
 * 
 * @author Francesco Cina
 *
 * 22/mag/2011
 */
public class OrmCRUDQueryGenerator<T> implements IOrmCRUDQueryGenerator<T> {

	private final IClassMapper<T> classMapper;
	private boolean generatedKey = false;
	private final IDialect dialect;

	public OrmCRUDQueryGenerator(IDialect dialect, IClassMapper<T> classMapper) {
		this.dialect = dialect;
		this.classMapper = classMapper;
	}

	/* (non-Javadoc)
	 * @see com.googlecode.jpattern.orm.generator.IOrmCRUDQueryGenerator#generate()
	 */
	@Override
	public IOrmCRUDQuery generate() {
		String deleteQuery = generateDeleteQuery();
		String loadQuery = generateLoadQuery();
		String saveQuery = generateSaveQuery();
		String updateQuery = generateUpdateQuery();
		String baseSelectClause = generateBaseSelectClause();
		String baseFromClause = generateBaseFromClause();
		return new OrmCRUDQuery(deleteQuery, loadQuery, saveQuery, updateQuery, baseSelectClause, baseFromClause, generatedKey);
	}

	private String generateBaseFromClause() {
		return classMapper.getTableMap().getTableNameWithSchema();
	}

	private String generateBaseSelectClause() {
		return columnToCommaSepareted( getColumnNames(classMapper , IOrmConstants.ROW_NAME_PREFIX_PLACEHOLDER, classMapper.getAllColumnJavaNames()) );
	}

	private String generateUpdateQuery() {
		StringBuilder builder = new StringBuilder("UPDATE ");
		builder.append(classMapper.getTableMap().getTableNameWithSchema());
		builder.append(" SET ");
		if ( classMapper.getNotPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append( columnToSetClause( getColumnNames(classMapper, "", classMapper.getNotPrimaryKeyColumnJavaNames()) ));
		}
		if ( classMapper.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( columnToWhereClause( getColumnNames(classMapper, "", classMapper.getPrimaryKeyColumnJavaNames()) ));
		}
		return builder.toString();
	}

	private String generateSaveQuery() {
		StringBuilder builder = new StringBuilder("INSERT INTO ");
		builder.append(classMapper.getTableMap().getTableNameWithSchema());
		builder.append(" (");
		builder.append( columnToCommaSepareted( getColumnNames(classMapper , "", classMapper.getAllColumnJavaNames()) ));
		builder.append(") VALUES (");
		builder.append( questionCommaSepareted( classMapper, getColumnNames(classMapper , "", classMapper.getAllColumnJavaNames()) ));
		builder.append(")");
		return builder.toString();
	}

	private String generateLoadQuery() {
		StringBuilder builder = new StringBuilder("SELECT * FROM ");
		builder.append(classMapper.getTableMap().getTableNameWithSchema());
		if ( classMapper.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( columnToWhereClause( getColumnNames(classMapper, "", classMapper.getPrimaryKeyColumnJavaNames()) ));
		}
		return builder.toString();
	}

	private String generateDeleteQuery() {
		StringBuilder builder = new StringBuilder("DELETE FROM ");
		builder.append(classMapper.getTableMap().getTableNameWithSchema());
		if ( classMapper.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( columnToWhereClause( getColumnNames(classMapper, "", classMapper.getPrimaryKeyColumnJavaNames()) ));
		}
		return builder.toString();
	}

	
	private String questionCommaSepareted(IClassMapper<T> classMapper, List<String> columnNames) {
		StringBuilder builder = new StringBuilder();
		int length = columnNames.size();
		
		if (length > 0) {
			for (int i=0; i<length-1 ; i++) {
				AColumnValueGenerator columnValueGenerator = classMapper.getTableMap().getGeneratorByColumnName(columnNames.get(i));
				generatedKey = generatedKey || columnValueGenerator.isAutoGenerated();
				builder.append( columnValueGenerator.insertQueryParameter(dialect, "?") );
				builder.append(", ");
			}
			AColumnValueGenerator columnValueGenerator = classMapper.getTableMap().getGeneratorByColumnName(columnNames.get(length-1));
			generatedKey = generatedKey || columnValueGenerator.isAutoGenerated();
			builder.append( columnValueGenerator.insertQueryParameter(dialect, "?") );
		}
		return builder.toString();
	}
	
	private String columnToCommaSepareted(List<String> columnNames) {
		StringBuilder builder = new StringBuilder();
		int length = columnNames.size();
		
		if (length > 0) {
			for (int i=0; i<length-1 ; i++) {
				builder.append(columnNames.get(i) + ", ");
			}
			builder.append(columnNames.get(length-1) );
		}
		return builder.toString();
	}
	
	private String columnToWhereClause(List<String> columnNames) {
		StringBuilder builder = new StringBuilder();
		int length = columnNames.size();
		
		if (length > 0) {
			for (int i=0; i<length-1 ; i++) {
				builder.append(columnNames.get(i) + " = ? AND ");
			}
			builder.append(columnNames.get(length-1) + " = ? ");
		}
		return builder.toString();
	}
	
	private String columnToSetClause(List<String> columnNames) {
		StringBuilder builder = new StringBuilder();
		int length = columnNames.size();
		
		if (length > 0) {
			for (int i=0; i<length-1 ; i++) {
				builder.append(columnNames.get(i) + " = ? , ");
			}
			builder.append(columnNames.get(length-1) + " = ? ");
		}
		return builder.toString();
	}
	
	private List<String> getColumnNames(IClassMapper<?> classMapper, String rowNamePrefix, String[] javaNames) {
		List<String> result = new ArrayList<String>();
		for (String javaName : javaNames) {
			result.add( rowNamePrefix + classMapper.getColumnWithJavaName(javaName).getName());
		}
		return result;
	}
}
