package com.googlecode.jpattern.orm.query;

import java.util.LinkedHashMap;
import java.util.Map;

import com.googlecode.jpattern.orm.IOrmClassToolMap;
import com.googlecode.jpattern.orm.exception.OrmException;
import com.googlecode.jpattern.orm.mapper.IClassMapper;

/**
 * 
 * @author Francesco Cina
 *
 * 22/giu/2011
 */
public class OrmClassToolMapNameSolver implements INameSolver {

	private Map<String, IClassMapper<?>> registeredClass = new LinkedHashMap<String, IClassMapper<?>>();
	private Map<Class<?>, String> classAlias = new LinkedHashMap<Class<?>, String>();
	private final IOrmClassToolMap ormClassToolMap;
	private boolean resolveWithoutAlias = false;

	public OrmClassToolMapNameSolver(IOrmClassToolMap ormClassToolMap) {
		this.ormClassToolMap = ormClassToolMap;
	}
	
	@Override
	public String solvePropertyName(String property) throws OrmException {
		if (resolveWithoutAlias) {
			return solvePropertyNameWithoutAlias(property);
		}
		String alias = alias(property);
		String field = field(property);
		if (!registeredClass.containsKey(alias)) {
			throw new OrmException("Alias [" + alias + "] is not associated with an Orm Entity. Registered alias are: " + registeredClass.keySet());
		}
		String dbColumn = registeredClass.get(alias).getColumnWithJavaName(field).getName();
		return alias + "." + dbColumn;
	}
	
	@Override
	public String solvePropertyName(String property, String defaultValue) throws OrmException {
		String alias = alias(property);
		String field = field(property);
		if (!registeredClass.containsKey(alias)) {
			return defaultValue;
		}
		String dbColumn = registeredClass.get(alias).getColumnWithJavaName(field).getName();
		return alias + "." + dbColumn;
	}

	@Override
	public void register(Class<?> clazz) throws OrmException {
		register(clazz, clazz.getSimpleName());		
	}

	@Override
	public void register(Class<?> clazz, String alias) throws OrmException {
		registeredClass.put(alias, ormClassToolMap.getOrmClassTool(clazz).getClassMapper());
		classAlias.put(clazz, alias);
	}
	
	@Override
	public String alias(Class<?> clazz) throws OrmException {
		if (!classAlias.containsKey(clazz)) {
			throw new OrmException("Class " + clazz + " is not in the query" );
		}
		return classAlias.get(clazz);
	}
	
	private String alias(String property) {
		return property.substring(0, property.lastIndexOf("."));
	}
	
	private String field(String property) {
		return property.substring(property.lastIndexOf(".")+1);
	}

	@Override
	public String solvePropertyNameWithoutAlias( String property) throws OrmException {
		String alias = alias(property);
		String field = field(property);
		if (!registeredClass.containsKey(alias)) {
			throw new OrmException("Alias [" + alias + "] is not associated with an Orm Entity. Registered alias are: " + registeredClass.keySet());
		}
		return registeredClass.get(alias).getColumnWithJavaName(field).getName();
	}

	@Override
	public void alwaysResolveWithoutAlias(boolean resolveWithoutAlias) {
		this.resolveWithoutAlias = resolveWithoutAlias;
		
	}

	@Override
	public boolean getAlwaysResolveWithoutAlias() {
		return resolveWithoutAlias;
	}

}
