package com.googlecode.jpattern.orm.query;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import com.googlecode.jpattern.orm.IOrmClassTool;
import com.googlecode.jpattern.orm.IOrmClassToolMap;
import com.googlecode.jpattern.orm.exception.OrmException;
import com.googlecode.jpattern.orm.exception.OrmNotUniqueResultException;
import com.googlecode.jpattern.orm.session.IResultSetReader;
import com.googlecode.jpattern.orm.session.ISessionSqlPerformer;
import com.googlecode.jpattern.orm.session.ISqlPerformer;

/**
 * 
 * @author Francesco Cina
 *
 * 20/giu/2011
 */
public class OrmQuery<T> extends ABaseOrmQuery implements IOrmQuery<T>, INameSolverConsumer {

	private INameSolver nameSolver = new NullNameSolver();
	private final IOrmClassToolMap ormClassToolMap;
	private final Class<T> clazz;
	private final Class<?>[] joinClasses;
	private boolean distinct;
	private final ISessionSqlPerformer session;

	public OrmQuery(IOrmClassToolMap ormClassToolMap, ISessionSqlPerformer session, Class<T> clazz, Class<?>... joinClasses) {
		this.ormClassToolMap = ormClassToolMap;
		this.session = session;
		this.clazz = clazz;
		this.joinClasses = joinClasses;
		setJoin( new Join(ormClassToolMap) );
	}

	@Override
	public void setNameSolver(INameSolver nameSolver) {
		this.nameSolver = nameSolver;
		where().setNameSolver(nameSolver);
		orderBy().setNameSolver(nameSolver);
		join().setNameSolver(nameSolver);
	}

	@Override
	public List<T> findList() {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		final IOrmClassTool<T> ormClassTool = ormClassToolMap.getOrmClassTool(clazz);
		
		IResultSetReader<List<T>> resultSetReader = new IResultSetReader<List<T>>() {
			@Override
			public List<T> read(ResultSet resultSet) throws SQLException { 
				List<T> resultList = new ArrayList<T>();
				int rowCount = 0;
				while ( resultSet.next() ) {
					resultList.add( ormClassTool.getOrmPersistor().mapRow("", resultSet, rowCount++) );
				}
				return resultList;
			}
		};
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.query(renderSql(), resultSetReader, values.toArray());
	}

	@Override
	public long findRowCount() {
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		ISqlPerformer sqlExec = session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForLong(getGeneratedRowCountSql(), values.toArray());
	}

	@Override
	public T findUnique() throws OrmNotUniqueResultException {
		int oldMaxRow = getMaxRows();
		setMaxRows(2);
		List<Object> values = new ArrayList<Object>();
		where().appendValues(values);
		List<T> result = findList();
		setMaxRows(oldMaxRow);
		if (result.size()==0) {
			throw new OrmNotUniqueResultException("The query execution returned no rows; 1 row expected");
		}
		if (result.size()>1) {
			throw new OrmNotUniqueResultException("The query execution returned a number of rows higher than 1");
		}
		return result.get(0);
	}

	@Override
	public String getGeneratedRowCountSql() {
		StringBuffer stringBuffer = new StringBuffer();
		renderSelectRowCount(stringBuffer);
		renderFrom(stringBuffer);
		renderWhere(stringBuffer);
		return stringBuffer.toString();
	}

	@Override
	public IOrmQuery<T> setDistinct() {
		this.distinct = true;
		return this;
	}

	private void renderSelectRowCount(StringBuffer stringBuffer) {
		stringBuffer.append("SELECT COUNT(*) ");
	}
	
	@Override
	protected void renderSelect(StringBuffer stringBuffer) {
		String alias = nameSolver.alias(clazz); 
		stringBuffer.append("SELECT ");
		if (distinct) {
			stringBuffer.append("DISTINCT ");
		}
		stringBuffer.append(ormClassToolMap.getOrmClassTool(clazz).getOrmCRUDQuery().getBaseSelectClause(alias + ".") );
		stringBuffer.append(" ");
	}

	@Override
	protected void renderFrom(StringBuffer stringBuffer) {
		String alias = nameSolver.alias(clazz);
		stringBuffer.append("FROM ");
		stringBuffer.append(ormClassToolMap.getOrmClassTool(clazz).getClassMapper().getTableMap().getTableNameWithSchema() );
		stringBuffer.append( " " );
		stringBuffer.append(alias);
		stringBuffer.append(" ");
		join().renderSql(stringBuffer);
		if (joinClasses!=null && joinClasses.length>0) {
			for (Class<?> joinClass : joinClasses) {
				stringBuffer.append( ", " );
				stringBuffer.append(ormClassToolMap.getOrmClassTool(joinClass).getClassMapper().getTableMap().getTableNameWithSchema() );
				stringBuffer.append( " " );
				stringBuffer.append(nameSolver.alias(joinClass));
			}
			stringBuffer.append(" ");
		}
	}
	
	@Override
	protected void renderWhere(StringBuffer stringBuffer) {
		where().renderSql(stringBuffer);
	}
	
	@Override
	protected void renderOrderBy(StringBuffer stringBuffer) {
		orderBy().renderSql(stringBuffer);
	}

	@Override
	public boolean isDistinct() throws OrmException {
		return distinct;
	}

}
