package com.googlecode.jpattern.orm.session.datasource;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

import com.googlecode.jpattern.orm.exception.OrmException;
import com.googlecode.jpattern.orm.session.ASqlPerformer;
import com.googlecode.jpattern.orm.session.IGeneratedKeyReader;
import com.googlecode.jpattern.orm.session.IPreparedStatementCreator;
import com.googlecode.jpattern.orm.session.IResultSetReader;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 * 
 * ISqlExecutor implementation using java.sql.Connection as backend.
 */
public class DataSourceSqlPerformer extends ASqlPerformer {

	private final Connection conn;
	private int queryTimeout = 0;
	private int maxRows = 0;

	public DataSourceSqlPerformer(Connection conn) {
		this.conn = conn;
	}

	@Override
	public void setMaxRows(int maxRows) {
		this.maxRows = maxRows;
	}

	@Override
	public int getMaxRows() {
		return maxRows;
	}

	@Override
	public void setQueryTimeout(int queryTimeout) {
		this.queryTimeout = queryTimeout;
	}

	@Override
	public int getQueryTimeout() {
		return queryTimeout;
	}

	@Override
	public void execute(String sql) throws OrmException {
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(getQueryTimeout());
			preparedStatement.execute();
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

	@Override
	public <T> T query(String sql, IResultSetReader<T> rse, Object... args) throws OrmException {
			ResultSet resultSet = null;
			PreparedStatement preparedStatement = null;
			try {
				preparedStatement = conn.prepareStatement( sql );
				int i = 0;
				for (Object value : args) {
					preparedStatement.setObject(++i, value);
				}
				preparedStatement.setMaxRows(getMaxRows());
				preparedStatement.setQueryTimeout(getQueryTimeout());
				resultSet = preparedStatement.executeQuery();
				return rse.read(resultSet);
			} catch (Exception e) {
				throw new OrmException(e);
			} finally {
				try {
					if (resultSet!=null && !resultSet.isClosed()) {
						resultSet.close();
					}
					if (preparedStatement!=null) {
						preparedStatement.close();
					}
				} catch (SQLException e) {
					throw new OrmException(e);
				}
			}
	}

	@Override
	public int update(String sql, Object... args) throws OrmException {
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(getQueryTimeout());
			int i = 0;
			for (Object value : args ) {
				preparedStatement.setObject(++i, value);
			}
			return preparedStatement.executeUpdate();
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

	@Override
	public int update(String sql, IGeneratedKeyReader generatedKeyExtractor, Object... args) throws OrmException {
		ResultSet generatedKeyResultSet = null;
		PreparedStatement preparedStatement = null;
		int result = 0;
		try {
			preparedStatement = conn.prepareStatement( sql , generatedKeyExtractor.generatedColumnNames());
			preparedStatement.setQueryTimeout(getQueryTimeout());
			int i = 0;
			for (Object value : args) {
				preparedStatement.setObject(++i, value);
			}
			result = preparedStatement.executeUpdate();
			generatedKeyResultSet = preparedStatement.getGeneratedKeys();
			generatedKeyExtractor.read(generatedKeyResultSet);
			return result;
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
				if (generatedKeyResultSet!=null && !generatedKeyResultSet.isClosed()) {
					generatedKeyResultSet.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

	@Override
	public int[] batchUpdate(List<String> sqls) throws OrmException {
		Statement statement = null;
		try {
			statement = conn.createStatement();
			statement.setQueryTimeout(getQueryTimeout());
			for (String sql : sqls) {
				statement.addBatch(sql);
			}
			return statement.executeBatch();
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (statement!=null) {
					statement.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

	@Override
	public int[] batchUpdate(String sql, List<Object[]> args) throws OrmException {
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(getQueryTimeout());
			for (Object[] arg : args) {
				int i = 0;
				for (Object value : arg) {
					preparedStatement.setObject(++i, value);
				}
				preparedStatement.addBatch();
			}
			return preparedStatement.executeBatch();
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

	@Override
	public int[] batchUpdate(String sql, IPreparedStatementCreator psc) throws OrmException {
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(getQueryTimeout());
			for (int i=0; i<psc.getBatchSize(); i++) {
				psc.set(preparedStatement, i);
				preparedStatement.addBatch();
			}
			return preparedStatement.executeBatch();
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (SQLException e) {
				throw new OrmException(e);
			}
		}
	}

}
