package com.googlecode.jpattern.orm.session.jdbctemplate;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import com.googlecode.jpattern.orm.exception.OrmException;
import com.googlecode.jpattern.orm.session.ASqlPerformer;
import com.googlecode.jpattern.orm.session.IGeneratedKeyReader;
import com.googlecode.jpattern.orm.session.IPreparedStatementCreator;
import com.googlecode.jpattern.orm.session.IResultSetReader;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 * 
 * ISqlExecutor implementation using JdbcTemplate as backend
 */
public class JdbcTemplateSqlPerformer extends ASqlPerformer {

	private final JdbcTemplate jdbcTemplate;
	private int maxRows = 0;
	private int queryTimeout = 0;

	public JdbcTemplateSqlPerformer(JdbcTemplate jdbcTemplate) {
		this.jdbcTemplate = jdbcTemplate;
	}

	@Override
	public void setMaxRows(int maxRows) {
		this.maxRows = maxRows;

	}

	@Override
	public int getMaxRows() {
		return maxRows;
	}

	@Override
	public void setQueryTimeout(int queryTimeout) {
		this.queryTimeout = queryTimeout;

	}

	@Override
	public int getQueryTimeout() {
		return queryTimeout;
	}

	@Override
	public void execute(String sql) throws OrmException {
		int oldMaxRows = jdbcTemplate.getMaxRows();
		int oldTimeout = jdbcTemplate.getQueryTimeout();
		try {
			jdbcTemplate.setMaxRows(getMaxRows());
			jdbcTemplate.setQueryTimeout(getQueryTimeout());
			jdbcTemplate.execute(sql);
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			jdbcTemplate.setMaxRows(oldMaxRows);
			jdbcTemplate.setQueryTimeout(oldTimeout);
		}
	}

	@Override
	public <T> T query(String sql, IResultSetReader<T> rse, Object... args) throws OrmException {
		int oldMaxRows = jdbcTemplate.getMaxRows();
		int oldTimeout = jdbcTemplate.getQueryTimeout();
		try {
			jdbcTemplate.setMaxRows(getMaxRows());
			jdbcTemplate.setQueryTimeout(getQueryTimeout());
			return jdbcTemplate.query(sql, args , new ResultSetReaderWrapper<T>(rse) );
		} catch (Exception e) {
			throw new OrmException(e);
		} finally {
			jdbcTemplate.setMaxRows(oldMaxRows);
			jdbcTemplate.setQueryTimeout(oldTimeout);
		}
	}

	@Override
	public int update(String sql, Object... args) throws OrmException {
		try {
			return jdbcTemplate.update(sql, args);
		} catch (Exception e) {
			throw new OrmException(e);
		}
	}

	@Override
	public int update(final String sql, final IGeneratedKeyReader generatedKeyReader, final Object... args) throws OrmException {
		try {
			PreparedStatementCreator psc = new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					PreparedStatement ps = null;
					ps = con.prepareStatement( sql , generatedKeyReader.generatedColumnNames());
										
					int i = 0;
					for (Object value : args ) {
						ps.setObject(++i, value);
					}
					return ps;
				}
			};
			
			KeyHolder keyHolder = new GeneratedKeyHolder();
			int result = jdbcTemplate.update(psc, keyHolder);
			generatedKeyReader.read(new GeneratorKeyResultSet(keyHolder));
			return result;
		} catch (Exception e) {
			throw new OrmException(e);
		} 
	}

	@Override
	public int[] batchUpdate(List<String> sqls) throws OrmException {
		try {
			return jdbcTemplate.batchUpdate(sqls.toArray(new String[0]));
		} catch (Exception e) {
			throw new OrmException(e);
		}
	}

	@Override
	public int[] batchUpdate(final String sql, final List<Object[]> args) throws OrmException {
		try {
			BatchPreparedStatementSetter bpss = new BatchPreparedStatementSetter() {
				
				@Override
				public void setValues(PreparedStatement ps, int i) throws SQLException {
					int count = 0;
					for ( Object object : args.get(i) ) {
						ps.setObject(++count, object);
					}
				}
				
				@Override
				public int getBatchSize() {
					return args.size();
				}
			};
			
			return jdbcTemplate.batchUpdate(sql, bpss);
		} catch (Exception e) {
			throw new OrmException(e);
		}
	}

	@Override
	public int[] batchUpdate(String sql, final IPreparedStatementCreator psc) throws OrmException {
		try {
			BatchPreparedStatementSetter bpss = new BatchPreparedStatementSetter() {
				@Override
				public void setValues(PreparedStatement ps, int i) throws SQLException {
					psc.set(ps, i);
				}
				@Override
				public int getBatchSize() {
					return psc.getBatchSize();
				}
			};
			return jdbcTemplate.batchUpdate(sql, bpss);
		} catch (Exception e) {
			throw new OrmException(e);
		}
	}

}
